<?php

namespace Drupal\commerce_shipping_pickup_api;

use Drupal\Core\Form\FormStateInterface;
use Drupal\profile\Entity\ProfileInterface;

/**
 * Interface of the pickup shipping method.
 */
interface PickupShippingInterface {

  /**
   * Defines the type of input field(s) the pane should provide.
   *
   * List case: if your service provides a full list of pickup points indexed by a unique identifier,
   * you can simply use a `select` so that the user can choose one from the list.
   * The selection will be stored into `pickup_location_data` in the shipping profile.
   * 
   * In the case of selectors that need a starting point to return a list of pickup points in the vicinity,
   * you can use `$start_address` to obtain the address provided by the customer. An empty array means
   * that the address is not yet complete. A value of NULL means that the checkout pane has the setting
   * _Needs address from customer_ unset.
   * 
   * ```
   * public function buildFormElement(ProfileInterface $profile, array $start_address = NULL): array {
   *   return [
   *     '#type' => 'select',
   *     '#title' => $this->t('Select a pickup point:'),
   *     '#default_value' => $profile->getData('pickup_location_data'),
   *     '#options' => array_map(function ($address) {
   *       return $address['organization'];
   *     }, $this->dealers),
   *   ];
   * }
   * ```
   *
   * Online selector case: some services have no fixed list but an online selector to be embedded.
   * When the user selects a point, the selector will inform us about the selection.
   * In this case, you can use `textfield` and `hidden` elements to store the details of the selected pickup point.
   * These details will be stored into `pickup_location_data` in the shipping profile.
   *
   * ```
   * public function buildFormElement(ProfileInterface $profile, array $start_address = NULL): array {
   *   $element = [
   *     '#type' => 'container',
   *   ];
   *   $element['selector'] = [
   *     '#type' => 'inline_template',
   *     '#template' => '<iframe width="100%" height="500px" src="{{ url }}"></iframe>',
   *     '#context' => ['url' => 'https://...'],
   *   ];
   *   $element['id'] = [
   *     '#type' => 'hidden',
   *     '#required' => TRUE,
   *   ];
   *   $element['known_location'] = [
   *     '#type' => 'hidden',
   *   ];
   *   $element['organization'] = [
   *     '#type' => 'textfield',
   *     '#title' => $this->t('Name'),
   *     '#attributes' => ['readonly' => 'readonly'],
   *   ];
   *   $element['address_line1'] = [
   *     '#type' => 'textfield',
   *     '#title' => $this->t('Address'),
   *     '#attributes' => ['readonly' => 'readonly'],
   *   ];
   *
   *   return $element;
   * }
   * ```
   * 
   * Map-based selectors can include a `known_location` value to indicate that the user has already selected a location on the map,
   * so displaying the original shipping profile to ask for an address is no longer necessary. The type and contents aren't important,
   * so it may be used to store data (eg. the location itself). The framework only checks if the value is present or not.
   * If present, it will be added to the `$start_address` argument as well (without processing, as is, so it's up to the shipping
   * method to handle its contents.)
   * 
   * This only applies to checkout panes with _Needs address from customer_ set, it's ignored otherwise.
   *
   * @param \Drupal\profile\ProfileInterface $profile
   *   The pickup profile as context.
   * @param \Drupal\profile\ProfileInterface $shipping_profile
   *   The shipping profile as context (needed when the pickup point selection depends on the customer location).
   */
  public function buildFormElement(ProfileInterface $profile, array $start_address = NULL): array;

  /**
   * Populates the profile with data of the selected pickup point.
   *
   * `populateProfile()` retrieves what [buildFormElement()] stored when the user selected
   * the desired pickup point. In the list case mentioned there, this data was the ID
   * of the point, so we use it to look up other details of the point from a `$dealers`
   * list containing data of all pickup points:
   *
   * ```
   * $id = $profile->getData('pickup_location_data');
   * $data = $this->dealers[$id];
   * ```
   *
   * In the online selector case, both the ID and other details of the point
   * were stored directly as an array:
   *
   * ```
   * $data = $profile->getData('pickup_location_data');
   * ```
   *
   * So, we can put all details into the shipping profile (don't forget the country code, it's required
   * for a Commerce `address` to function correctly, the rest is up to you):
   *
   * ```
   * $profile->set('address', [
   *   "country_code" => "CH",
   *   "organization" => $data['organization'],
   *   "address_line1" => $data['address_line1'],
   *   "address_line3" => $data['id'],
   * ]);
   * ```
   *
   * @param \Drupal\profile\ProfileInterface $profile
   *   The profile to populate.
   */
  public function populateProfile(ProfileInterface $profile): void;

  /**
   * Validates the form with data. Optional.
   *
   * ```
   * $address = $form_state->getValue(['pickup_capable_shipping_information', 'shipping_profile', 'pickup_dealer', 'address']);
   * if (empty($address)) {
   *   $form_state->setErrorByName('pickup_capable_shipping_information][shipping_profile][pickup_dealer', $this->t('Please, select a pickup point.'));
   * }
   * ```
   *
   * @param array $form
   * @param Drupal\Core\Form\FormStateInterface $form_state
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void;
}
