<?php

namespace Drupal\commerce_shipping_pickup_api\Plugin\Commerce\CheckoutPane;

use Drupal\commerce_checkout\Plugin\Commerce\CheckoutFlow\CheckoutFlowInterface;
use Drupal\commerce_order\Entity\OrderInterface;
use Drupal\commerce_shipping\Entity\ShipmentInterface;
use Drupal\commerce_shipping\OrderShipmentSummaryInterface;
use Drupal\commerce_shipping\PackerManagerInterface;
use Drupal\commerce_shipping\Plugin\Commerce\CheckoutPane\ShippingInformation;
use Drupal\commerce_shipping\ShipmentManagerInterface;
use Drupal\commerce_shipping\ShippingOrderManagerInterface;
use Drupal\commerce_shipping\ShippingRate;
use Drupal\commerce\InlineFormManager;
use Drupal\Component\Utility\NestedArray;
use Drupal\Core\Entity\{Entity\EntityFormDisplay, EntityTypeBundleInfo, EntityTypeManagerInterface};
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\Element;
use Drupal\profile\Entity\ProfileInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides the shipping information pane.
 *
 * Collects the shipping profile, then the information for each shipment.
 * Assumes that all shipments share the same shipping profile.
 *
 * @CommerceCheckoutPane(
 *   id = "pickup_capable_shipping_information",
 *   label = @Translation("Shipping information"),
 *   wrapper_element = "fieldset",
 * )
 */
final class PickupCapableShippingInformation extends ShippingInformation {
  /**
   * The shipment manager.
   *
   * @var \Drupal\commerce_shipping\ShipmentManagerInterface
   */
  protected $shipmentManager;

  /**
   * Constructs a new PickupCapableShippingInformation object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\commerce_checkout\Plugin\Commerce\CheckoutFlow\CheckoutFlowInterface $checkout_flow
   *   The parent checkout flow.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Entity\EntityTypeBundleInfo $entity_type_bundle_info
   *   The entity type bundle info.
   * @param \Drupal\commerce\InlineFormManager $inline_form_manager
   *   The inline form manager.
   * @param \Drupal\commerce_shipping\PackerManagerInterface $packer_manager
   *   The packer manager.
   * @param \Drupal\commerce_shipping\OrderShipmentSummaryInterface $order_shipment_summary
   *   The order shipment summary.
   * @param \Drupal\commerce_shipping\ShippingOrderManagerInterface $shipping_order_manager
   *   The shipping order manager.
   * @param \Drupal\commerce_shipping\ShipmentManagerInterface $shipment_manager
   *   The shipment manager service.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, CheckoutFlowInterface $checkout_flow, EntityTypeManagerInterface $entity_type_manager, EntityTypeBundleInfo $entity_type_bundle_info, InlineFormManager $inline_form_manager, PackerManagerInterface $packer_manager, OrderShipmentSummaryInterface $order_shipment_summary, ShippingOrderManagerInterface $shipping_order_manager, ShipmentManagerInterface $shipment_manager) {
    parent::__construct($configuration, $plugin_id, $plugin_definition, $checkout_flow, $entity_type_manager, $entity_type_bundle_info, $inline_form_manager, $packer_manager, $order_shipment_summary, $shipping_order_manager);
    $this->entityTypeBundleInfo = $entity_type_bundle_info;
    $this->inlineFormManager = $inline_form_manager;
    $this->packerManager = $packer_manager;
    $this->orderShipmentSummary = $order_shipment_summary;
    $this->shippingOrderManager = $shipping_order_manager;
    $this->shipmentManager = $shipment_manager;
  }

  /**
   * {@inheritdoc}
   * @noinspection PhpParamsInspection
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition, ?CheckoutFlowInterface $checkout_flow = NULL) {
    return new PickupCapableShippingInformation(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $checkout_flow,
      $container->get('entity_type.manager'),
      $container->get('entity_type.bundle.info'),
      $container->get('plugin.manager.commerce_inline_form'),
      $container->get('commerce_shipping.packer_manager'),
      $container->get('commerce_shipping.order_shipment_summary'),
      $container->get('commerce_shipping.order_manager'),
      $container->get('commerce_shipping.shipment_manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'pickup_need_address' => FALSE,
      'auto_recalculate' => FALSE,
      'require_shipping_profile' => FALSE,
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationSummary(): string {
    $header = '<strong>' . $this->t('Supports pickup') . '</strong>';
    $parent_summary = parent::buildConfigurationSummary();

    if (!empty($this->configuration['pickup_need_address'])) {
      $summary = $this->t('Needs address from customer: Yes') . '<br>';
    } else {
      $summary = $this->t('Needs address from customer: No') . '<br>';
    }

    return implode('<br>', [$header, $parent_summary, $summary]);
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);

    // To ensure proper order
    unset($form['require_shipping_profile']);
    unset($form['auto_recalculate']);

    $form['support_pickup'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Supports pickup'),
      '#default_value' => TRUE,
      '#disabled' => TRUE,
    ];
    $form['pickup_need_address'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Needs address from customer'),
      '#description' => $this->t('To be used with pickup shipping methods that require an initial location to provide pickup points in the vicinity. Methods that either provide a list or map of all pickup points, irrespective of the preferred location of the customer, don\'t need this setting.'),
      '#default_value' => $this->configuration['pickup_need_address'],
    ];
    $form['require_shipping_profile'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Hide shipping costs until an address is entered'),
      '#default_value' => $this->configuration['require_shipping_profile'],
      '#states' => [
        'visible' => [
          ':input[name="configuration[panes][pickup_capable_shipping_information][configuration][pickup_need_address]"]' => ['checked' => TRUE],
        ],
      ],
    ];
    $form['auto_recalculate'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Auto recalculate shipping costs when the shipping address changes'),
      '#default_value' => $this->configuration['auto_recalculate'],
      '#states' => [
        'visible' => [
          ':input[name="configuration[panes][pickup_capable_shipping_information][configuration][pickup_need_address]"]' => ['checked' => TRUE],
          ':input[name="configuration[panes][pickup_capable_shipping_information][configuration][require_shipping_profile]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    parent::submitConfigurationForm($form, $form_state);

    if (!$form_state->getErrors()) {
      $values = $form_state->getValue($form['#parents']);
      $this->configuration['pickup_need_address'] = !empty($values['pickup_need_address']);
      $this->configuration['require_shipping_profile'] = !empty($values['require_shipping_profile']) && $this->configuration['pickup_need_address'];
      $this->configuration['auto_recalculate'] = !empty($values['auto_recalculate']) && $this->configuration['require_shipping_profile'];
    }
  }

  /**
   * {@inheritdoc}
   * @throws \Drupal\Component\Plugin\Exception\PluginException
   */
  public function buildPaneForm(array $pane_form, FormStateInterface $form_state, array &$complete_form): array {
    $store = $this->order->getStore();
    $available_countries = [];
    foreach ($store->get('shipping_countries') as $country_item) {
      $available_countries[] = $country_item->value;
    }
    /** @var \Drupal\commerce\Plugin\Commerce\InlineForm\EntityInlineFormInterface $inline_form */
    $shipping_form = $this->inlineFormManager->createInstance('customer_profile', [
      'profile_scope' => 'pickup',
      'available_countries' => $available_countries,
      'address_book_uid' => $this->order->getCustomerId(),
      // Don't copy the profile to address book until the order is placed.
      'copy_on_save' => FALSE,
    ], $this->getShippingProfile());

    $pickup_need_address = !empty($this->configuration['pickup_need_address']);
    if ($pickup_need_address) {
      // Prepare the form for ajax.
      // Not using Html::getUniqueId() on the wrapper ID to avoid #2675688.
      $pane_form['#wrapper_id'] = 'pickup-capable-shipping-information-wrapper';
      $pane_form['#prefix'] = '<div id="' . $pane_form['#wrapper_id'] . '">';
      $pane_form['#suffix'] = '</div>';
      // Auto recalculation is enabled only when a shipping profile is required.
      $pane_form['#auto_recalculate'] = !empty($this->configuration['auto_recalculate']) && !empty($this->configuration['require_shipping_profile']) && $pickup_need_address;
      $pane_form['#after_build'][] = [static::class, 'autoRecalculateProcess'];
    }

    $pane_form['shipping_profile'] = [
      '#parents' => array_merge($pane_form['#parents'], ['shipping_profile']),
      '#inline_form' => $shipping_form,
    ];
    $pane_form['shipping_profile'] = $shipping_form->buildInlineForm($pane_form['shipping_profile'], $form_state);
    $triggering_element = $form_state->getTriggeringElement();
    // The shipping_profile should always exist in form state (and not just
    // after "Recalculate shipping" is clicked).
    if (!$form_state->has('shipping_profile') ||
      // For some reason, when the address selected is changed, the shipping
      // profile in form state is stale.
      (isset($triggering_element['#parents']) && in_array('select_address', $triggering_element['#parents'], TRUE))) {
      $form_state->set('shipping_profile', $shipping_form->getEntity());
    }

    if ($pickup_need_address) {
      // Ensure selecting a different address refreshes the entire form.
      if (isset($pane_form['shipping_profile']['select_address'])) {
        $pane_form['shipping_profile']['select_address']['#ajax'] = [
          'callback' => [$this, 'ajaxRefreshForm'],
          'element' => $pane_form['#parents'],
        ];
        // Selecting a different address should trigger a recalculation.
        $pane_form['shipping_profile']['select_address']['#recalculate'] = TRUE;
      }

      $pane_form['recalculate_shipping'] = [
        '#type' => 'button',
        '#value' => $this->t('Recalculate shipping'),
        '#recalculate' => TRUE,
        '#ajax' => [
          'callback' => [$this, 'ajaxRefreshForm'],
          'element' => $pane_form['#parents'],
        ],
        // The calculation process only needs a valid shipping profile.
        '#limit_validation_errors' => [
          array_merge($pane_form['#parents'], ['shipping_profile']),
        ],
        '#after_build' => [
          [static::class, 'clearValues'],
        ],
      ];
    }

    $pane_form['removed_shipments'] = [
      '#type' => 'value',
      '#value' => [],
    ];
    $pane_form['shipments'] = [
      '#type' => 'container',
      // Pickup custom: place at top.
      '#weight' => -999,
    ];

    $shipping_profile = $form_state->get('shipping_profile');
    /** @var \Drupal\commerce_shipping\Entity\ShipmentInterface[] $shipments */
    $shipments = $this->order->get('shipments')->referencedEntities();
    $recalculate_shipping = $form_state->get('recalculate_shipping');
    $can_calculate_rates = $this->canCalculateRates($shipping_profile);

    // If the shipping recalculation is triggered, ensure the rates can
    // be recalculated (i.e a valid address is entered).
    if ($recalculate_shipping && !$can_calculate_rates) {
      $recalculate_shipping = FALSE;
      $shipments = [];
    }

    // Ensure the profile is saved with the latest address, it's necessary
    // to do that in case the profile isn't new, otherwise the shipping profile
    // referenced by the shipment won't reflect the updated address.
    if (!$shipping_profile->isNew() && $shipping_profile->hasTranslationChanges() && $can_calculate_rates) {
      $shipping_profile->save();
      $shipping_form->setEntity($shipping_profile);
    }

    $force_packing = empty($shipments) && $can_calculate_rates;
    if ($recalculate_shipping || $force_packing) {
      // We're still relying on the packer manager for packing the order since
      // we don't want the shipments to be saved for performance reasons.
      // The shipments are saved on pane submission.
      [$shipments, $removed_shipments] = $this->packerManager->packToShipments($this->order, $shipping_profile, $shipments);

      // Store the IDs of removed shipments for submitPaneForm().
      $pane_form['removed_shipments']['#value'] = array_map(function ($shipment) {
        /** @var \Drupal\commerce_shipping\Entity\ShipmentInterface $shipment */
        return $shipment->id();
      }, $removed_shipments);
    }

    $default_rate = NULL;
    $single_shipment = count($shipments) === 1;
    foreach ($shipments as $index => $shipment) {
      /** @var \Drupal\commerce_shipping\Entity\ShipmentInterface $shipment */
      $pane_form['shipments'][$index] = [
        '#parents' => array_merge($pane_form['#parents'], ['shipments', $index]),
        '#array_parents' => array_merge($pane_form['#parents'], ['shipments', $index]),
        '#type' => $single_shipment ? 'container' : 'fieldset',
        '#title' => $shipment->getTitle(),
      ];
      $form_display = EntityFormDisplay::collectRenderDisplay($shipment, 'checkout');
      $form_display->removeComponent('shipping_profile');
      $form_display->buildForm($shipment, $pane_form['shipments'][$index], $form_state);
      $pane_form['shipments'][$index]['#shipment'] = $shipment;

      // Pickup custom: Add ajax.
      $widget = &$pane_form['shipments'][$index]['shipping_method']['widget'][0];
      $widget['#ajax'] = [
        'callback' => [$this, 'ajaxRefreshForm'],
        'element' => $widget['#field_parents'],
      ];
      $widget['#limit_validation_errors'] = [
        $widget['#field_parents'],
      ];
      $rates = $this->shipmentManager->calculateRates($shipment);
      $default_rate ??= $this->shipmentManager->selectDefaultRate($shipment, $rates);
    }

    // Update the shipments and save the order if no rate was explicitly
    // selected, that usually occurs when changing addresses, this will ensure
    // the default rate is selected/applied.
    if (!$this->hasRateSelected($pane_form, $form_state) && ($recalculate_shipping || $force_packing)) {
      $new_profile = $shipping_profile->isNew();
      array_map(function (ShipmentInterface $shipment) {
        if (!$shipment->isNew()) {
          $shipment->save();
        }
      }, $shipments);
      $this->order->set('shipments', $shipments);
      $this->order->save();
      if ($new_profile) {
        $shipping_form->setEntity($shipping_profile);
      }

      $shipment_storage = $this->entityTypeManager->getStorage('commerce_shipment');
      foreach ($shipments as $index => $shipment) {
        if ($shipment->isNew()) {
          continue;
        }
        // Reload the shipment in case it was updated e.g. the tax adjustments
        // were applied to the shipment.
        $pane_form['shipments'][$index]['#shipment'] = $shipment_storage->load($shipment->id());
        $rates = $this->shipmentManager->calculateRates($shipment);
        $default_rate ??= $this->shipmentManager->selectDefaultRate($shipment, $rates);
      }
    }

    // Replace or enlarge the inline form with our own if a pickup shipping method is
    // selected or default.
    if ($this->isPickupSelected($pane_form, $form_state, $this->order, $default_rate)) {
      $pickup_form = $this->inlineFormManager->createInstance('pickup_profile', [
        'profile_scope' => 'shipping',
        'available_countries' => $available_countries,
      ], $this->getShippingProfile(TRUE));

      $pane_form_pickup_profile = [
        '#parents' => array_merge($pane_form['#parents'], ['shipping_profile']),
        '#inline_form' => $pickup_form,
        '#pickup_options' => [
          '#default_rate' => $default_rate,
          '#need_address' => $pickup_need_address,
        ],
      ];

      if ($pickup_need_address) {
        $user_input = $form_state->getUserInput();
        // If the shipping method signals that the address is no longer required (eg. the customer clicked on a map already).
        $known_location = NestedArray::getValue($user_input, ['pickup_capable_shipping_information', 'shipping_profile', 'pickup_dealer', 'known_location']);
        if (empty($known_location)) {
          $pane_form['pickup_need_address_info'] = [
            '#type' => 'markup',
            '#markup' => $this->t('Provide an address (or click on the map) near you where we can look for pickup points.'),
            '#weight' => -99,
          ];
        }

        // We need both the original shipping profile (for the customer to specify the start address) and our own pickup profile to store the point selection.
        // When finished, we will overwrite the shipping profile with our selected pickup point.
        $pane_form['pickup_profile'] = $pickup_form->buildInlineForm($pane_form_pickup_profile, $form_state);
      } else {
        // A single profile is enough but we replace the original shipping profile with our pickup profile.
        $pane_form['shipping_profile'] = $pickup_form->buildInlineForm($pane_form_pickup_profile, $form_state);
        $form_state->set('shipping_profile', NULL);
      }
    }

    return $pane_form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitPaneForm(array &$pane_form, FormStateInterface $form_state, array &$complete_form) {
    $profile_name = $pane_form['pickup_profile'] ? 'pickup_profile' : 'shipping_profile';
    $inline_form = $pane_form[$profile_name]['#inline_form'];
    /** @var \Drupal\profile\Entity\ProfileInterface $profile */
    $profile = $inline_form->getEntity();

    // Save the modified shipments.
    $shipments = [];
    foreach (Element::children($pane_form['shipments']) as $index) {
      if (!isset($pane_form['shipments'][$index]['#shipment'])) {
        continue;
      }

      /** @var \Drupal\commerce_shipping\Entity\ShipmentInterface $shipment */
      $shipment = clone $pane_form['shipments'][$index]['#shipment'];
      $form_display = EntityFormDisplay::collectRenderDisplay($shipment, 'checkout');
      $form_display->removeComponent('shipping_profile');
      $form_display->extractFormValues($shipment, $pane_form['shipments'][$index], $form_state);
      $shipment->setShippingProfile($profile);
      $shipment->save();
      $shipments[] = $shipment;
    }
    $this->order->shipments = $shipments;

    // Delete shipments that are no longer in use.
    $removed_shipment_ids = $pane_form['removed_shipments']['#value'];
    if (!empty($removed_shipment_ids)) {
      $shipment_storage = $this->entityTypeManager->getStorage('commerce_shipment');
      $removed_shipments = $shipment_storage->loadMultiple($removed_shipment_ids);
      $shipment_storage->delete($removed_shipments);
    }
  }

  /**
   * Determines if a pickup shipping method is selected.
   *
   * This function is static, to not duplicate code.
   * Also used in ProfileFieldCopyWithoutPickup.
   *
   * @param array $form
   *   The form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   * @param \Drupal\commerce_order\Entity\OrderInterface $order
   *   The order.
   * @param \Drupal\commerce_shipping\ShippingRate|NULL $default_rate
   *   The default shipping rate;
   *
   * @return bool
   *   Whether the selected method is a pickup shipping method or not.
   */
  public function isPickupSelected(array $form, FormStateInterface $form_state, OrderInterface $order, ShippingRate $default_rate = NULL) {
    $is_pickup = FALSE;
    $shipping_method = NestedArray::getValue(
      $form_state->getUserInput(),
      array_merge($form['#parents'], ['shipments', 0, 'shipping_method', 0])
    );
    if (!empty($shipping_method)) {
      $is_pickup = str_contains($shipping_method, 'pickup');
    } else {
      /** @var \Drupal\commerce_shipping\Entity\ShipmentInterface[] $shipments */
      $shipments = $order->get('shipments')->referencedEntities();
      $shipment = reset($shipments);
      if ($shipment !== FALSE) {
        $plugin = $shipment->getShippingMethod()->getPlugin();
        $is_pickup = str_contains($plugin->getPluginId(), 'pickup');
      } else {
        // Fallback on the default shipping rate.
        if ($default_rate instanceof ShippingRate) {
          $is_pickup = str_contains($default_rate->getId(), 'pickup');
        }
      }
    }
    return $is_pickup;
  }

  /**
   * Gets the shipping profile and clears address data on switch.
   *
   * @param bool $isPickup
   *   Is pickup store profile?
   *
   * @return \Drupal\profile\Entity\ProfileInterface
   *   The shipping profile.
   */
  protected function getShippingProfile(bool $isPickup = FALSE) {
    $profile = parent::getShippingProfile();

    $profile_data = $profile->getData('pickup_location_data', FALSE);
    $not_pickup_has_data = !$isPickup && $profile_data !== FALSE;
    $is_pickup_no_data = $isPickup && $profile_data === FALSE;
    if ($not_pickup_has_data || $is_pickup_no_data) {
      $profile = $this->entityTypeManager->getStorage('profile')->create([
        'type' => $profile->bundle(),
        'uid' => 0,
      ]);
    }

    return $profile;
  }

  /**
   * Gets whether shipping rates can be calculated for the given profile.
   * Ensures that a required profile address is present and valid.
   *
   * @param \Drupal\profile\Entity\ProfileInterface $profile
   *   The profile.
   *
   * @return bool
   *   TRUE if shipping rates can be calculated, FALSE otherwise.
   */
  protected function canCalculateRates(ProfileInterface $profile) {
    if (!empty($this->configuration['require_shipping_profile'])) {
      foreach ($profile->get('address')->getValue() as $address) {
        if (empty($address['country_code']) || empty($address['locality']) || empty($address['postal_code']) || empty($address['address_line1']))
          return FALSE;
      }
    }
    return TRUE;
  }

  /**
   * Returns the current order.
   *
   * @return OrderInterface
   */
  public function getOrder(): OrderInterface {
    return $this->order;
  }
}
