<?php

namespace Drupal\commerce_shipping_pickup_api\Plugin\Commerce\InlineForm;

use Drupal\commerce_checkout\Plugin\Commerce\CheckoutFlow\MultistepDefault;
use Drupal\commerce_shipping_pickup_api\PickupShippingInterface;
use Drupal\commerce_shipping_pickup_api\Plugin\Commerce\CheckoutPane\PickupCapableShippingInformation;
use Drupal\commerce\Plugin\Commerce\InlineForm\EntityInlineFormBase;
use Drupal\Component\Utility\NestedArray;
use Drupal\Core\Form\FormStateInterface;
use Drupal\profile\Entity\ProfileInterface;

/**
 * Provides an inline form for managing a pickup profile.
 * 
 * Shipping methods that _Need address from the customer_ for their selection UI take that address
 * from the regular shipping profile and only use this profile temporarily to store the pickup point.
 * Those without a starting address have their regular shipping profile already replaced by this one.
 * During submit, the selected pickup point becomes the actual shipping address.
 *
 * @CommerceInlineForm(
 *   id = "pickup_profile",
 *   label = @Translation("Pickup profile"),
 * )
 */
final class PickupProfile extends EntityInlineFormBase {

  /**
   * The pickup shipping method.
   *
   * @var \Drupal\commerce_shipping_pickup_api\PickupShippingInterface
   */
  protected $shippingMethod;

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      // Unique. Passed along to field widgets. Examples: 'billing', 'shipping'.
      'profile_scope' => 'pickup',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildInlineForm(array $inline_form, FormStateInterface $form_state) {
    $inline_form = parent::buildInlineForm($inline_form, $form_state);

    // Follow the chain: form > pane > order > shipment
    /** @var MultistepDefault $form_object */
    $form_object = $form_state->getFormObject();
    /** @var PickupCapableShippingInformation $pane */
    $pane = $form_object->getPane('pickup_capable_shipping_information');
    /** @var \Drupal\commerce_shipping\Entity\ShipmentInterface[] $shipments */
    $shipments = $pane->getOrder()->get('shipments')->referencedEntities();

    $this->shippingMethod = NULL;
    if (empty($shipments)) {
      $shipment = NULL;
      // First display, order doesn't have stored shipment yet, show the default
      // Follow the chain: rate > shipping method
      $shipping_method_id = $inline_form['#pickup_options']['#default_rate']->getShippingMethodId();
      $shipping_method = \Drupal::entityTypeManager()->getStorage('commerce_shipping_method')->load($shipping_method_id);
      $this->shippingMethod = $shipping_method->getPlugin();
      $shipping_profile = $form_state->get('shipping_profile');
    } else {
      // Order already has stored shipment
      // Follow the chain: shipment > shipping method
      $shipment = reset($shipments);
      $this->shippingMethod = $shipment->getShippingMethod()?->getPlugin();
      $shipping_profile = $shipment->getShippingProfile();
    }

    if ($this->shippingMethod instanceof PickupShippingInterface) {
      $start_address = $this->hasPrimaryAddress($shipping_profile) ? $this->getPrimaryAddress($shipping_profile) : [];
      $known_location = NestedArray::getValue($form_state->getUserInput(), ['pickup_capable_shipping_information', 'shipping_profile', 'pickup_dealer', 'known_location']);
      if (!empty($known_location)) {
        $start_address['known_location'] = $known_location;
      }
      $inline_form['pickup_dealer'] = $this->shippingMethod->buildFormElement($this->entity, $inline_form['#pickup_options']['#need_address'] ? $start_address : NULL);
    }

    return $inline_form;
  }

  private function hasPrimaryAddress($profile): bool {
    return isset($profile) && $profile->hasField('address') && !$profile->get('address')->isEmpty();
  }

  private function getPrimaryAddress($profile): array {
    return $profile->get('address')?->getValue()[0] ?? [];
  }

  /**
   * {@inheritdoc}
   */
  public function validateInlineForm(array &$inline_form, FormStateInterface $form_state) {
    parent::validateInlineForm($inline_form, $form_state);

    if ($this->shippingMethod != NULL && $this->shippingMethod instanceof PickupShippingInterface) {
      $trigger = $form_state->getTriggeringElement();
      if (!$trigger || empty($trigger['#recalculate'])) {
        // We have to suppress custom validation during recalculation, there's no point selected yet to validate.
        $this->shippingMethod->validateForm($inline_form, $form_state);
      }
    }
  }

  /**
   * {@inheritdoc}
   * @throws \Drupal\Core\Entity\EntityStorageException
   */
  public function submitInlineForm(array &$inline_form, FormStateInterface $form_state) {
    parent::submitInlineForm($inline_form, $form_state);

    if ($this->shippingMethod != NULL) {
      assert($this->shippingMethod instanceof PickupShippingInterface);

      /** @var ProfileInterface $profile */
      $profile = $this->entity;

      $values = NestedArray::getValue($form_state->getValues(), $inline_form['#parents']);
      $profile->setData('pickup_location_data', $values['pickup_dealer']);
      // Do not save a dealer pickup address into the user's address book.
      $profile->unsetData('address_book_profile_id');
      $profile->unsetData('copy_to_address_book');
      $this->shippingMethod->populateProfile($profile);
      $profile->save();
    } else {
      \Drupal::messenger()->addError('Error in inline form.');
    }
  }
}
