<?php

namespace Drupal\commerce_shipping_pickup_store\Plugin\Commerce\ShippingMethod;

use CommerceGuys\Addressing\AddressFormat\AddressField;
use CommerceGuys\Addressing\AddressFormat\FieldOverride;
use Drupal\commerce_shipping_pickup_api\Plugin\Commerce\ShippingMethod\PickupShippingMethodBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\profile\Entity\ProfileInterface;

/**
 * Provides a specific pickup shipping method.
 *
 * @CommerceShippingMethod(
 *   id = "pickup_store",
 *   label = @Translation("Pickup shipping - In-store"),
 * )
 */
class PickupStoreShipping extends PickupShippingMethodBase {
  use StringTranslationTrait;

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'address' => NULL,
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);

    $form['address_wrapper'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Pickup address'),
    ];
    $form['address_wrapper']['address'] = [
      '#type' => 'address',
      '#default_value' => $this->configuration['address'],
      '#field_overrides' => [
        AddressField::GIVEN_NAME => FieldOverride::HIDDEN,
        AddressField::FAMILY_NAME => FieldOverride::HIDDEN,
      ],
      '#required' => TRUE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    parent::submitConfigurationForm($form, $form_state);

    if (!$form_state->getErrors()) {
      $values = $form_state->getValue($form['#parents']);
      $this->configuration['address'] = $values['address_wrapper']['address'];
    }
  }

  /**
   * {@inheritdoc}
   */
  public function buildFormElement(ProfileInterface $profile, array $start_address = NULL): array {
    $data = $this->configuration['address'];
    $element = [
      '#type' => 'container',
    ];
    $element['address'] = [
      '#type' => 'hidden',
      '#value' => $data,
      '#required' => TRUE,
    ];
    $element['visible'] = [
      '#theme' => 'address_plain',
      '#country_code' => $data['country_code'] ?? '',
      '#administrative_area' => $data['administrative_area'] ?? '',
      '#locality' => $data['locality'] ?? '',
      '#dependent_locality' => $data['dependent_locality'] ?? '',
      '#postal_code' => $data['postal_code'] ?? '',
      '#sorting_code' => $data['sorting_code'] ?? '',
      '#address_line1' => $data['address_line1'] ?? '',
      '#address_line2' => $data['address_line2'] ?? '',
      '#address_line3' => $data['address_line3'] ?? '',
      '#organization' => $data['organization'] ?? '',
    ];
    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function populateProfile(ProfileInterface $profile): void {
    $data = $profile->getData('pickup_location_data');
    $profile->set('address', $data['address']);
  }
}
