<?php

namespace Drupal\commerce_shipping_pickup_brt\Plugin\Commerce\ShippingMethod;

use Drupal\commerce_shipping_pickup_api\Plugin\Commerce\ShippingMethod\PickupShippingMethodBase;
use Drupal\commerce_shipping\PackageTypeManagerInterface;
use Drupal\commerce_shipping\ShippingService;
use Drupal\Component\Serialization\Json;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\profile\Entity\ProfileInterface;
use Drupal\state_machine\WorkflowManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a specific pickup shipping method.
 *
 * @CommerceShippingMethod(
 *   id = "pickup_it_brt_fermopoint_map",
 *   label = @Translation("Pickup shipping - BRT-fermopoint (map)"),
 * )
 */
class FermopointMapShipping extends PickupShippingMethodBase {
  use StringTranslationTrait;

  /**
   * The config manager.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Nearby pickup points returned by BRT.
   *
   * @var array
   */
  protected $pickupPoints;

  public function __construct(array $configuration, $plugin_id, $plugin_definition, PackageTypeManagerInterface $package_type_manager, WorkflowManagerInterface $workflow_manager, ConfigFactoryInterface $configFactory) {
    parent::__construct($configuration, $plugin_id, $plugin_definition, $package_type_manager, $workflow_manager);
    $this->configFactory = $configFactory;

    $this->services['pickup'] = new ShippingService('pickup', $this->configuration['rate_label']);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('plugin.manager.commerce_package_type'),
      $container->get('plugin.manager.workflow'),
      $container->get('config.factory')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'map_default_latitude' => 41.8986052,
      'map_default_longitude' => 12.4757582,
      'show_data' => TRUE,
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);

    $config = $this->configFactory->get('commerce_shipping_pickup_brt.settings');
    $form['show_data'] = [
      '#type' => 'checkbox',
      '#title' => t('Show point info'),
      '#description' => t('Show opening hours, parking facilities, distance from customer, etc.'),
      '#default_value' => $this->configuration['show_data'],
    ];
    $form['map_default'] = [
      '#type' => 'fieldset',
      '#tree' => TRUE,
      '#title' => t('Default map position'),
      '#description' => t('Default map position if geolocation is disabled or fails. Defaults to the center of Rome.'),
    ];
    $form['map_default']['map_default_latitude'] = [
      '#type' => 'textfield',
      '#title' => t('Latitude'),
      '#default_value' => $this->configuration['map_default_latitude'],
    ];
    $form['map_default']['map_default_longitude'] = [
      '#type' => 'textfield',
      '#title' => t('Longitude'),
      '#default_value' => $this->configuration['map_default_longitude'],
    ];
    $form['brt_api_key'] = [
      '#type' => 'textfield',
      '#title' => t('BRT API key'),
      '#description' => t('Visit the <a href="@brt-service">BRT dashboard</a> to get your key.', ['@brt-service' => 'https://www.brt.it/it/servizio-clienti/codici-assistenza/']),
      '#default_value' => $config->get('brt_pickup_api_key'),
      '#required' => TRUE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    parent::submitConfigurationForm($form, $form_state);

    if (!$form_state->getErrors()) {
      $values = $form_state->getValue($form['#parents']);
      $this->configuration['show_data'] = $values['show_data'];
      $this->configuration['map_default_latitude'] = $values['map_default']['map_default_latitude'];
      $this->configuration['map_default_longitude'] = $values['map_default']['map_default_longitude'];
      $config = $this->configFactory->getEditable('commerce_shipping_pickup_brt.settings');
      $config->set('brt_pickup_api_key', $values['brt_api_key']);
      $config->save();
    }
  }

  /**
   * {@inheritdoc}
   */
  public function buildFormElement(ProfileInterface $profile, array $start_address = NULL): array {
    $config = $this->configFactory->get('commerce_shipping_pickup_brt.settings');

    $map_data = 'data-api-key="' . $config->get('brt_pickup_api_key') . '"';
    if ($this->validAddress($start_address)) {
      $map_data .= ' data-city="' . $start_address['locality'] . '"';
      $map_data .= ' data-zip="' . $start_address['postal_code'] . '"';
      $map_data .= ' data-address="' . $start_address['address_line1'] . '"';
    } else {
      $known_location = Json::decode($start_address['known_location'] ?? '[]');
      $map_data .= ' data-latitude="' . ($known_location[0] ?? $this->configuration['map_default_latitude'] ?? 41.8986052) . '"';
      $map_data .= ' data-longitude="' . ($known_location[1] ?? $this->configuration['map_default_longitude'] ?? 12.4757582) . '"';
    }
    $map_data .= ' data-show-data="' . $this->configuration['show_data'] . '"';

    $element = [
      '#type' => 'container',
    ];
    $element['selector'] = [
      '#type' => 'inline_template',
      '#template' => "<div id=\"pickup-brt-map-canvas\" $map_data data-pudo-type=\"100\"></div>",
    ];
    $element['id'] = [
      '#type' => 'hidden',
    ];
    $element['known_location'] = [
      '#type' => 'hidden',
      '#value' => $start_address['known_location'] ?? NULL,
    ];
    $element['name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Point name'),
      '#attributes' => [
        'readonly' => 'readonly',
        'placeholder' => t('Select point on map'),
      ],
    ];
    $element['address'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Point address'),
      '#attributes' => [
        'readonly' => 'readonly',
        'placeholder' => t('Select point on map'),
      ],
    ];

    return $element;
  }

  private function validAddress(array $address = NULL): bool {
    if (empty($address))
      return FALSE;
    else if (empty($address['country_code']) || empty($address['locality']) || empty($address['postal_code']) || empty($address['address_line1']))
      return FALSE;
    else
      return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function populateProfile(ProfileInterface $profile): void {
    $data = $profile->getData('pickup_location_data');
    $profile->set('address', [
      'country_code' => 'IT',
      'organization' => $data['name'],
      'address_line1' => $data['address'],
      'address_line3' => $data['id'],
    ]);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    $name = $form_state->getValue(['pickup_capable_shipping_information', 'shipping_profile', 'pickup_dealer', 'name']);
    $address = $form_state->getValue(['pickup_capable_shipping_information', 'shipping_profile', 'pickup_dealer', 'address']);
    if (empty($name) || empty($address)) {
      $form_state->setErrorByName('pickup_capable_shipping_information][shipping_profile][pickup_dealer', $this->t('Please, select a pickup point.'));
    }
  }
}
