(function ($, Drupal, drupalSettings) {
  var apiKey;
  var pudoType;
  var showData;
  var map;
  var group;
  var locationWarning;

  function getPointsCoords(lat, lng) {
    return new Promise((resolve, reject) => {
      $.ajax({
        url: "https://api.brt.it/pudo/v1/open/pickup/get-pudo-by-lat-lng?" + $.param({
          "countryCode": "IT",
          // BRT doesn't accept extra digits. It obviously has no physical sense, but they could just round it...
          "latitude": lat.toFixed(5),
          "longitude": lng.toFixed(5),
          "pudotype": pudoType,
        }),
        method: "GET",
        headers: {
          "X-API-Auth": apiKey,
          "Accept": "application/json",
        },
        cache: false,
        dataType: "json",
        success: function (list, status, xhr) {
          resolve(list);
        },
        error: function (xhr, status, error) {
          reject(xhr.status);
        }
      });
    });
  }

  function getPointsAddress(zip, city, address) {
    return new Promise((resolve, reject) => {
      $.ajax({
        url: "https://api.brt.it/pudo/v1/open/pickup/get-pudo-by-address?" + $.param({
          "countryCode": "IT",
          "city": city,
          "zipCode": zip,
          "address": address,
          "pudotype": pudoType,
        }),
        method: "GET",
        headers: {
          "X-API-Auth": apiKey,
          "Accept": "application/json",
        },
        cache: false,
        dataType: "json",
        success: function (list, status, xhr) {
          resolve(list);
        },
        error: function (xhr, status, error) {
          reject(xhr.status);
        }
      });
    });
  }

  function showCurrentLocation(latlng, radius) {
    group ??= new L.FeatureGroup();
    group.clearLayers();
    group.addTo(map);

    if (locationWarning != undefined)
      map.removeControl(locationWarning);

    $("input[name^='pickup_capable_shipping_information[shipping_profile][pickup_dealer][known_location]']").val(JSON.stringify([latlng.lat, latlng.lng]));
    L.circle(latlng, {
      weight: 1,
      color: "blue",
      fillColor: "#cacaff",
      fillOpacity: 0.2,
      radius: radius,
    })?.addTo(group);
  }

  async function onLocationFound(data) {
    showCurrentLocation([data.latitude, data.longitude], data.accuracy / 2);
    await onMapClick({ latlng: L.latLng(data.latitude, data.longitude) });
  }

  function onLocationError(data) {
    if (locationWarning == undefined) {
      L.Control.Button = L.Control.extend({
        options: { position: "topright" },
        onAdd: function (map) {
          const container = L.DomUtil.create("div", "leaflet-bar leaflet-control leaflet-legend");
          container.title = "Location denied";
          const button = L.DomUtil.create("div", "leaflet-legend-action", container);
          const error = Drupal.t("Location access is denied.<br>Click on the map manually.");
          button.insertAdjacentHTML("afterbegin", `<div class='action'>${error}</div>`);
          return container;
        },
        onRemove: function (map) { },
      });

      locationWarning = new L.Control.Button();
      locationWarning.addTo(map);
    }
  }

  async function onAddressSelected(data) {
    group ??= new L.FeatureGroup();
    group.clearLayers();
    group.addTo(map);

    try {
      const list = await getPointsAddress(data.zip, data.city, data.address);
      return list.pudo;
    }
    catch (e) {
      const messages = new Drupal.Message();
      const error = Drupal.t("Failed to get reply from BRT. (@error)", { "@error": e });
      messages.add(error, { type: "error" });
    }
  }

  async function onMapClick(data) {
    showCurrentLocation(data.latlng, 250);

    try {
      const list = await getPointsCoords(data.latlng.lat, data.latlng.lng);
      populateMap(list.pudo);
    }
    catch (e) {
      const messages = new Drupal.Message();
      const error = Drupal.t("Failed to get reply from BRT. (@error)", { "@error": e });
      messages.add(error, { type: "error" });
    }
  }

  function populateMap(pointList) {
    if (pointList == 0) {
      const messages = new Drupal.Message();
      const error = Drupal.t("BRT only operates in Italy.");
      messages.add(error, { type: "warning" });
    }
    else {
      for (const point of pointList) {
        if (point.available) {
          L.marker([point.latitude, point.longitude])
            .addTo(group)
            .bindPopup("", { pudoPoint: point })
            .on("popupopen", onPointClick);
        }
      }
    }
  }

  function getPointLabel(point) {
    var label = `<div class='brt-pudo' data-pudoid='${point.pudoId}'>`;
    label += `<h3>${point.pointName}</h3>`;
    label += `<p>${point.fullAddress.replace(/\s*,\s*/, "<br>")}</p>`;
    if (showData) {
      label += `<h3>${Drupal.t("Opening hours")}</h3>`;
      const allHours = new Map();
      for (const item of point.hours) {
        if (item.enabled && item.open) {
          if (allHours.has(item.dayOfWeek))
            allHours.get(item.dayOfWeek).push(`${item.openTime}-${item.closeTime}`);
          else
            allHours.set(item.dayOfWeek, [`${item.openTime}-${item.closeTime}`]);
        }
      }
      for (const [day, open] of allHours.entries()) {
        label += `<p>${drupalSettings.days_of_week[day % 7]}: ${open.join(", ")}</p>`;
      }
    }
    return label + '</div>';
  }

  function onPointClick(data) {
    const point = data.popup.options.pudoPoint;
    data.popup.setContent(getPointLabel(point));
    $("input[name^='pickup_capable_shipping_information[shipping_profile][pickup_dealer][id]']").val(point.pudoId);
    $("input[name^='pickup_capable_shipping_information[shipping_profile][pickup_dealer][name]']").val(point.pointName);
    $("input[name^='pickup_capable_shipping_information[shipping_profile][pickup_dealer][address]']").val(point.fullAddress.replace(/\s*,\s*/, ", "));
    $("input[name^='pickup_capable_shipping_information[shipping_profile][pickup_dealer][known_location]']").val(JSON.stringify([point.latitude, point.longitude]));
  }

  /**
   * Handles the BRT (Italy) selector in checkout.
   *
   * @type {Drupal~behavior}
   *
   * @prop {Drupal~behaviorAttach} attach
   */
  Drupal.behaviors.brtPudoSelected = {
    attach: function attach(context) {
      const container = L.DomUtil.get("pickup-brt-map-canvas");
      if (!container || container._leaflet_id || container._leaflet) {
        return;
      }
      apiKey = container.dataset.apiKey;
      pudoType = container.dataset.pudoType;
      showData = container.dataset.showData;

      map = L.map("pickup-brt-map-canvas")
        .on("click", onMapClick)
        .on("locationfound", onLocationFound)
        .on("locationerror", onLocationError);
      if (container.dataset.latitude != undefined && container.dataset.longitude != undefined) {
        map
          .setView([container.dataset.latitude, container.dataset.longitude], 13)
          .locate({ setView: true });
      }
      L.tileLayer("https://tile.openstreetmap.org/{z}/{x}/{y}.png", {
        attribution: "&copy; <a href='https://www.openstreetmap.org/copyright'>OpenStreetMap</a> contributors"
      }).addTo(map);
      L.control.locate().addTo(map)

      if (container.dataset.city != undefined && container.dataset.zip != undefined && container.dataset.address != undefined) {
        onAddressSelected(container.dataset).then((list) => {
          const firstPoint = list.find(point => point !== undefined);
          if (firstPoint != undefined) {
            const center = [firstPoint.latitude, firstPoint.longitude];
            map.setView(center, 13);
            showCurrentLocation(center, 250);
          }
          populateMap(list);
        });
      }
    }
  };
})(jQuery, Drupal, drupalSettings);