<?php

namespace Drupal\commerce_shipping_pickup_brt\Plugin\Commerce\ShippingMethod;

use Drupal\commerce_shipping_pickup_api\Plugin\Commerce\ShippingMethod\PickupShippingMethodBase;
use Drupal\commerce_shipping\PackageTypeManagerInterface;
use Drupal\commerce_shipping\ShippingService;
use Drupal\Component\Serialization\Json;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\profile\Entity\ProfileInterface;
use Drupal\state_machine\WorkflowManagerInterface;
use GuzzleHttp\Exception\GuzzleException;
use GuzzleHttp\RequestOptions;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a specific pickup shipping method.
 *
 * @CommerceShippingMethod(
 *   id = "pickup_it_brt_fermopoint",
 *   label = @Translation("Pickup shipping - BRT-fermopoint"),
 * )
 */
class FermopointShipping extends PickupShippingMethodBase {
  use StringTranslationTrait;

  /**
   * The config manager.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  protected $pickupPoints;

  public function __construct(array $configuration, $plugin_id, $plugin_definition, PackageTypeManagerInterface $package_type_manager, WorkflowManagerInterface $workflow_manager, ConfigFactoryInterface $configFactory) {
    parent::__construct($configuration, $plugin_id, $plugin_definition, $package_type_manager, $workflow_manager);
    $this->configFactory = $configFactory;

    $this->services['pickup'] = new ShippingService('pickup', $this->configuration['rate_label']);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('plugin.manager.commerce_package_type'),
      $container->get('plugin.manager.workflow'),
      $container->get('config.factory')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);

    $config = $this->configFactory->get('commerce_shipping_pickup_brt.settings');
    $form['brt_api_key'] = [
      '#type' => 'textfield',
      '#title' => t('BRT API key'),
      '#description' => t('Visit the <a href="@brt-service">BRT dashboard</a> to get your key.', ['@brt-service' => 'https://www.brt.it/it/servizio-clienti/codici-assistenza/']),
      '#default_value' => $config->get('brt_pickup_api_key'),
      '#required' => TRUE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    parent::submitConfigurationForm($form, $form_state);

    if (!$form_state->getErrors()) {
      $values = $form_state->getValue($form['#parents']);
      $config = $this->configFactory->getEditable('commerce_shipping_pickup_brt.settings');
      $config->set('brt_pickup_api_key', $values['brt_api_key']);
      $config->save();
    }
  }

  /**
   * {@inheritdoc}
   */
  public function buildFormElement(ProfileInterface $profile, array $start_address = NULL): array {
    if ($start_address === NULL) {
      $this->messenger()->addError('This shipping method requires the <em>Needs address from customer</em> option set in the checkout pane.');
    } else if (!empty($start_address)) {
      $this->loadPickups($start_address);
    }

    if (!empty($this->pickupPoints)) {
      return [
        '#type' => 'select',
        '#title' => $this->t('Select a pickup point:'),
        '#default_value' => $profile->getData('pickup_location_data'),
        '#options' => $this->array_map_keys(function ($pudo) {
          return [$pudo['id'], $pudo['organization']];
        }, $this->pickupPoints),
      ];
    } else
      return [];
  }

  /**
   * {@inheritdoc}
   */
  public function populateProfile(ProfileInterface $profile): void {
    $id = $profile->getData('pickup_location_data');
    if (array_key_exists($id, $this->pickupPoints)) {
      $profile->set('address', [
        'country_code' => 'IT',
        'address_line1' => $this->pickupPoints[$id]['organization'],
        'address_line3' => $id,
      ]);
    }
  }

  /**
   * Load pickup points from provider.
   *
   * @return array
   *   List of pickup points.
   */
  private function loadPickups(array $address) {
    $config = $this->configFactory->get('commerce_shipping_pickup_brt.settings');
    try {
      $url = 'https://api.brt.it/pudo/v1/open/pickup/get-pudo-by-address';
      $headers = [
        'X-API-Auth' => $config->get('brt_pickup_api_key'),
      ];
      $query = [
        'city' => $address['locality'] ?? '',
        'countryCode' => 'IT',
        'zipCode' => $address['postal_code'] ?? '',
        'address' => $address['address_line1'] ?? '',
        'pudotype' => '100',
      ];
      $response = \Drupal::httpClient()->request('GET', $url, [
        RequestOptions::HEADERS => $headers,
        RequestOptions::QUERY => $query,
      ]);
      $json = Json::decode($response->getBody());
      $this->pickupPoints = $this->array_map_keys(function ($pudo) {
        return [$pudo['pudoId'], [
          'id' => $pudo['pudoId'],
          'name' => $pudo['pointName'],
          'organization' => $pudo['pointName'] . ' (' . str_replace(',', ', ', $pudo['fullAddress']) . ')',
          'locality' => $pudo['town'],
          'postal_code' => $pudo['zipCode'],
          'address_line1' => $pudo['street'],
          'address_line2' => $pudo['street2'],
          'address_line3' => $pudo['street3'],
        ]];
      }, $json['pudo']);
    } catch (GuzzleException $e) {
      $this->messenger()->addError('Failed to get reply from BRT.');
      $this->pickupPoints = [];
    }
  }

  private function array_map_keys(callable $callback, array $array): array {
    $result = [];
    foreach ($array as $value) {
      [$new_key, $new_value] = $callback($value);
      $result[$new_key] = $new_value;
    }
    return $result;
  }
}
