<?php

namespace Drupal\commerce_shipping_pickup_gls_map_widget\Plugin\Commerce\ShippingMethod;

use Drupal\commerce_shipping_pickup_api\Plugin\Commerce\ShippingMethod\PickupShippingMethodBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Template\Attribute;
use Drupal\profile\Entity\ProfileInterface;

/**
 * Provides a specific pickup shipping method.
 *
 * @CommerceShippingMethod(
 *   id = "pickup_gls_map_widget",
 *   label = @Translation("Pickup shipping - GLS Map Widget"),
 * )
 */
class GlsMapWidgetShipping extends PickupShippingMethodBase {

  use StringTranslationTrait;

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
        'widget_height'             => 600,
        'widget_modal'              => TRUE,
        'widget_modal_btn_text'     => 'Select shipping address',
        'widget_modal_btn_class'    => '',
        'widget_country'            => '',
        'widget_language'           => '',
        'widget_dropoffpoints_only' => FALSE,
        'widget_filter_type'        => '',
      ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);

    $form['widget_modal'] = [
      '#type'          => 'checkbox',
      '#title'         => $this->t('Use modal dialog'),
      '#description'   => $this->t('The GLS map will open in a modal dialog.'),
      '#default_value' => $this->configuration['widget_modal'],
    ];

    $form['widget_height'] = [
      '#type'          => 'number',
      '#title'         => $this->t('Widget height'),
      '#default_value' => $this->configuration['widget_height'],
      '#required'      => TRUE,
      '#states'        => [
        'visible' => [
          'input[name="plugin[0][target_plugin_configuration][pickup_gls_map_widget][widget_modal]"]' => ['checked' => FALSE],
        ],
      ],
    ];

    $form['widget_modal_btn_text'] = [
      '#type'          => 'textfield',
      '#title'         => $this->t('Text of the button used to open the Modal dialog'),
      '#description'   => $this->t("Enter the button text in english, as the text you enter will be passed to the translation function."),
      '#default_value' => $this->configuration['widget_modal_btn_text'],
      '#required'      => TRUE,
      '#states'        => [
        'invisible' => [
          'input[name="plugin[0][target_plugin_configuration][pickup_gls_map_widget][widget_modal]"]' => ['checked' => FALSE],
        ],
      ],
    ];

    $form['widget_modal_btn_class'] = [
      '#type'          => 'textfield',
      '#title'         => $this->t('Extra class for Modal opener button'),
      '#default_value' => $this->configuration['widget_modal_btn_class'],
      '#states'        => [
        'invisible' => [
          'input[name="plugin[0][target_plugin_configuration][pickup_gls_map_widget][widget_modal]"]' => ['checked' => FALSE],
        ],
      ],
    ];

    $form['widget_country'] = [
      '#type'          => 'select',
      '#title'         => $this->t('Country'),
      '#description'   => $this->t('The map will only show the delivery points of this country.'),
      '#options'       => [
        ''   => $this->t('-- Select --'),
        'AT' => $this->t('Austria'),
        'BE' => $this->t('Belgium'),
        'BG' => $this->t('Bulgaria'),
        'CZ' => $this->t('Czech Republic'),
        'DE' => $this->t('Germany'),
        'DK' => $this->t('Denmark'),
        'ES' => $this->t('Spain'),
        'FI' => $this->t('Finland'),
        'FR' => $this->t('France'),
        'GR' => $this->t('Greece'),
        'HR' => $this->t('Croatia'),
        'HU' => $this->t('Hungary'),
        'IT' => $this->t('Italy'),
        'LU' => $this->t('Luxembourg'),
        'NL' => $this->t('Netherlands'),
        'PL' => $this->t('Poland'),
        'PT' => $this->t('Portugal'),
        'RO' => $this->t('Romania'),
        'RS' => $this->t('Serbia'),
        'SI' => $this->t('Slovenia'),
        'SK' => $this->t('Slovakia'),
      ],
      '#default_value' => $this->configuration['widget_country'],
    ];

    $form['widget_language'] = [
      '#type'          => 'select',
      '#title'         => $this->t('Language'),
      '#description'   => $this->t('If omitted, the default language is based on the selected country.'),
      '#options'       => [
        ''   => $this->t('-- Select --'),
        'CS' => $this->t('Czech'),
        'HR' => $this->t('Croatian'),
        'HU' => $this->t('Hungarian'),
        'RO' => $this->t('Romanian'),
        'SR' => $this->t('Serbian'),
        'SL' => $this->t('Slovene'),
        'SK' => $this->t('Slovak'),
        'PL' => $this->t('Polish'),
        'EN' => $this->t('English'),
        'DE' => $this->t('German'),
        'FR' => $this->t('French'),
        'ES' => $this->t('Spanish'),
        'IT' => $this->t('Italian'),
      ],
      '#default_value' => $this->configuration['widget_language'],
    ];

    $form['widget_dropoffpoints_only'] = [
      '#type'          => 'checkbox',
      '#title'         => $this->t('Drop-off points only'),
      '#description'   => $this->t('The widget will list only drop-off points.'),
      '#default_value' => $this->configuration['widget_dropoffpoints_only'],
    ];

    $form['widget_filter_type'] = [
      '#type'          => 'select',
      '#title'         => $this->t('Filter type'),
      '#description'   => $this->t('The widget will list only this type of delivery points.'),
      '#options'       => [
        ''              => $this->t('-- Select --'),
        'parcel-shop'   => $this->t('ParcelShop'),
        'parcel-locker' => $this->t('ParcelLocker'),
      ],
      '#default_value' => $this->configuration['widget_filter_type'],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    parent::submitConfigurationForm($form, $form_state);

    if (!$form_state->getErrors()) {
      $values = $form_state->getValue($form['#parents']);
      $this->configuration['widget_height'] = (int) $values['widget_height'];
      $this->configuration['widget_modal'] = !empty($values['widget_modal']);
      $this->configuration['widget_modal_btn_text'] = $values['widget_modal_btn_text'];
      $this->configuration['widget_modal_btn_class'] = $values['widget_modal_btn_class'];
      $this->configuration['widget_country'] = $values['widget_country'];
      $this->configuration['widget_language'] = $values['widget_language'];
      $this->configuration['widget_dropoffpoints_only'] = !empty($values['widget_dropoffpoints_only']);
      $this->configuration['widget_filter_type'] = $values['widget_filter_type'];
    }
  }

  /**
   * {@inheritdoc}
   */
  public function buildFormElement(ProfileInterface $profile, array $start_address = NULL): array {
    $height = (int) ($this->configuration['widget_height'] ?? 0);
    $is_modal = !empty($this->configuration['widget_modal']);

    $attributes = [];

    if (!empty($this->configuration['widget_country'])) {
      $attributes['country'] = $this->configuration['widget_country'];
    }

    if (!empty($this->configuration['widget_language'])) {
      $attributes['language'] = $this->configuration['widget_language'];
    }

    if (!empty($this->configuration['widget_dropoffpoints_only'])) {
      $attributes['dropoffpoints-only'] = 1;
    }

    if (!empty($this->configuration['widget_filter_type'])) {
      $attributes['filter-type'] = $this->configuration['widget_filter_type'];
    }

    $attributes = new Attribute($attributes);

    $tag_name = $is_modal ? 'gls-dpm-dialog' : 'gls-dpm';
    $html_tag = '<' . $tag_name . ' id="gls-map-widget--map" ' . $attributes . '></' . $tag_name . '>';

    $values = $profile->getData('pickup_location_data');

    $element = [
      '#type' => 'container',
    ];

    $element['name'] = [
      '#type'          => 'textfield',
      '#title'         => $this->t('Shipping address'),
      '#placeholder'   => $this->t('Please select a shipping address on the map.'),
      '#default_value' => $values['name'] ?? '',
      '#attributes'    => [
        'readonly' => 'readonly',
        'class'    => [
          'commerce-shipping-pickup--gls-name',
        ],
      ],
      '#required'      => TRUE,
    ];

    if (!$is_modal) {
      $html_tag = '<div style="height: ' . $height . 'px">' . $html_tag . '</div>';
    }

    $element['selector'] = [
      '#type'     => 'inline_template',
      '#template' => $html_tag,
    ];

    if ($is_modal) {
      $element['modal_opener'] = [
        '#type'       => 'html_tag',
        '#tag'        => 'button',
        '#value'      => t(!empty($this->configuration['widget_modal_btn_text']) ? $this->configuration['widget_modal_btn_text'] : 'Select shipping address'),
        '#attributes' => [
          'class' => [
            'commerce-shipping-pickup--modal-opener',
            'button',
            'button--secondary',
            ($this->configuration['widget_modal_btn_class'] ?? ''),
          ],
        ],
      ];
    }

    $element['id'] = [
      '#type'               => 'textfield',
      '#title'              => $this->t('Identifier'),
      '#default_value'      => $values['id'] ?? '',
      '#attributes'         => [
        'readonly' => 'readonly',
        'class'    => [
          'commerce-shipping-pickup--gls-identifier',
        ],
      ],
      '#wrapper_attributes' => [
        'class' => [
          'commerce-shipping-pickup-hidden',
        ],
      ],
    ];

    $element['country_code'] = [
      '#type'               => 'textfield',
      '#title'              => $this->t('Country code'),
      '#default_value'      => $values['country_code'] ?? '',
      '#attributes'         => [
        'readonly' => 'readonly',
        'class'    => [
          'commerce-shipping-pickup--gls-country-code',
        ],
      ],
      '#wrapper_attributes' => [
        'class' => [
          'commerce-shipping-pickup-hidden',
        ],
      ],
    ];

    $element['address_line1'] = [
      '#type'               => 'textfield',
      '#title'              => $this->t('Postal address'),
      '#default_value'      => $values['address_line1'] ?? '',
      '#attributes'         => [
        'readonly' => 'readonly',
        'class'    => [
          'commerce-shipping-pickup--gls-postal-address',
        ],
      ],
      '#wrapper_attributes' => [
        'class' => [
          'commerce-shipping-pickup-hidden',
        ],
      ],
    ];

    $element['locality'] = [
      '#type'               => 'textfield',
      '#title'              => $this->t('Locality'),
      '#default_value'      => $values['locality'] ?? '',
      '#attributes'         => [
        'readonly' => 'readonly',
        'class'    => [
          'commerce-shipping-pickup--gls-locality',
        ],
      ],
      '#wrapper_attributes' => [
        'class' => [
          'commerce-shipping-pickup-hidden',
        ],
      ],
    ];

    $element['postal_code'] = [
      '#type'               => 'textfield',
      '#title'              => $this->t('Postal code'),
      '#default_value'      => $values['postal_code'] ?? '',
      '#attributes'         => [
        'readonly' => 'readonly',
        'class'    => [
          'commerce-shipping-pickup--gls-postal-code',
        ],
      ],
      '#wrapper_attributes' => [
        'class' => [
          'commerce-shipping-pickup-hidden',
        ],
      ],
    ];

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function populateProfile(ProfileInterface $profile): void {
    $data = $profile->getData('pickup_location_data');
    $profile->set('address', [
      'organization'  => $data['name'] . ' (' . $data['id'] . ')',
      'country_code'  => $data['country_code'],
      'address_line1' => $data['address_line1'],
      'locality'      => $data['locality'],
      'postal_code'   => $data['postal_code'],
    ]);
  }

}
