<?php

namespace Drupal\commerce_shipping_po_box_condition\Plugin\Commerce\Condition;

use Drupal\commerce\Plugin\Commerce\Condition\ConditionBase;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Form\FormStateInterface;

/**
 * Provides the shipping PO Box condition for shipments.
 *
 * @CommerceCondition(
 *   id = "shipment_po_box",
 *   label = @Translation("Shipping PO Box"),
 *   category = @Translation("Customer"),
 *   entity_type = "commerce_shipment",
 *   weight = 10,
 * )
 */
class ShipmentPOBox extends ConditionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'negate' => NULL,
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $form = parent::buildConfigurationForm($form, $form_state);

    $form['info'] = [
      '#type' => 'item',
      '#markup' => 'Checks for the existence of a PO Box in the address.',
    ];
    $form['negate'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Negate'),
      '#description' => $this->t('If checked, the condition succeeds if there is no PO Box.'),
      '#default_value' => $this->configuration['negate'],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    parent::submitConfigurationForm($form, $form_state);

    $values = $form_state->getValue($form['#parents']);

    $this->configuration['negate'] = $values['negate'];
  }

  /**
   * {@inheritdoc}
   */
  public function evaluate(EntityInterface $entity) {
    $this->assertEntity($entity);
    /** @var \Drupal\commerce_shipping\Entity\ShipmentInterface $shipment */
    $shipment = $entity;
    $shipping_profile = $shipment->getShippingProfile();
    if (!$shipping_profile) {
      return FALSE;
    }

    /** @var \Drupal\address\Plugin\Field\FieldType\AddressItem $address */
    $address = $shipping_profile->get('address')->first();
    if (!$address) {
      // The conditions can't be applied until the shipping address is known.
      return FALSE;
    }

    $contains_po_box = FALSE;
    // Match case insensitive various spellings of PO BOX ##.
    // PO Box 555, PO Box555, POBox 555, POBox555, and variations with dots.
    if (preg_match('/\bP\.?O\.?\s*Box[\d\s]/i', $address->getAddressLine1())) {
      $contains_po_box = TRUE;
    }

    // Match case insensitive various spellings of HC ## BOX ##.
    // HC 55 Box 555, HC 55 Box555, HC55Box 555, and HC55Box555.
    elseif (preg_match('/\bHC\s*\d*\s*Box[\d\s]/i', $address->getAddressLine1())) {
      $contains_po_box = TRUE;
    }

    // If negate is checked then return the opposite condition result.
    return $this->configuration['negate'] ? !$contains_po_box : $contains_po_box;
  }

}
