<?php

declare(strict_types=1);

namespace Drupal\commerce_store_dashboard\Access;

use Drupal\commerce_store\Entity\StoreInterface;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Routing\Access\AccessInterface;
use Drupal\Core\Session\AccountInterface;

/**
 * Provides access check for the store dashboard.
 */
class CommerceStoreDashboardAccessCheck implements AccessInterface {

  /**
   * Check if the user can access the store dashboard.
   *
   * @param \Drupal\commerce_store\Entity\StoreInterface $commerce_store
   *   The commerce store.
   * @param \Drupal\Core\Session\AccountInterface $account
   *   The current user account.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   The access result.
   */
  public function access(StoreInterface $commerce_store, AccountInterface $account): AccessResultInterface {
    // If the user can bypass the permission or owns the store and has
    // permission to access their own store dashboard, allow access.
    // Otherwise, deny access.
    $bypass = $account->hasPermission('bypass commerce_store dashboard access');
    $own_store = $commerce_store->getOwnerId() === $account->id() && $account->hasPermission('access own commerce_store dashboard');

    if ($bypass || $own_store) {
      return AccessResult::allowed()->addCacheContexts(['user']);
    }

    return AccessResult::forbidden()->addCacheContexts(['user.permissions']);
  }

}
