<?php

declare(strict_types=1);

namespace Drupal\commerce_store_dashboard\Hook;

use Drupal\commerce_store_dashboard\Access\CommerceStoreDashboardAccessCheck;
use Drupal\Core\DependencyInjection\ClassResolverInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\Routing\RedirectDestinationInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Url;
use Drupal\user\UserInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Provides hooks for Commerce Store Dashboard.
 */
class CommerceStoreDashboardHooks {

  use StringTranslationTrait;

  /**
   * Creates a CommerceStoreDashboardHooks instance.
   *
   * @param \Drupal\Core\DependencyInjection\ClassResolverInterface $classResolver
   *   The class resolver.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Symfony\Component\HttpFoundation\RequestStack $requestStack
   *   The request stack.
   * @param \Drupal\Core\Routing\RouteMatchInterface $currentRouteMatch
   *   The current route match.
   * @param \Drupal\Core\Routing\RedirectDestinationInterface $redirectDestination
   *   The redirect destination service.
   * @param \Drupal\Core\Session\AccountProxyInterface $currentUser
   *   The current user.
   */
  public function __construct(
    protected ClassResolverInterface $classResolver,
    protected EntityTypeManagerInterface $entityTypeManager,
    protected RequestStack $requestStack,
    protected RouteMatchInterface $currentRouteMatch,
    protected RedirectDestinationInterface $redirectDestination,
    protected AccountProxyInterface $currentUser,
  ) {
  }

  /**
   * Implements hook_entity_type_alter().
   *
   * Adds dashboard link template to commerce_store entity.
   */
  #[Hook('entity_type_alter')]
  public function entityTypeAlter(array &$entity_types): void {
    /** @var \Drupal\Core\Entity\EntityTypeInterface[] $entity_types */
    $entity_types['commerce_store']->setLinkTemplate('dashboard', '/store/{commerce_store}/dashboard');
  }

  /**
   * Implements hook_entity_operation_alter().
   *
   * Adds a "Dashboard" operation to commerce_store entities.
   */
  #[Hook('entity_operation_alter')]
  public function entityOperationAlter(array &$operations, EntityInterface $entity): void {
    /** @var \Drupal\commerce_store\Entity\StoreInterface $entity */
    if ($entity->getEntityTypeId() === 'commerce_store') {
      $access_check = $this->classResolver->getInstanceFromDefinition(CommerceStoreDashboardAccessCheck::class);
      if (!$access_check->access($entity, $this->currentUser)->isAllowed()) {
        return;
      }

      $operations['dashboard'] = [
        'title' => $this->t('Dashboard'),
        'url' => Url::fromRoute('entity.commerce_store.dashboard', [
          'commerce_store' => $entity->id(),
        ]),
      ];
    }
  }

  /**
   * Implements hook_form_FORM_ID_alter() for commerce_store_type_form.
   */
  #[Hook('form_commerce_store_type_form_alter')]
  public function formCommerceStoreTypeFormAlter(array &$form, FormStateInterface $form_state): void {
    /** @var \Drupal\commerce_store\Entity\StoreTypeInterface $store_type */
    $store_type = $form_state->getFormObject()->getEntity();

    $form['dashboard'] = [
      '#type' => 'details',
      '#title' => $this->t('Dashboard settings'),
    ];
    $form['dashboard']['redirect_after_login'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Redirect to store dashboard after login'),
      '#description' => $this->t('If enabled, users associated with this store will be redirected to the store dashboard upon login.'),
      '#default_value' => $store_type->getThirdPartySetting('commerce_store_dashboard', 'redirect_after_login', FALSE),
      '#return_value' => TRUE,
      '#config_target' => 'commerce_store.commerce_store_type.' . $store_type->id() . ':third_party_settings.commerce_store_dashboard.redirect_after_login',
    ];
    $form['actions']['submit']['#submit'][] = 'commerce_store_dashboard_store_type_form_submit';
  }

  /**
   * Implements hook_user_login().
   */
  #[Hook('user_login')]
  public function userLogin(UserInterface $account): void {
    // If a destination is already set, do not override it.
    $destination = $this->redirectDestination->getAsArray();
    if ((!empty($destination['destination']) && $destination['destination'] !== Url::fromRoute('user.login')->toString()) ||
      // Or if the current route is the password reset login route, do not
      // override the destination, and allow the user to set a new password.
      $this->currentRouteMatch->getRouteName() === 'user.reset.login') {
      return;
    }

    // Otherwise, check if the user is associated with a store that has the
    // redirect_after_login setting enabled and redirect them to the store
    // dashboard.
    /** @var \Drupal\commerce_store\Entity\StoreInterface[] $stores */
    $stores = $this->entityTypeManager
      ->getStorage('commerce_store')
      ->loadByProperties(['uid' => $account->id()]);
    foreach ($stores as $store) {
      /** @var \Drupal\commerce_store\Entity\StoreTypeInterface $store_type */
      $store_type = $store->type->entity;
      $redirect_after_login = $store_type->getThirdPartySetting('commerce_store_dashboard', 'redirect_after_login', FALSE);
      if ($redirect_after_login) {
        $url = Url::fromRoute('entity.commerce_store.dashboard', [
          'commerce_store' => $store->id(),
        ]);
        $this->requestStack->getCurrentRequest()->query->set('destination', $url->toString());
        return;
      }
    }
  }

}
