<?php

namespace Drupal\Tests\commerce_store_dashboard\Functional;

use PHPUnit\Framework\Attributes\Group;
use PHPUnit\Framework\Attributes\IgnoreDeprecations;

/**
 * Tests store dashboard access.
 */
#[Group('commerce_store_dashboard')]
#[IgnoreDeprecations]
class CommerceStoreDashboardAccessTest extends CommerceStoreDashboardFunctionalTestBase {

  /**
   * Tests 'bypass commerce_store dashboard access' permission.
   */
  public function testBypassAccess() {
    // Ensure that the admin user has the bypass permission and can access the
    // store dashboard.
    $this->drupalLogin($this->adminUser);
    $this->assertTrue($this->adminUser->hasPermission('bypass commerce_store dashboard access'));
    $this->drupalGet($this->store->toUrl('dashboard')->toString());
    $this->assertSession()->statusCodeEquals(200);
    $this->assertCacheContext('user');

    // Create a new user without the bypass permission and ensure they cannot
    // access the store dashboard.
    $user = $this->drupalCreateUser();
    $this->drupalLogin($user);
    $this->assertFalse($user->hasPermission('bypass commerce_store dashboard access'));
    $this->drupalGet($this->store->toUrl('dashboard')->toString());
    $this->assertSession()->statusCodeEquals(403);
    $this->assertCacheContext('user.permissions');

    // Create a new store owned by the new user and ensure that admin can still
    // access the dashboard.
    $user_store = $this->createStore();
    $user_store->setOwner($user);
    $user_store->save();

    $this->drupalLogin($this->adminUser);
    $this->drupalGet($user_store->toUrl('dashboard')->toString());
    $this->assertSession()->statusCodeEquals(200);
    $this->assertCacheContext('user');

    // Remove the bypass permission from admin and ensure they can no longer
    // access the dashboard.
    foreach ($this->adminUser->getRoles() as $role_id) {
      user_role_revoke_permissions($role_id, ['bypass commerce_store dashboard access']);
    }
    $this->drupalGet($user_store->toUrl('dashboard')->toString());
    $this->assertSession()->statusCodeEquals(403);
    $this->assertCacheContext('user.permissions');
  }

  /**
   * Tests 'access own commerce_store dashboard' permission.
   */
  public function testOwnAccess() {
    // Create three test users.
    $user_1 = $this->drupalCreateUser(['access own commerce_store dashboard']);
    $user_1_store = $this->createStore();
    $user_1_store->setOwner($user_1);
    $user_1_store->save();

    $user_2 = $this->drupalCreateUser(['access own commerce_store dashboard']);
    $user_2_store = $this->createStore();
    $user_2_store->setOwner($user_2);
    $user_2_store->save();

    // Another user without the permission or store.
    $user_3 = $this->drupalCreateUser();

    // Test admin user can access both dashboards.
    $this->drupalGet($user_1_store->toUrl('dashboard')->toString());
    $this->assertSession()->statusCodeEquals(200);
    $this->assertCacheContext('user');
    $this->drupalGet($user_2_store->toUrl('dashboard')->toString());
    $this->assertSession()->statusCodeEquals(200);
    $this->assertCacheContext('user');

    // Test user 1 can access their own dashboard but not user 2's.
    $this->drupalLogin($user_1);
    $this->drupalGet($user_1_store->toUrl('dashboard')->toString());
    $this->assertSession()->statusCodeEquals(200);
    $this->assertCacheContext('user');
    $this->drupalGet($user_2_store->toUrl('dashboard')->toString());
    $this->assertSession()->statusCodeEquals(403);
    $this->assertCacheContext('user.permissions');

    // Test user 2 can access their own dashboard but not user 1's.
    $this->drupalLogin($user_2);
    $this->drupalGet($user_2_store->toUrl('dashboard')->toString());
    $this->assertSession()->statusCodeEquals(200);
    $this->assertCacheContext('user');
    $this->drupalGet($user_1_store->toUrl('dashboard')->toString());
    $this->assertSession()->statusCodeEquals(403);
    $this->assertCacheContext('user.permissions');

    // Test user 3 cannot access either dashboard.
    $this->drupalLogin($user_3);
    $this->drupalGet($user_1_store->toUrl('dashboard')->toString());
    $this->assertSession()->statusCodeEquals(403);
    $this->assertCacheContext('user.permissions');
    $this->drupalGet($user_2_store->toUrl('dashboard')->toString());
    $this->assertSession()->statusCodeEquals(403);
    $this->assertCacheContext('user.permissions');
  }

}
