<?php

namespace Drupal\Tests\commerce_store_dashboard\Functional;

use PHPUnit\Framework\Attributes\Group;
use PHPUnit\Framework\Attributes\IgnoreDeprecations;

/**
 * Tests destination path after user login.
 */
#[Group('commerce_store_dashboard')]
#[IgnoreDeprecations]
class CommerceStoreDashboardUserLoginTest extends CommerceStoreDashboardFunctionalTestBase {

  /**
   * {@inheritdoc}
   *
   * Disable one-time login links for this test since they interfere with
   * testing redirection after login.
   */
  protected bool $useOneTimeLoginLinks = FALSE;

  /**
   * Checks that the user is redirected to the dashboard after login.
   *
   * When the "Redirect to dashboard after login" option is enabled, the user
   * should be redirected to the store dashboard after login.
   *
   * @see \Drupal\commerce_store_dashboard\Hook\CommerceStoreDashboardHooks::userLogin()
   */
  public function testRedirectionEnabled(): void {
    $store_type = $this->store->type->entity;
    $store_type->setThirdPartySetting('commerce_store_dashboard', 'redirect_after_login', TRUE);
    $store_type->save();

    // Test for admin user, who is not associated with the store.
    // It should not redirect to the dashboard.
    $this->drupalLogout();
    $this->drupalLogin($this->adminUser);
    $this->assertSession()->addressNotEquals('/store/' . $this->store->id() . '/dashboard');

    // Set the admin user as the store owner and test again. It should redirect
    // to the dashboard.
    $this->store->setOwner($this->adminUser);
    $this->store->save();
    $this->drupalLogout();
    $this->drupalLogin($this->adminUser);
    $this->assertSession()->addressEquals('/store/' . $this->store->id() . '/dashboard');
  }

  /**
   * Checks that the user is not redirected to the dashboard after login.
   *
   * When the "Redirect to dashboard after login" option is enabled, the user
   * should be redirected to the store dashboard after login, but if the user
   * uses a one-time login link for password reset, they should not be
   * redirected.
   *
   * @see \Drupal\commerce_store_dashboard\Hook\CommerceStoreDashboardHooks::userLogin()
   */
  public function testRedirectionEnabledPasswordRest(): void {
    $this->useOneTimeLoginLinks = TRUE;
    $store_type = $this->store->type->entity;
    $store_type->setThirdPartySetting('commerce_store_dashboard', 'redirect_after_login', TRUE);
    $store_type->save();

    // Test for admin user, who is not associated with the store.
    // It should not redirect to the dashboard.
    $this->drupalLogout();
    $this->drupalLogin($this->adminUser);
    $this->assertSession()->addressNotEquals('/store/' . $this->store->id() . '/dashboard');

    // Set the admin user as the store owner and test again. It should not be
    // redirected to the dashboard.
    $this->store->setOwner($this->adminUser);
    $this->store->save();
    $this->drupalLogout();
    $this->drupalLogin($this->adminUser);
    $this->assertSession()->addressNotEquals('/store/' . $this->store->id() . '/dashboard');
  }

  /**
   * Checks that the user is not redirected to the dashboard after login.
   *
   * When the "Redirect to dashboard after login" option is disabled, the user
   * should not be redirected to the store dashboard after login.
   *
   * @see \Drupal\commerce_store_dashboard\Hook\CommerceStoreDashboardHooks::userLogin()
   */
  public function testRedirectionNotEnabled(): void {
    $store_type = $this->store->type->entity;
    $store_type->setThirdPartySetting('commerce_store_dashboard', 'redirect_after_login', FALSE);
    $store_type->save();

    // Test for admin user, who is not associated with the store.
    // It should not be redirected to the dashboard.
    $this->drupalLogout();
    $this->drupalLogin($this->adminUser);
    $this->assertSession()->addressNotEquals('/store/' . $this->store->id() . '/dashboard');

    // Set the admin user as the store owner and test again. It should not
    // be redirected to the dashboard as the setting is disabled.
    $this->store->setOwner($this->adminUser);
    $this->store->save();
    $this->drupalLogout();
    $this->drupalLogin($this->adminUser);
    $this->assertSession()->addressNotEquals('/store/' . $this->store->id() . '/dashboard');
  }

}
