<?php

namespace Drupal\commerce_swish\Plugin\Commerce\PaymentGateway;

use Drupal\commerce_payment\Attribute\CommercePaymentGateway;
use Drupal\commerce_payment\Entity\PaymentInterface;
use Drupal\commerce_payment\Plugin\Commerce\PaymentGateway\OffsitePaymentGatewayBase;
use Drupal\commerce_payment\Plugin\Commerce\PaymentGateway\SupportsRefundsInterface;
use Drupal\commerce_payment\Plugin\Commerce\PaymentGateway\SupportsVoidsInterface;
use Drupal\commerce_payment\PluginForm\PaymentReceiveForm;
use Drupal\commerce_price\Price;
use Drupal\commerce_swish\PluginForm\OffsiteRedirect\SwishCheckoutPaymentForm;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;

/**
 * Provides the Swish checkout payment gateway.
 */
#[CommercePaymentGateway(
  id: "commerce_swish_checkout",
  label: new TranslatableMarkup("Swish"),
  display_label: new TranslatableMarkup("Swish"),
  modes: [
    "qr_code" => new TranslatableMarkup("QR code mode"),
    "sandbox" => new TranslatableMarkup("Swish Sandbox"),
    "production" => new TranslatableMarkup("Production Environment"),
  ],
  forms: [
    "offsite-payment" => SwishCheckoutPaymentForm::class,
    "receive-payment" => PaymentReceiveForm::class,
  ],
  payment_type: "payment_manual",
  requires_billing_information: FALSE,
)]
class SwishCheckoutPaymentGateway extends OffsitePaymentGatewayBase implements SupportsVoidsInterface, SupportsRefundsInterface {

  /**
   * The logger.
   */
  protected LoggerInterface $logger;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->logger = $container->get('logger.channel.commerce_swish');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'information_value' => '',
      'information_format' => 'plain_text',
      'log' => 0,
      'qr_code' => 1,
      'payee' => '',
      'show_vat_included' => 1,
      'prefix' => '',
      'submit_status' => 'pending',
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);

    // Production and sandbox modes are not yet implemented.
    $form['not_implemented'] = [
      '#type' => 'container',
      '#markup' => $this->t('<strong>Note:</strong> Production and Sandbox modes are not yet implemented. Currently, only QR Code mode is available.'),
      '#states' => [
        'invisible' => [
          ':input[name="configuration[commerce_swish_checkout][mode]"]' => ['value' => 'qr_code'],
        ],
      ],
    ];

    $form['payee'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Swish number'),
      '#default_value' => $this->configuration['payee'],
      '#description' => $this->t('The Swish number (e.g. 1231234567) that will receive the payment.'),
      '#required' => TRUE,
    ];

    $form['qr_code'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Display QR code on the payment page'),
      '#default_value' => $this->configuration['qr_code'],
      '#options' => [
        1 => $this->t('Yes'),
        0 => $this->t('No'),
      ],
    ];
    $form['show_vat_included'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show VAT included after the amount'),
      '#default_value' => $this->configuration['show_vat_included'],
      '#options' => [
        1 => $this->t('Yes'),
        0 => $this->t('No'),
      ],
    ];
    $form['prefix'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Prefix'),
      '#default_value' => $this->configuration['prefix'],
      '#description' => $this->t('Prefix for the payment message, e.g. your store name.'),
    ];

    $form['information'] = [
      '#type' => 'text_format',
      '#title' => $this->t('Information'),
      '#default_value' => $this->configuration['information_value'],
      '#format' => $this->configuration['information_format'],
    ];

    $form['log'] = [
      '#type' => 'select',
      '#title' => $this->t('Extended logging'),
      '#default_value' => $this->configuration['log'],
      '#options' => [
        '0' => $this->t('No'),
        '1' => $this->t('Yes'),
      ],
      '#required' => TRUE,
    ];

    $form['submit_status'] = [
      '#type' => 'select',
      '#title' => $this->t('After user has confirmed that the payment is done. What status should the payment have?'),
      '#default_value' => $this->configuration['submit_status'],
      '#options' => [
        'pending' => $this->t('Pending'),
        'completed' => $this->t('Completed'),
      ],
      '#required' => TRUE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    parent::submitConfigurationForm($form, $form_state);
    $values = $form_state->getValue($form['#parents']);
    $this->configuration['payee'] = $values['payee'];
    $this->configuration['qr_code'] = $values['qr_code'];
    $this->configuration['information_value'] = $values['information']['value'];
    $this->configuration['information_format'] = $values['information']['format'];
    $this->configuration['log'] = $values['log'];
    $this->configuration['show_vat_included'] = $values['show_vat_included'];
    $this->configuration['prefix'] = $values['prefix'];
    $this->configuration['submit_status'] = $values['submit_status'];
  }

  /**
   * {@inheritdoc}
   */
  public function onNotify(Request $request): void {
    $order_id = $request->query->get('order_id');

    if ($this->configuration['log']) {
      // Log Notify.
      $this->logger->debug('Notify called with order_id: %order_id', ['%order_id' => $order_id]);
    }

    // @todo Load payment information from Swish.
    // @todo Update payment status in Drupal Commerce.
  }

  /**
   * {@inheritdoc}
   */
  public function buildPaymentOperations(PaymentInterface $payment): array {
    $operations = [];
    $operations['receive'] = [
      'title' => $this->t('Receive'),
      'page_title' => $this->t('Receive payment'),
      'plugin_form' => 'receive-payment',
      'access' => $payment->getState()->getId() === 'pending',
    ];
    $operations['void'] = [
      'title' => $this->t('Void'),
      'page_title' => $this->t('Void payment'),
      'plugin_form' => 'void-payment',
      'access' => $this->canVoidPayment($payment),
    ];
    $operations['refund'] = [
      'title' => $this->t('Refund'),
      'page_title' => $this->t('Refund payment'),
      'plugin_form' => 'refund-payment',
      'access' => $this->canRefundPayment($payment),
    ];

    return $operations;
  }

  /**
   * {@inheritdoc}
   */
  public function receivePayment(PaymentInterface $payment, ?Price $amount = NULL): void {
    $this->assertPaymentState($payment, ['pending']);

    // If not specified, use the entire amount.
    $amount = $amount ?: $payment->getAmount();
    $payment->setState('completed');
    $payment->setAmount($amount);
    $payment->save();
  }

  /**
   * {@inheritdoc}
   */
  public function canVoidPayment(PaymentInterface $payment): bool {
    return $payment->getState()->getId() === 'pending';
  }

  /**
   * {@inheritdoc}
   */
  public function voidPayment(PaymentInterface $payment): void {
    $this->assertPaymentState($payment, ['pending']);
    $payment->setState('voided');
    $payment->save();
  }

  /**
   * {@inheritdoc}
   */
  public function refundPayment(PaymentInterface $payment, ?Price $amount = NULL): void {
    $this->assertPaymentState($payment, ['completed', 'partially_refunded']);
    // If not specified, refund the entire amount.
    $amount = $amount ?: $payment->getAmount();
    $this->assertRefundAmount($payment, $amount);

    $old_refunded_amount = $payment->getRefundedAmount();
    $new_refunded_amount = $old_refunded_amount->add($amount);
    if ($new_refunded_amount->lessThan($payment->getAmount())) {
      $payment->setState('partially_refunded');
    }
    else {
      $payment->setState('refunded');
    }

    $payment->setRefundedAmount($new_refunded_amount);
    $payment->save();
  }

}
