<?php

namespace Drupal\commerce_swish\PluginForm\OffsiteRedirect;

use Drupal\commerce_payment\PluginForm\PaymentOffsiteForm;
use Drupal\Component\Uuid\UuidInterface;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use GuzzleHttp\Client;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Swish Checkout.
 */
class SwishCheckoutPaymentForm extends PaymentOffsiteForm implements ContainerInjectionInterface {

  /**
   * Constructs a SwishCheckoutPaymentForm object.
   *
   * @param \Drupal\Component\Uuid\UuidInterface $uuid
   *   The UUID service.
   * @param \GuzzleHttp\Client $httpClient
   *   The HTTP client.
   */
  public function __construct(protected readonly UuidInterface $uuid, protected readonly Client $httpClient) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('uuid'),
      $container->get('http_client')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);

    /** @var \Drupal\commerce_payment\Entity\PaymentInterface $payment */
    $payment = $this->entity;
    $order = $payment->getOrder();

    if (!$order->getTotalPrice()->isPositive()) {
      throw new \Exception('Order ID "' . $order->id() . '": Total price is not a positive amount.');
    }

    $configuration = $payment->getPaymentGateway()
      ->getPlugin()
      ->getConfiguration();

    $form_state->set('remoteId', $this->uuid->generate());
    $form_state->set('payment', $payment);

    $orderId = $order->id();
    $form['orderId'] = [
      '#type' => 'markup',
      '#markup' => '<p>' . $this->t("Order ID: @orderId", ['@orderId' => $orderId]) . '</p>',
    ];

    $amount = $payment->getAmount();

    if ($configuration['show_vat_included']) {
      $form['amount'] = [
        '#type' => 'markup',
        '#markup' => '<p>' . $this->t("Amount: @amount", ['@amount' => $amount]) . ' ' . $this->t('including VAT') . '</p>',
      ];
    }
    else {
      $form['amount'] = [
        '#type' => 'markup',
        '#markup' => '<p>' . $this->t("Amount: @amount", ['@amount' => $amount]) . '</p>',
      ];
    }

    $amount_value = round($amount->getNumber(), 2);
    $message = $configuration['prefix'] . $orderId;

    if ($configuration['qr_code']) {
      // Generate Swish code.
      $url = "https://mpc.getswish.net/qrg-swish/api/v1/prefilled";
      $body = json_encode([
        'format' => 'png',
        'size' => '400',
        'payee' => [
          'value' => $configuration['payee'],
        ],
        'amount' => [
          'value' => $amount_value,
          'editable' => FALSE,
        ],
        'message' => [
          'value' => $message,
          'editable' => FALSE,
        ],
      ]);

      $response = $this->httpClient->post($url, [
        'body' => $body,
        'headers' => [
          'Content-Type' => 'application/json',
        ],
      ]);

      $png = $response->getBody()->getContents();
      $dataUri = 'data:image/png;base64,' . base64_encode($png);

      $form['swish'] = [
        '#type' => 'inline_template',
        '#template' => '<div class="commerce-swish-qr-code"><a href="{{ mobile_uri }}"><img src="{{ data_uri }}" alt="Swish QR Code"/></a></div><div class="commerce-swish-same-device-link"><a href="{{ mobile_uri }}">{{ open_text }}</a></div>',
        '#context' => [
          'data_uri' => $dataUri,
          'mobile_uri' => 'https://app.swish.nu/1/p/sw/?sw=' . $configuration['payee'] . '&amt=' . $amount_value . '&msg=' . $message,
          'open_text' => $this->t('Open Swish on this device'),
        ],
      ];
    }

    $form['information'] = [
      '#type' => 'processed_text',
      '#text' => $configuration['information_value'],
      '#format' => $configuration['information_format'],
    ];

    $submit_status = 'pending';
    if (!empty($configuration['submit_status'])) {
      $submit_status = $configuration['submit_status'];
    }
    $form_state->set('submit_status', $submit_status);

    // No need to call buildRedirectForm as we are embedding an iframe.
    $form['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('I certify that payment has been made.'),
    ];

    // Disable cache.
    $form['#cache']['max-age'] = 0;

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    /** @var \Drupal\commerce_payment\Entity\Payment $payment */
    $payment = $form_state->get('payment');

    $payment->setRemoteId($form_state->get('remoteId'));
    $payment->setState($form_state->get('submit_status'));
    $payment->set('captured', time());
    $payment->save();

    // Redirect to #$form['#return_url'].
    $url = Url::fromUri($form['#return_url']);
    $form_state->setRedirectUrl($url);
  }

}
