<?php

namespace Drupal\Tests\commerce_swish\Unit\Plugin\Commerce\PaymentGateway;

use Drupal\commerce_payment\Entity\PaymentInterface;
use Drupal\commerce_payment\Exception\InvalidRequestException;
use Drupal\commerce_payment\PaymentMethodTypeManager;
use Drupal\commerce_payment\PaymentTypeManager;
use Drupal\commerce_price\MinorUnitsConverter;
use Drupal\commerce_price\Price;
use Drupal\commerce_swish\Plugin\Commerce\PaymentGateway\SwishCheckoutPaymentGateway;
use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\DependencyInjection\ContainerBuilder;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\state_machine\Plugin\Field\FieldType\StateItemInterface;
use Drupal\Tests\UnitTestCase;
use PHPUnit\Framework\MockObject\MockObject;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\TaggedContainerInterface;
use Symfony\Component\HttpFoundation\Request;

/**
 * Tests the SwishCheckoutPaymentGateway class.
 *
 * @coversDefaultClass \Drupal\commerce_swish\Plugin\Commerce\PaymentGateway\SwishCheckoutPaymentGateway
 * @group commerce_swish
 */
class SwishCheckoutPaymentGatewayUnitTest extends UnitTestCase {

  /**
   * Container for this test.
   */
  private TaggedContainerInterface $container;

  /**
   * The entity type manager.
   */
  protected EntityTypeManagerInterface&MockObject $entityTypeManager;

  /**
   * The payment type manager.
   */
  protected PaymentTypeManager&MockObject $paymentTypeManager;

  /**
   * The payment method type manager.
   */
  protected PaymentMethodTypeManager&MockObject $paymentMethodTypeManager;

  /**
   * The time service.
   */
  protected TimeInterface&MockObject $time;

  /**
   * The minor units converter.
   */
  protected MinorUnitsConverter&MockObject $minorUnitsConverter;

  /**
   * The logger.
   */
  protected LoggerInterface&MockObject $logger;

  /**
   * The payment mock.
   */
  protected PaymentInterface&MockObject $payment;

  /**
   * The state mock.
   */
  protected StateItemInterface&MockObject $state;

  /**
   * The gateway instance.
   */
  protected SwishCheckoutPaymentGateway $gateway;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->container = new ContainerBuilder();
    \Drupal::setContainer($this->container);

    $this->entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    $this->paymentTypeManager = $this->createMock(PaymentTypeManager::class);
    $this->paymentMethodTypeManager = $this->createMock(PaymentMethodTypeManager::class);
    $this->time = $this->createMock(TimeInterface::class);
    $this->minorUnitsConverter = $this->createMock(MinorUnitsConverter::class);
    $this->logger = $this->createMock(LoggerInterface::class);

    $this->container->set('entity_type.manager', $this->entityTypeManager);
    $this->container->set('plugin.manager.commerce_payment_type', $this->paymentTypeManager);
    $this->container->set('plugin.manager.commerce_payment_method_type', $this->paymentMethodTypeManager);
    $this->container->set('datetime.time', $this->time);
    $this->container->set('commerce_price.minor_units_converter', $this->minorUnitsConverter);
    $this->container->set('logger.channel.commerce_swish', $this->logger);

    $this->payment = $this->createMock(PaymentInterface::class);
    $this->state = $this->createMock(StateItemInterface::class);
  }

  /**
   * Creates a SwishCheckoutPaymentGateway instance.
   */
  protected function createGateway($configuration = [], $plugin_definition = []): SwishCheckoutPaymentGateway {
    $plugin_definition += [
      'payment_type' => 'swish',
      'payment_method_types' => ['credit_card'],
      'forms' => [],
      'modes' => [
        'qr_code' => 'QR Code mode',
        'sandbox' => 'Swish Sandbox',
        'production' => 'Production Environment',
      ],
      'display_label' => 'Swish',
    ];
    return SwishCheckoutPaymentGateway::create($this->container, $configuration, 'commerce_swish_checkout', $plugin_definition);
  }

  /**
   * Sets the state of the payment mock.
   */
  protected function setStatePayment(string $state_id): void {
    $this->state->method('getId')->willReturn($state_id);
    $this->payment->method('getState')->willReturn($this->state);
  }

  /**
   * Tests the constructor.
   */
  public function testConstructor(): void {
    $gateway = $this->createGateway();
    $this->assertInstanceOf(SwishCheckoutPaymentGateway::class, $gateway);
    $this->assertSame('commerce_swish_checkout', $gateway->getPluginId());
  }

  /**
   * Tests onNotify with logging enabled.
   */
  public function testOnNotifyDebug(): void {
    $gateway = $this->createGateway(['log' => 1]);
    $request = new Request();
    $request->query->set('order_id', '12345');

    $this->logger->expects($this->once())->method('debug')
      ->with('Notify called with order_id: %order_id', ['%order_id' => '12345']);
    $gateway->onNotify($request);
  }

  /**
   * Tests onNotify with logging disabled.
   */
  public function testOnNotifyNoDebug(): void {
    $gateway = $this->createGateway();
    $request = new Request();
    $request->query->set('order_id', '12345');

    $this->logger->expects($this->never())->method('debug');
    $gateway->onNotify($request);
  }

  /**
   * Tests receiving a payment that is not pending.
   */
  public function testReceivePaymentNotPending(): void {
    $gateway = $this->createGateway();
    $this->setStatePayment('new');

    $this->expectException(\InvalidArgumentException::class);
    $gateway->receivePayment($this->payment);
  }

  /**
   * Tests receiving part of the amount on a pending payment.
   */
  public function testReceivePaymentPart(): void {
    $gateway = $this->createGateway();
    $this->setStatePayment('pending');

    $this->payment->expects($this->once())
      ->method('setState')
      ->with('completed');
    $this->payment->expects($this->once())
      ->method('setAmount')
      ->with(new Price('10.00', 'SEK'));
    $gateway->receivePayment($this->payment, new Price('10.00', 'SEK'));
  }

  /**
   * Tests receiving the full amount on a pending payment.
   */
  public function testReceivePaymentFull(): void {
    $gateway = $this->createGateway();
    $this->setStatePayment('pending');
    $this->payment->method('getState')->willReturn($this->state);
    $this->payment->method('getAmount')->willReturn(new Price('100.00', 'SEK'));

    $this->payment->expects($this->once())
      ->method('setState')
      ->with('completed');
    $this->payment->expects($this->once())
      ->method('setAmount')
      ->with(new Price('100.00', 'SEK'));
    $gateway->receivePayment($this->payment);
  }

  /**
   * Tests voiding a payment that is not pending.
   */
  public function testVoidPaymentNotPending(): void {
    $gateway = $this->createGateway();
    $this->setStatePayment('new');

    $this->assertFalse($gateway->canVoidPayment($this->payment));
    $this->expectException(\InvalidArgumentException::class);
    $gateway->voidPayment($this->payment);
  }

  /**
   * Tests voiding a pending payment.
   */
  public function testVoidPayment(): void {
    $gateway = $this->createGateway();
    $this->setStatePayment('pending');

    $this->payment->expects($this->once())->method('setState')->with('voided');
    $gateway->voidPayment($this->payment);
  }

  /**
   * Tests refunding a payment that is not completed.
   */
  public function testReFundPaymentNotCompleted() {
    $gateway = $this->createGateway();
    $this->setStatePayment('new');

    $this->expectException(\InvalidArgumentException::class);
    $gateway->refundPayment($this->payment);
  }

  /**
   * Tests refunding the full amount on a completed payment.
   */
  public function testReFundPaymentFull() {
    $gateway = $this->createGateway();
    $this->setStatePayment('completed');

    $price = new Price('100', 'SEK');

    $this->payment->method('getAmount')->willReturn($price);
    $this->payment->method('getBalance')->willReturn($price);
    $this->payment->method('getRefundedAmount')
      ->willReturn(new Price('0.00', 'SEK'));

    $this->payment->expects($this->once())
      ->method('setState')
      ->with('refunded');
    $this->payment->expects($this->once())
      ->method('setRefundedAmount')
      ->with($price);

    $gateway->refundPayment($this->payment);
  }

  /**
   * Tests refunding more than the balance on a completed payment.
   */
  public function testReFundPaymentMore() {
    $gateway = $this->createGateway();
    $this->setStatePayment('completed');

    $this->payment->method('getBalance')
      ->willReturn(new Price('100.00', 'SEK'));

    $this->expectException(InvalidRequestException::class);
    $gateway->refundPayment($this->payment, new Price('150', 'SEK'));
  }

  /**
   * Tests refunding part of the amount on a completed payment.
   */
  public function testReFundPaymentLess() {
    $gateway = $this->createGateway();
    $this->setStatePayment('completed');

    $this->payment->method('getAmount')->willReturn(new Price('100.00', 'SEK'));
    $this->payment->method('getBalance')->willReturn(new Price('90.00', 'SEK'));
    $this->payment->method('getRefundedAmount')
      ->willReturn(new Price('10.00', 'SEK'));

    $this->payment->expects($this->once())
      ->method('setState')
      ->with('partially_refunded');
    $this->payment->expects($this->once())
      ->method('setRefundedAmount')
      ->with(new Price('40', 'SEK'));

    $gateway->refundPayment($this->payment, new Price('30', 'SEK'));
  }

  /**
   * Tests refunding the rest of the amount on a partially refunded payment.
   */
  public function testReFundPaymentRestOfRefund() {
    $gateway = $this->createGateway();
    $this->setStatePayment('partially_refunded');

    $this->payment->method('getAmount')->willReturn(new Price('100.00', 'SEK'));
    $this->payment->method('getBalance')->willReturn(new Price('70.00', 'SEK'));
    $this->payment->method('getRefundedAmount')
      ->willReturn(new Price('30.00', 'SEK'));

    $this->payment->expects($this->once())
      ->method('setState')
      ->with('refunded');
    $this->payment->expects($this->once())
      ->method('setRefundedAmount')
      ->with(new Price('100', 'SEK'));

    $gateway->refundPayment($this->payment, new Price('70', 'SEK'));
  }

}
