<?php

namespace Drupal\commerce_swish\PluginForm\OffsiteRedirect;

use Drupal\commerce_payment\PluginForm\PaymentOffsiteForm;
use Drupal\commerce_price\Calculator;
use Drupal\Component\Uuid\UuidInterface;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Site\Settings;
use Drupal\Core\State\State;
use Drupal\Core\Url;
use GuzzleHttp\Client;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Swish Checkout.
 */
class SwishCheckoutPaymentForm extends PaymentOffsiteForm implements ContainerInjectionInterface {

  /**
   * Constructs a SwishCheckoutPaymentForm object.
   *
   * @param \Drupal\Component\Uuid\UuidInterface $uuid
   *   The UUID service.
   * @param \GuzzleHttp\Client $httpClient
   *   The HTTP client.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger.
   * @param \Drupal\Core\State\State $state
   *   The state service.
   */
  public function __construct(protected readonly UuidInterface $uuid, protected readonly Client $httpClient, protected readonly LoggerInterface $logger, protected readonly State $state) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('uuid'),
      $container->get('http_client'),
      $container->get('logger.channel.commerce_swish'),
      $container->get('state'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);

    /** @var \Drupal\commerce_payment\Entity\PaymentInterface $payment */
    $payment = $this->entity;
    $order = $payment->getOrder();

    if (!$order->getTotalPrice()->isPositive()) {
      throw new \Exception('Order ID "' . $order->id() . '": Total price is not a positive amount.');
    }

    $configuration = $payment->getPaymentGateway()
      ->getPlugin()
      ->getConfiguration();

    // Remove the dashes and make it uppercase for Swish liking.
    $remoteId = $payment->getRemoteId();
    /** @phpstan-ignore identical.alwaysFalse */
    if ($remoteId === NULL || $remoteId === '') {
      $remoteId = strtoupper(str_replace('-', '', $this->uuid->generate()));
      $payment->setRemoteId($remoteId);
    }
    $form_state->set('payment', $payment);

    $orderId = $order->id();
    $form['orderId'] = [
      '#type' => 'markup',
      '#markup' => '<p>' . $this->t("Order ID: @orderId", ['@orderId' => $orderId]) . '</p>',
    ];

    $amount = $payment->getAmount();

    if ($configuration['show_vat_included']) {
      $form['amount'] = [
        '#type' => 'markup',
        '#markup' => '<p>' . $this->t("Amount: @amount", ['@amount' => $amount]) . ' ' . $this->t('including VAT') . '</p>',
      ];
    }
    else {
      $form['amount'] = [
        '#type' => 'markup',
        '#markup' => '<p>' . $this->t("Amount: @amount", ['@amount' => $amount]) . '</p>',
      ];
    }

    $amount_value = Calculator::round($amount->getNumber(), 2);
    $message = $configuration['prefix'] . $orderId;

    if ($configuration['mode'] === 'qr_code') {
      if ($configuration['qr_code']) {
        // Generate prefilled Swish code.
        $body = json_encode([
          'format' => 'png',
          'size' => '400',
          'payee' => [
            'value' => $configuration['payee'],
          ],
          'amount' => [
            'value' => $amount_value,
            'editable' => FALSE,
          ],
          'message' => [
            'value' => $message,
            'editable' => FALSE,
          ],
        ]);

        $response = $this->httpClient->post("https://mpc.getswish.net/qrg-swish/api/v1/prefilled", [
          'body' => $body,
          'headers' => [
            'Content-Type' => 'application/json',
          ],
        ]);

        $png = $response->getBody()->getContents();
        $dataUri = 'data:image/png;base64,' . base64_encode($png);

        $form['swish'] = [
          '#type' => 'inline_template',
          '#template' => '<div class="commerce-swish-qr-code"><a href="{{ mobile_uri }}" class="js-hide-warning"><img src="{{ data_uri }}" alt="Swish QR Code"/></a></div><div class="commerce-swish-same-device-link"><a href="{{ mobile_uri }}" class="js-hide-warning">{{ open_text }}</a></div>',
          '#context' => [
            'data_uri' => $dataUri,
            'mobile_uri' => 'https://app.swish.nu/1/p/sw/?sw=' . $configuration['payee'] . '&amt=' . $amount_value . '&msg=' . $message,
            'open_text' => $this->t('Open Swish on this device'),
          ],
        ];
      }
      $form['#attached']['library'][] = 'commerce_swish/warning_leaving_qr_code';

      // No need to call buildRedirectForm as we are embedding an iframe.
      $form['submit'] = [
        '#type' => 'submit',
        '#value' => $this->t('I certify that payment has been made.'),
        '#attributes' => [
          'class' => ['js-hide-warning'],
        ],

      ];

      $submit_status = 'pending';
      if (isset($configuration['submit_status']) || $configuration['submit_status'] !== '') {
        $submit_status = $configuration['submit_status'];
      }
      $form_state->set('submit_status', $submit_status);
    }
    if ($configuration['mode'] === 'mss' || $configuration['mode'] === 'production') {
      /** @var \Drupal\commerce_payment\Plugin\Commerce\PaymentGateway\OffsitePaymentGatewayInterface $plugin */
      $plugin = $payment->getPaymentGateway()->getPlugin();
      // Saving the payment as we start handing over control to Swish.
      // Reloading the page will start another payment.
      $payment->setRemoteState('CREATED');
      $payment->save();
      if ($configuration['log']) {
        $this->logger->debug('Payment created: %payment_id in order: %order_id', [
          '%payment_id' => $payment->id(),
          '%order_id' => $order->id(),
        ]);
      }

      $body = json_encode([
        "payeePaymentReference" => $orderId,
        "callbackUrl" => $plugin->getNotifyUrl()->toString() . '?order_id=' . $orderId,
        "payeeAlias" => $configuration['payee'],
        "amount" => $amount_value,
        "currency" => "SEK",
        "message" => $message,
        "callbackIdentifier" => self::generateHash($remoteId),
      ]);
      $url_payment_requests = 'https://mss.cpc.getswish.net/swish-cpcapi/api/v2/paymentrequests/';
      $url_commerce = 'https://mpc.getswish.net/qrg-swish/api/v1/commerce';
      $cert = $configuration['mss_cert_path'];
      $key = $configuration['mss_key_path'];
      $password = $this->state->get('swish:' . $configuration['password_identifier'] . ':mss');
      $verify = $configuration['mss_cacert_path'];
      if ($configuration['mode'] === 'production') {
        $url_payment_requests = 'https://cpc.getswish.net/swish-cpcapi/api/v2/paymentrequests/';
        $cert = $configuration['cpc_cert_path'];
        $key = $configuration['cpc_key_path'];
        $password = $this->state->get('swish:' . $configuration['password_identifier'] . ':cpc');
        $verify = $configuration['cpc_cacert_path'];
      }

      try {
        $response = $this->httpClient->put($url_payment_requests . $remoteId, [
          'body' => $body,
          'headers' => [
            'Content-Type' => 'application/json',
          ],
          'cert' => $cert,
          'ssl_key' => [$key, $password],
          'verify' => $verify,
        ]);
        $token = $response->getHeader('PaymentRequestToken');
        $token = reset($token);
        if ($configuration['log']) {
          $this->logger->debug('Payment created at Swish with token: %token', [
            '%token' => $token,
          ]);
        }
      }
      catch (\Exception $e) {
        $this->logger->error('Swish error: ' . $e->getMessage());
        return $form;
      }

      // Generate Swish code.
      $body = json_encode([
        'format' => 'png',
        'size' => '400',
        'token' => $token,
      ]);

      $response = $this->httpClient->post($url_commerce, [
        'body' => $body,
        'headers' => [
          'Content-Type' => 'application/json',
        ],
      ]);

      $png = $response->getBody()->getContents();
      $dataUri = 'data:image/png;base64,' . base64_encode($png);

      $form['swish'] = [
        '#type' => 'inline_template',
        '#template' => '<div class="commerce-swish-qr-code"><a href="{{ mobile_uri }}" class="js-hide-warning"><img src="{{ data_uri }}" alt="Swish QR Code"/></a></div><div class="commerce-swish-same-device-link"><a href="{{ mobile_uri }}" class="js-hide-warning">{{ open_text }}</a></div>',
        '#context' => [
          'data_uri' => $dataUri,
          'mobile_uri' => 'swish://paymentrequest?token=' . $token . '&callbackurl=' . $form['#return_url'],
          'open_text' => $this->t('Open Swish on this device'),
        ],
      ];
      // Div tag with name swish-payment-status to be replaced by javascript.
      $form['payment_status'] = [
        '#type' => 'markup',
        '#markup' => '<div id="js-swish-payment-status">' . $this->t('Payment created at Swish') . '</div>',
      ];
      $form['#attached'] = [
        'library' => ['commerce_swish/update_status'],
        'drupalSettings' => [
          'commerce_swish' => [
            'orderId' => $orderId,
            'remoteId' => $remoteId,
          ],
        ],
      ];
    }

    $form['information'] = [
      '#type' => 'processed_text',
      '#text' => $configuration['information_value'],
      '#format' => $configuration['information_format'],
    ];

    // Disable cache.
    $form['#cache']['max-age'] = 0;

    return $form;
  }

  /**
   * Generates a hash for the callback url.
   *
   * @param string $remoteId
   *   The remote id of the payment.
   */
  public static function generateHash($remoteId) {
    $hash = hash('sha256', $remoteId . Settings::get('hash_salt', ''));
    return substr($hash, 0, 36);
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    /** @var \Drupal\commerce_payment\Entity\Payment $payment */
    $payment = $form_state->get('payment');
    $payment->setState($form_state->get('submit_status'));
    $payment->set('captured', time());
    $payment->save();

    // Redirect to #$form['#return_url'].
    $url = Url::fromUri($form['#return_url']);
    $form_state->setRedirectUrl($url);
  }

}
