<?php

namespace Drupal\commerce_swish\Plugin\Commerce\PaymentGateway;

use Drupal\commerce_payment\Attribute\CommercePaymentGateway;
use Drupal\commerce_payment\Entity\PaymentInterface;
use Drupal\commerce_payment\Plugin\Commerce\PaymentGateway\OffsitePaymentGatewayBase;
use Drupal\commerce_payment\Plugin\Commerce\PaymentGateway\SupportsRefundsInterface;
use Drupal\commerce_payment\Plugin\Commerce\PaymentGateway\SupportsVoidsInterface;
use Drupal\commerce_payment\PluginForm\PaymentReceiveForm;
use Drupal\commerce_price\Price;
use Drupal\commerce_swish\PluginForm\OffsiteRedirect\SwishCheckoutPaymentForm;
use Drupal\Component\Uuid\UuidInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\State\StateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Exception\BadRequestHttpException;

/**
 * Provides the Swish checkout payment gateway.
 */
#[CommercePaymentGateway(
  id: "commerce_swish_checkout",
  label: new TranslatableMarkup("Swish"),
  display_label: new TranslatableMarkup("Swish"),
  modes: [
    "qr_code" => new TranslatableMarkup("QR code mode"),
    "sandbox" => new TranslatableMarkup("Swish Sandbox"),
    "mss" => new TranslatableMarkup("Merchant Swish Simulator"),
    "production" => new TranslatableMarkup("Production Environment"),
  ],
  forms: [
    "offsite-payment" => SwishCheckoutPaymentForm::class,
    "receive-payment" => PaymentReceiveForm::class,
  ],
  payment_type: "payment_manual",
  requires_billing_information: FALSE,
)]
class SwishCheckoutPaymentGateway extends OffsitePaymentGatewayBase implements SupportsVoidsInterface, SupportsRefundsInterface {

  /**
   * The logger.
   */
  protected LoggerInterface $logger;

  /**
   * Uuid.
   */
  protected UuidInterface $uuid;

  /**
   * State.
   */
  protected StateInterface $state;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->logger = $container->get('logger.channel.commerce_swish');
    $instance->uuid = $container->get('uuid');
    $instance->state = $container->get('state');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'information_value' => '',
      'information_format' => 'plain_text',
      'log' => 0,
      'qr_code' => 1,
      'payee' => '',
      'show_vat_included' => 1,
      'prefix' => '',
      'submit_status' => 'pending',
      'cpc_cert_path' => '',
      'cpc_key_path' => '',
      'password_identifier' => '',
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);

    if (!isset($this->configuration['password_identifier']) || $this->configuration['password_identifier'] === '') {
      // Generate UUID for Prefixing password in State.
      $this->configuration['password_identifier'] = $this->uuid->generate();
    }

    $form['password_identifier'] = [
      '#type' => 'value',
      '#value' => $this->configuration['password_identifier'],
    ];

    $form['mode']['#weight'] = 1;

    // Production and sandbox modes are not yet implemented.
    $form['not_implemented'] = [
      '#type' => 'container',
      '#markup' => $this->t('<p><strong>Note:</strong> Production and Sandbox modes are not yet implemented.</p>'),
      '#states' => [
        'visible' => [
          ':input[name="configuration[commerce_swish_checkout][mode]"]' => [
            ['value' => 'sandbox'],
            'or',
            ['value' => 'production'],
          ],
        ],
      ],
      '#weight' => 2,
    ];
    $form['mode_description'] = [
      '#markup' => '
<strong>QR Code mode</strong><br>Generates a QR code for the payment. No automatic feedback if the customer has paid. If the customer clicks the submit button under QR code payment is marked as paid. Only require a phone number to receive Swish payments, no extra cost.<br>
<strong>Swish Sandbox</strong><br>Used for development where the full flow of QR code generation and payment handling is simulated, using developer version of both Swish and BankID. Use separate mobile phone. Need Sandbox access at Swish<br>
<strong>Merchant Swish Simulator</strong><br>Used for testing payment flow on Drupal site. Developer certificates are required. A few seconds after QR code is generated, Swish will ping callback to mark payment as paid. No extra details are required, useful to test callback functionality and firewalls. Useful in development and staging sites.<br>
<strong>Production Environment</strong><br>Production mode of Swish företag. Needs certificate from Swish.<br><br>

Right now we only use the M-Commerce payment flow. <a href="https://developer.swish.nu/documentation/environments" target="_blank">More information about environments here</a>
',
      '#weight' => 10,
    ];

    $form['payee'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Swish number'),
      '#default_value' => $this->configuration['payee'],
      '#description' => $this->t('The Swish number (e.g. 1231234567) that will receive the payment.'),
      '#weight' => 20,
    ];

    $form['cpc_cert_path'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Production certificate path'),
      '#default_value' => $this->configuration['cpc_cert_path'],
      '#description' => $this->t('Path to the Production certificate. In PEM format.'),
      '#weight' => 20,
      '#states' => [
        'visible' => [
          ':input[name="configuration[commerce_swish_checkout][mode]"]' => ['value' => 'production'],
        ],
      ],
    ];
    $form['cpc_key_path'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Production key path'),
      '#default_value' => $this->configuration['cpc_key_path'],
      '#description' => $this->t('Path to the Production key. In PEM format.'),
      '#weight' => 20,
      '#states' => [
        'visible' => [
          ':input[name="configuration[commerce_swish_checkout][mode]"]' => ['value' => 'production'],
        ],
      ],
    ];
    $password_set = $this->state->get('swish:' . $this->configuration['password_identifier'] . ':cpc', '') !== '' ?
      $this->t('Password set, but not shown.') : $this->t('Password missing.');

    $form['cpc_cert_password'] = [
      '#type' => 'password',
      '#title' => $this->t('Production certificate password'),
      '#description' => $this->t('Password for the Production certificate.') . ' ' . $password_set,
      '#weight' => 20,
      '#states' => [
        'visible' => [
          ':input[name="configuration[commerce_swish_checkout][mode]"]' => ['value' => 'production'],
        ],
      ],
    ];

    $form['qr_code'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Display QR code on the payment page'),
      '#default_value' => $this->configuration['qr_code'],
      '#options' => [
        1 => $this->t('Yes'),
        0 => $this->t('No'),
      ],
      '#weight' => 21,
      '#states' => [
        'visible' => [
          ':input[name="configuration[commerce_swish_checkout][mode]"]' => ['value' => 'qr_code'],
        ],
      ],
    ];
    $form['show_vat_included'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show VAT included after the amount'),
      '#default_value' => $this->configuration['show_vat_included'],
      '#options' => [
        1 => $this->t('Yes'),
        0 => $this->t('No'),
      ],
      '#weight' => 22,
    ];
    $form['prefix'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Prefix'),
      '#default_value' => $this->configuration['prefix'],
      '#description' => $this->t('Prefix for the payment message, e.g. your store name.'),
      '#weight' => 23,
    ];

    $form['information'] = [
      '#type' => 'text_format',
      '#title' => $this->t('Information'),
      '#default_value' => $this->configuration['information_value'],
      '#format' => $this->configuration['information_format'],
      '#description' => $this->t('The information that will be displayed on the payment page.'),
      '#weight' => 24,
    ];

    $form['submit_status'] = [
      '#type' => 'select',
      '#title' => $this->t('After user has confirmed that the payment is done. What status should the payment have?'),
      '#default_value' => $this->configuration['submit_status'],
      '#options' => [
        'pending' => $this->t('Pending'),
        'completed' => $this->t('Completed'),
      ],
      '#states' => [
        'visible' => [
          ':input[name="configuration[commerce_swish_checkout][mode]"]' => ['value' => 'qr_code'],
        ],
      ],
      '#required' => TRUE,
      '#weight' => 30,

    ];

    $form['log'] = [
      '#type' => 'select',
      '#title' => $this->t('Extended logging'),
      '#default_value' => $this->configuration['log'],
      '#options' => [
        '0' => $this->t('No'),
        '1' => $this->t('Yes'),
      ],
      '#required' => TRUE,
      '#weight' => 40,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state): void {
    $values = $form_state->getValue($form['#parents']);
    $file_fields = [
      'cpc_cert_path',
      'cpc_key_path',
    ];
    foreach ($file_fields as $field) {
      if (isset($values[$field]) && $values[$field] !== '') {
        // Check if a file exists.
        if (!file_exists($values[$field])) {
          $form_state->setError($form[$field], $this->t('The file %file does not exist in field %field.', [
            '%file' => $values[$field],
            '%field' => $field,
          ]));
        }
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    parent::submitConfigurationForm($form, $form_state);
    $values = $form_state->getValue($form['#parents']);
    $this->configuration['payee'] = $values['payee'];
    $this->configuration['qr_code'] = $values['qr_code'];
    $this->configuration['information_value'] = $values['information']['value'];
    $this->configuration['information_format'] = $values['information']['format'];
    $this->configuration['log'] = $values['log'];
    $this->configuration['show_vat_included'] = $values['show_vat_included'];
    $this->configuration['prefix'] = $values['prefix'];
    $this->configuration['submit_status'] = $values['submit_status'];
    $this->configuration['cpc_cert_path'] = $values['cpc_cert_path'];
    $this->configuration['cpc_key_path'] = $values['cpc_key_path'];
    $this->configuration['password_identifier'] = $values['password_identifier'];

    // Save password to State as it should not be exported to config.
    if (isset($values['cpc_cert_password']) && $values['cpc_cert_password'] !== '') {
      $this->state->set('swish:' . $this->configuration['password_identifier'] . ':cpc', $values['cpc_cert_password']);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function onNotify(Request $request): Response|null {
    $order_id = $request->query->get('order_id');
    $body = $request->getContent();

    if ($this->configuration['log']) {
      $this->logger->debug('Notify called with order_id: %order_id, headers: %headers, body: %body', [
        '%order_id' => $order_id,
        '%headers' => print_r($request->headers->all(), TRUE),
        '%body' => $body,
      ]);
    }
    $swish_content = json_decode($body, TRUE);
    $remoteId = $swish_content['id'] ?? NULL;
    if ($remoteId === NULL || $remoteId === '') {
      $this->logger->error('No remoteId found in Swish response.');
      throw new BadRequestHttpException('No payment found with remoteId.');
    }
    $callbackidentifier = $request->headers->get('callbackidentifier', '');
    if (!isset($callbackidentifier) || $callbackidentifier === '') {
      $this->logger->error('No callbackidentifier found in Swish response.');
      throw new BadRequestHttpException('No callbackidentifier found in Swish response.');
    }

    if ($callbackidentifier !== SwishCheckoutPaymentForm::generateHash($remoteId)) {
      if ($this->configuration['log']) {
        $this->logger->debug('|%callbackidentifier| |%remoteId| |%hash|', [
          '%callbackidentifier' => print_r($callbackidentifier, TRUE),
          '%remoteId' => $remoteId,
          '%hash' => SwishCheckoutPaymentForm::generateHash($remoteId),
        ]);
      }
      $this->logger->error('Invalid callbackidentifier found in Swish response.');
      throw new BadRequestHttpException('Invalid callbackidentifier found in Swish response.');
    }

    $order = $this->entityTypeManager->getStorage('commerce_order')->load($order_id);
    if ($order === NULL) {
      $this->logger->error('Order not found for order_id: %order_id', ['%order_id' => $order_id]);
      throw new BadRequestHttpException('Order not found.');
    }
    /** @var \Drupal\commerce_payment\PaymentStorageInterface $payment_storage */
    $payment_storage = $this->entityTypeManager->getStorage('commerce_payment');
    $payment = $payment_storage->loadByRemoteId($remoteId);
    if ($payment === NULL) {
      $this->logger->warning('No payment found with remoteId: %remote_id', ['%remote_id' => $remoteId]);
      throw new BadRequestHttpException('No payment found with remoteId.');
    }

    $status = $swish_content['status'] ?? '';
    $payment->setRemoteState($status);
    switch ($status) {
      case 'PAID':
        if ($this->configuration['log']) {
          $this->logger->debug('Payment completed for payment: %payment_id in order: %order_id', [
            '%payment_id' => $payment->id(),
            '%order_id' => $order->id(),
          ]);
        }
        $payment->setAmount(new Price($swish_content['amount'], $swish_content['currency']));
        $payment->setState('completed');
        $payment->save();
        break;

      case 'DECLINED':
        $this->logger->warning('Payment declined for payment: %payment_id in order: %order_id', [
          '%payment_id' => $payment->id(),
          '%order_id' => $order->id(),
        ]);
        $payment->delete();
        break;

      case 'CANCELLED':
        if ($this->configuration['log']) {
          $this->logger->debug('Payment cancelled for payment: %payment_id in order: %order_id', [
            '%payment_id' => $payment->id(),
            '%order_id' => $order->id(),
          ]);
        }
        $payment->setState('canceled');
        $payment->save();
        break;

      case 'ERROR':
        $this->logger->error('Payment error for payment: %payment_id, Error: %error', [
          '%payment_id' => $payment->id(),
          '%error' => print_r($swish_content, TRUE),
        ]);
        $payment->delete();
    }
    return NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function buildPaymentOperations(PaymentInterface $payment): array {
    $operations = [];
    $operations['receive'] = [
      'title' => $this->t('Receive'),
      'page_title' => $this->t('Receive payment'),
      'plugin_form' => 'receive-payment',
      'access' => $payment->getState()->getId() === 'pending',
    ];
    $operations['void'] = [
      'title' => $this->t('Void'),
      'page_title' => $this->t('Void payment'),
      'plugin_form' => 'void-payment',
      'access' => $this->canVoidPayment($payment),
    ];
    $operations['refund'] = [
      'title' => $this->t('Refund'),
      'page_title' => $this->t('Refund payment'),
      'plugin_form' => 'refund-payment',
      'access' => $this->canRefundPayment($payment),
    ];

    return $operations;
  }

  /**
   * {@inheritdoc}
   */
  public function receivePayment(PaymentInterface $payment, ?Price $amount = NULL): void {
    $this->assertPaymentState($payment, ['pending']);

    // If not specified, use the entire amount.
    $amount = ($amount !== NULL) ? $amount : $payment->getAmount();
    $payment->setState('completed');
    $payment->setAmount($amount);
    $payment->save();
  }

  /**
   * {@inheritdoc}
   */
  public function canVoidPayment(PaymentInterface $payment): bool {
    return $payment->getState()->getId() === 'pending';
  }

  /**
   * {@inheritdoc}
   */
  public function voidPayment(PaymentInterface $payment): void {
    $this->assertPaymentState($payment, ['pending']);
    $payment->setState('voided');
    $payment->save();
  }

  /**
   * {@inheritdoc}
   */
  public function refundPayment(PaymentInterface $payment, ?Price $amount = NULL): void {
    // @todo Add support for refund using Swish API.
    $this->assertPaymentState($payment, ['completed', 'partially_refunded']);
    // If not specified, refund the entire amount.
    $amount = ($amount !== NULL) ? $amount : $payment->getAmount();
    $this->assertRefundAmount($payment, $amount);

    $old_refunded_amount = $payment->getRefundedAmount();
    $new_refunded_amount = $old_refunded_amount->add($amount);
    if ($new_refunded_amount->lessThan($payment->getAmount())) {
      $payment->setState('partially_refunded');
    }
    else {
      $payment->setState('refunded');
    }

    $payment->setRefundedAmount($new_refunded_amount);
    $payment->save();
  }

}
