<?php

declare(strict_types=1);

namespace Drupal\Tests\commerce_swish\Unit\Plugin\Commerce\PaymentGateway;

use Drupal\commerce_order\Entity\OrderInterface;
use Drupal\commerce_payment\Entity\PaymentInterface;
use Drupal\commerce_payment\Exception\InvalidRequestException;
use Drupal\commerce_payment\PaymentMethodTypeManager;
use Drupal\commerce_payment\PaymentStorageInterface;
use Drupal\commerce_payment\PaymentTypeManager;
use Drupal\commerce_price\MinorUnitsConverter;
use Drupal\commerce_price\Price;
use Drupal\commerce_swish\Plugin\Commerce\PaymentGateway\SwishCheckoutPaymentGateway;
use Drupal\commerce_swish\PluginForm\OffsiteRedirect\SwishCheckoutPaymentForm;
use Drupal\Component\Datetime\TimeInterface;
use Drupal\Component\Uuid\UuidInterface;
use Drupal\Core\State\StateInterface;
use Drupal\Core\DependencyInjection\ContainerBuilder;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\EntityTypeRepositoryInterface;
use Drupal\state_machine\Plugin\Field\FieldType\StateItemInterface;
use Drupal\Tests\UnitTestCase;
use PHPUnit\Framework\MockObject\MockObject;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\TaggedContainerInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpKernel\Exception\BadRequestHttpException;

/**
 * Tests the SwishCheckoutPaymentGateway class.
 *
 * @coversDefaultClass \Drupal\commerce_swish\Plugin\Commerce\PaymentGateway\SwishCheckoutPaymentGateway
 * @group commerce_swish
 */
final class SwishCheckoutPaymentGatewayUnitTest extends UnitTestCase {

  /**
   * Container for this test.
   */
  private TaggedContainerInterface $container;

  /**
   * The logger.
   */
  protected LoggerInterface&MockObject $logger;

  /**
   * The payment mock.
   */
  protected PaymentInterface&MockObject $payment;

  /**
   * The state mock.
   */
  protected StateItemInterface&MockObject $stateItem;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->container = new ContainerBuilder();
    \Drupal::setContainer($this->container);

    $entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    $entityTypeRepository = $this->createMock(EntityTypeRepositoryInterface::class);
    $paymentTypeManager = $this->createMock(PaymentTypeManager::class);
    $paymentMethodTypeManager = $this->createMock(PaymentMethodTypeManager::class);
    $time = $this->createMock(TimeInterface::class);
    $minorUnitsConverter = $this->createMock(MinorUnitsConverter::class);
    $this->logger = $this->createMock(LoggerInterface::class);
    $uuid_generator = $this->createMock(UuidInterface::class);
    $stateApi = $this->createMock(StateInterface::class);

    $order = $this->createMock(OrderInterface::class);
    $commerce_order_storage = $this->createMock(EntityStorageInterface::class);
    $load_map = [
      ['12345', $order],
      ['123456', NULL],
    ];
    $commerce_order_storage->method('load')->willReturnMap($load_map);

    $this->payment = $this->createMock(PaymentInterface::class);
    $payment_storage = $this->createMock(PaymentStorageInterface::class);
    $payment_storage->method('load')->with('12345')->willReturn($this->payment);
    $payment_storage->method('loadByRemoteId')->with('123')->willReturn($this->payment);

    $storage_map = [
      ['commerce_order', $commerce_order_storage],
      ['commerce_payment', $payment_storage],
    ];

    $entityTypeManager->method('getStorage')->willReturnMap($storage_map);

    $uuid_generator->method('generate')->willReturn('uuid');

    $this->container->set('entity_type.manager', $entityTypeManager);
    $this->container->set('entity_type.repository', $entityTypeRepository);
    $this->container->set('plugin.manager.commerce_payment_type', $paymentTypeManager);
    $this->container->set('plugin.manager.commerce_payment_method_type', $paymentMethodTypeManager);
    $this->container->set('datetime.time', $time);
    $this->container->set('commerce_price.minor_units_converter', $minorUnitsConverter);
    $this->container->set('logger.channel.commerce_swish', $this->logger);
    $this->container->set('uuid', $uuid_generator);
    $this->container->set('state', $stateApi);

    $this->stateItem = $this->createMock(StateItemInterface::class);
  }

  /**
   * Creates a SwishCheckoutPaymentGateway instance.
   */
  protected function createGateway($configuration = [], $plugin_definition = []): SwishCheckoutPaymentGateway {
    $plugin_definition += [
      'payment_type' => 'swish',
      'payment_method_types' => ['credit_card'],
      'forms' => [],
      'modes' => [
        'qr_code' => 'QR Code mode',
        'sandbox' => 'Swish Sandbox',
        'production' => 'Production Environment',
      ],
      'display_label' => 'Swish',
    ];
    return SwishCheckoutPaymentGateway::create($this->container, $configuration, 'commerce_swish_checkout', $plugin_definition);
  }

  /**
   * Sets the state of the payment mock.
   */
  protected function setStatePayment(string $state_id): void {
    $this->stateItem->method('getId')->willReturn($state_id);
    $this->payment->method('getState')->willReturn($this->stateItem);
  }

  /**
   * Tests the constructor.
   */
  public function testConstructor(): void {
    $gateway = $this->createGateway();
    $this->assertSame('commerce_swish_checkout', $gateway->getPluginId());
  }

  /**
   * Tests onNotify with invalid hash.
   */
  public function testOnNotifyInvalidHash(): void {
    $gateway = $this->createGateway();
    $remoteId = '123';
    $server = [
      'HTTP_callbackidentifier' => SwishCheckoutPaymentForm::generateHash($remoteId) . 'e',
    ];
    $content = json_encode(['id' => $remoteId]);
    $request = new Request(server: $server, content: $content);
    $request->query->set('order_id', '12345');
    $this->logger->expects($this->never())->method('debug');
    $this->expectException(BadRequestHttpException::class);
    $this->expectExceptionMessage('Invalid callbackidentifier found in Swish response.');
    $gateway->onNotify($request);
  }

  /**
   * Tests onNotify with invalid order.
   */
  public function testOnNotifyInvalidOrder(): void {
    $gateway = $this->createGateway();
    $remoteId = '123';
    $server = [
      'HTTP_callbackidentifier' => SwishCheckoutPaymentForm::generateHash($remoteId),
    ];
    $content = json_encode(['id' => $remoteId]);
    $request = new Request(server: $server, content: $content);
    $request->query->set('order_id', '123456');
    $this->logger->expects($this->never())->method('debug');
    $this->expectException(BadRequestHttpException::class);
    $this->expectExceptionMessage('Order not found.');
    $gateway->onNotify($request);
  }

  /**
   * Tests onNotify for Paid payment.
   */
  public function testOnNotifyPaid(): void {
    $gateway = $this->createGateway();
    $remoteId = '123';
    $server = [
      'HTTP_callbackidentifier' => SwishCheckoutPaymentForm::generateHash($remoteId),
    ];
    $content = json_encode(['id' => $remoteId, 'status' => 'PAID', 'amount' => '1337', 'currency' => 'SEK']);
    $request = new Request(server: $server, content: $content);
    $request->query->set('order_id', '12345');
    $this->logger->expects($this->never())->method('debug');
    $this->payment->expects($this->once())
      ->method('setRemoteState')
      ->with('PAID');
    $this->payment->expects($this->once())
      ->method('setState')
      ->with('completed');
    $gateway->onNotify($request);

  }

  /**
   * Tests receiving a payment that is not pending.
   */
  public function testReceivePaymentNotPending(): void {
    $gateway = $this->createGateway();
    $this->setStatePayment('new');

    $this->expectException(\InvalidArgumentException::class);
    $gateway->receivePayment($this->payment);
  }

  /**
   * Tests receiving part of the amount on a pending payment.
   */
  public function testReceivePaymentPart(): void {
    $gateway = $this->createGateway();
    $this->setStatePayment('pending');

    $this->payment->expects($this->once())
      ->method('setState')
      ->with('completed');
    $this->payment->expects($this->once())
      ->method('setAmount')
      ->with(new Price('10.00', 'SEK'));
    $gateway->receivePayment($this->payment, new Price('10.00', 'SEK'));
  }

  /**
   * Tests receiving the full amount on a pending payment.
   */
  public function testReceivePaymentFull(): void {
    $gateway = $this->createGateway();
    $this->setStatePayment('pending');
    $this->payment->method('getState')->willReturn($this->stateItem);
    $this->payment->method('getAmount')->willReturn(new Price('100.00', 'SEK'));

    $this->payment->expects($this->once())
      ->method('setState')
      ->with('completed');
    $this->payment->expects($this->once())
      ->method('setAmount')
      ->with(new Price('100.00', 'SEK'));
    $gateway->receivePayment($this->payment);
  }

  /**
   * Tests voiding a payment that is not pending.
   */
  public function testVoidPaymentNotPending(): void {
    $gateway = $this->createGateway();
    $this->setStatePayment('new');

    $this->assertFalse($gateway->canVoidPayment($this->payment));
    $this->expectException(\InvalidArgumentException::class);
    $gateway->voidPayment($this->payment);
  }

  /**
   * Tests voiding a pending payment.
   */
  public function testVoidPayment(): void {
    $gateway = $this->createGateway();
    $this->setStatePayment('pending');

    $this->payment->expects($this->once())->method('setState')->with('voided');
    $gateway->voidPayment($this->payment);
  }

  /**
   * Tests refunding a payment that is not completed.
   */
  public function testReFundPaymentNotCompleted() {
    $gateway = $this->createGateway();
    $this->setStatePayment('new');

    $this->expectException(\InvalidArgumentException::class);
    $gateway->refundPayment($this->payment);
  }

  /**
   * Tests refunding the full amount on a completed payment.
   */
  public function testReFundPaymentFull() {
    $gateway = $this->createGateway();
    $this->setStatePayment('completed');

    $price = new Price('100', 'SEK');

    $this->payment->method('getAmount')->willReturn($price);
    $this->payment->method('getBalance')->willReturn($price);
    $this->payment->method('getRefundedAmount')
      ->willReturn(new Price('0.00', 'SEK'));

    $this->payment->expects($this->once())
      ->method('setState')
      ->with('refunded');
    $this->payment->expects($this->once())
      ->method('setRefundedAmount')
      ->with($price);

    $gateway->refundPayment($this->payment);
  }

  /**
   * Tests refunding more than the balance on a completed payment.
   */
  public function testReFundPaymentMore() {
    $gateway = $this->createGateway();
    $this->setStatePayment('completed');

    $this->payment->method('getBalance')
      ->willReturn(new Price('100.00', 'SEK'));

    $this->expectException(InvalidRequestException::class);
    $gateway->refundPayment($this->payment, new Price('150', 'SEK'));
  }

  /**
   * Tests refunding part of the amount on a completed payment.
   */
  public function testReFundPaymentLess() {
    $gateway = $this->createGateway();
    $this->setStatePayment('completed');

    $this->payment->method('getAmount')->willReturn(new Price('100.00', 'SEK'));
    $this->payment->method('getBalance')->willReturn(new Price('90.00', 'SEK'));
    $this->payment->method('getRefundedAmount')
      ->willReturn(new Price('10.00', 'SEK'));

    $this->payment->expects($this->once())
      ->method('setState')
      ->with('partially_refunded');
    $this->payment->expects($this->once())
      ->method('setRefundedAmount')
      ->with(new Price('40', 'SEK'));

    $gateway->refundPayment($this->payment, new Price('30', 'SEK'));
  }

  /**
   * Tests refunding the rest of the amount on a partially refunded payment.
   */
  public function testReFundPaymentRestOfRefund() {
    $gateway = $this->createGateway();
    $this->setStatePayment('partially_refunded');

    $this->payment->method('getAmount')->willReturn(new Price('100.00', 'SEK'));
    $this->payment->method('getBalance')->willReturn(new Price('70.00', 'SEK'));
    $this->payment->method('getRefundedAmount')
      ->willReturn(new Price('30.00', 'SEK'));

    $this->payment->expects($this->once())
      ->method('setState')
      ->with('refunded');
    $this->payment->expects($this->once())
      ->method('setRefundedAmount')
      ->with(new Price('100', 'SEK'));

    $gateway->refundPayment($this->payment, new Price('70', 'SEK'));
  }

}
