<?php

namespace Drupal\commerce_unzer\Plugin\Commerce\PaymentGateway;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\commerce_order\Entity\OrderInterface;
use Drupal\commerce_payment\Entity\PaymentInterface;
use Drupal\commerce_payment\Exception\InvalidRequestException;
use Drupal\commerce_payment\Plugin\Commerce\PaymentGateway\HasPaymentInstructionsInterface;
use Drupal\commerce_payment\Plugin\Commerce\PaymentGateway\OffsitePaymentGatewayBase;
use Drupal\commerce_unzer\DebugHandler;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use UnzerSDK\Constants\PaymentState;
use UnzerSDK\Exceptions\UnzerApiException;
use UnzerSDK\Resources\TransactionTypes\AbstractTransactionType;
use UnzerSDK\Unzer;

/**
 * Provides the Unzer payment gateway (Off Site).
 *
 * @CommercePaymentGateway(
 *     id="commerce_unzer_offsite",
 *     label="Unzer (Off Site)",
 *     display_label="Pay with credit or debit card",
 *     forms={
 *         "offsite-payment": "Drupal\commerce_unzer\PluginForm\OffsiteRedirect\UnzerForm",
 *     },
 *     payment_method_types={"credit_card"},
 *     credit_card_types={
 *         "amex",
 *         "dinersclub",
 *         "discover",
 *         "jcb",
 *         "mastercard",
 *         "visa",
 *     },
 *     requires_billing_information=FALSE,
 * )
 */
class UnzerOffSite extends OffsitePaymentGatewayBase implements HasPaymentInstructionsInterface {

  /**
   * The logger.
   */
  protected LoggerChannelInterface $logger;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->logger = $container->get('logger.factory')->get('commerce_unzer');
    return $instance;
  }

  /**
   * {@inheritdoc}
   *
   * @param array<string,mixed> $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array<string,mixed>
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);

    $form['private_key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Private Key'),
      '#default_value' => $this->configuration['private_key'],
      '#size' => 60,
    ];
    $form['public_key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Public Key'),
      '#default_value' => $this->configuration['public_key'],
      '#size' => 60,
    ];
    $form['instructions'] = [
      '#type' => 'text_format',
      '#title' => $this->t('Payment instructions'),
      '#description' => $this->t('Shown at the end of checkout, after the customer has placed their order.'),
      '#default_value' => $this->configuration['instructions']['value'],
      '#format' => $this->configuration['instructions']['format'],
    ];
    $form['debug_logging'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Debug Logging'),
      '#description' => $this->t('Whether to write debugging output to the Drupal logger channel.'),
      '#default_value' => $this->configuration['debug_logging'],
    ];

    $form['ept_details'] = [
      '#type' => 'details',
      '#title' => $this->t('Excluded payment types'),
      '#description' => $this->t('Unzer payment types that should <em>not</em> be available when using this payment gateway even if your Unzer account configuration allowed them.'),
      '#open' => FALSE,
    ];
    $form['ept_details']['excluded_payment_types'] = [
      '#type' => 'checkboxes',
      '#default_value' => $this->configuration['excluded_payment_types'],
      '#options' => array_map(static fn($data) => $data['label'], self::getUnzerPaymentTypes()),
      '#multiple' => TRUE,
    ];
    $form['paypage_settings'] = [
      '#type' => 'details',
      '#title' => $this->t('Paypage settings'),
      '#description' => $this->t('Optional settings to be applied to the Unzer paypage.'),
      '#open' => FALSE,
    ];

    foreach (self::getUnzerOptionalPaypageSettings() as $setting_name => $setting_data) {
      $form['paypage_settings'][$setting_name] = [
        '#type' => 'textfield',
        '#title' => $this->t($setting_data['title']),
        '#description' => $this->t($setting_data['description']),
        '#default_value' => $this->configuration['paypage_settings'][$setting_name],
        '#size' => 60,
      ];
    }

    // Make it clear that this option does not change how Unzer will treat the
    // transactions.
    $form['mode']['#description'] = $this->t('NB: This will mark the payments in Drupal but will <em>not</em> change how Unzer will treat the transactions. If you need to make test transactions you will need test keys from Unzer and set those as private and public keys.');

    return $form;
  }

  /**
   * {@inheritdoc}
   *
   * @param \Drupal\commerce_payment\Entity\PaymentInterface $payment
   *   The payment.
   *
   * @return array<string,mixed>
   *   A render array with payment instructions from the configuration.
   */
  public function buildPaymentInstructions(PaymentInterface $payment): array {
    $configuration = $this->getConfiguration();

    $instructions = [];

    if (!empty($configuration['instructions']['value'])) {
      $instructions = [
        '#type' => 'processed_text',
        '#text' => $configuration['instructions']['value'],
        '#format' => $configuration['instructions']['format'],
      ];
    }

    return $instructions;
  }

  /**
   * {@inheritdoc}
   *
   * @return array<string,mixed>
   */
  public function defaultConfiguration(): array {
    $defaults = [
      'private_key' => '',
      'public_key' => '',
      'instructions' => [
        'value' => '',
        'format' => 'plain_text',
      ],
      'debug_logging' => FALSE,
      'excluded_payment_types' => [],
    ];

    foreach ($this->getUnzerOptionalPaypageSettings() as $setting_name => $setting_data) {
      $defaults['paypage_settings'][$setting_name] = '';
    }

    return $defaults + parent::defaultConfiguration();
  }

  /**
   * Gets the mapping of optional attributes for the Unzer paypage window.
   *
   * @return array<string, array<string, string>> The mapping of optional settings for the Unzer paypage
   *   (Drupal plugin setting name => Unzer method name)
   */
  public static function getUnzerOptionalPaypageSettings(): array {
    return [
      'logo_image' => [
        'title' => 'The URL to your logo',
        'description' => 'Make sure the image file is accessible by Unzer or any transaction will error out.',
        'method' => 'setLogoImage',
        'type' => 'style',
      ],
      'full_page_image' => [
        'title' => 'The URL to the background image for the payment window',
        'description' => 'Make sure the imagefile is accessible by Unzer or any transaction will error out.',
        'method' => 'setFullPageImage',
        'type' => 'style',
      ],
      'shop_name' => [
        'title' => 'The name of your shop',
        'description' => '',
        'method' => 'setShopName',
        'type' => 'toplevel',
      ],
      'shop_description' => [
        'title' => 'The description of your shop',
        'description' => '',
        'method' => 'setShopDescription',
        'type' => 'toplevel',
      ],
      'tagline' => [
        'title' => 'Your tag line',
        'description' => 'Displayed as subtitle in the payment window.',
        'method' => 'setTagline',
        'type' => 'toplevel',
      ],
      'terms_and_conditions_url' => [
        'title' => 'The URL of your terms and conditions page',
        'description' => '',
        // The Unzer SDK method really has "condition" in the singular form.
        'method' => 'setTermsAndConditionUrl',
        'type' => 'url',
      ],
      'privacy_policy_url' => [
        'title' => 'The URL of your privacy policy page',
        'description' => '',
        'method' => 'setPrivacyPolicyUrl',
        'type' => 'url',
      ],
      'imprint_url' => [
        'title' => 'The URL of your imprint page',
        'description' => '',
        'method' => 'setImprintUrl',
        'type' => 'url',
      ],
      'help_url' => [
        'title' => 'The URL of your help page',
        'description' => '',
        'method' => 'setHelpUrl',
        'type' => 'url',
      ],
      'contact_url' => [
        'title' => 'The URL of your contact page',
        'description' => '',
        'method' => 'setContactUrl',
        'type' => 'url',
      ],
    ];
  }

  /**
   * Called if the user hits the "Back to merchant" button.
   *
   * Will send the user back to the previous step of the checkout flow.
   *
   * This method is called manually, since Unzer does not have a cancel URL.
   *
   * @throws \Drupal\commerce\Response\NeedsRedirectException
   */
  public function onCancel(OrderInterface $order, Request $request): void {
    parent::onCancel($order, $request);

    // Take one step back.
    /** @var \Drupal\commerce_checkout\Entity\CheckoutFlowInterface $checkout_flow */
    // @phpstan-ignore-next-line
    $checkout_flow = $order->get('checkout_flow')->entity;
    $checkout_flow_plugin = $checkout_flow->getPlugin();
    // @phpstan-ignore-next-line
    $step_id = $order->get('checkout_step')->value;
    $previous_step_id = $checkout_flow_plugin->getPreviousStepId($step_id);
    $checkout_flow_plugin->redirectToStep($previous_step_id);
    // The above does not return. It throws a NeedsRedirectException.
  }

  /**
   * {@inheritdoc}
   */
  public function onNotify(Request $request): ?Response {
    // @TODO.
    return NULL;
  }

  /**
   * {@inheritdoc}
   *
   * @throws \Drupal\commerce\Response\NeedsRedirectException
   */
  public function onReturn(OrderInterface $order, Request $request): void {
    $this->processNotification($order);
  }

  /**
   * {@inheritdoc}
   *
   * @param array<string,mixed> $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    parent::submitConfigurationForm($form, $form_state);

    if (!$form_state->getErrors()) {
      $values = $form_state->getValue($form['#parents']);

      $this->configuration['private_key'] = $values['private_key'];
      $this->configuration['public_key'] = $values['public_key'];
      $this->configuration['instructions'] = $values['instructions'];
      $this->configuration['debug_logging'] = $values['debug_logging'];
      // ept_details is just a wrapper, move values one level up in the config
      $this->configuration['excluded_payment_types'] = $values['ept_details']['excluded_payment_types'];
      // paypage_settings is a logical grouping of settings, keep extra level
      $this->configuration['paypage_settings'] = $values['paypage_settings'];
    }
  }

  /**
   * Creates an array of Unzer payment types keyed by their resource names.
   *
   * The class names are taken from the class definitions in
   * vendor/unzerdev/php-sdk/src/Resources/PaymentTypes.
   *
   * @return array<string,\Drupal\Core\StringTranslation\TranslatableMarkup> An array of payment types (resource name => translated name)
   */
  public static function getUnzerPaymentTypes(): array {
    $payment_types = [];
    $classes = [
      'Alipay' => 'Alipay',
      'Applepay' => 'Applepay',
      'Bancontact' => 'Bancontact',
      'Card' => 'Card',
      'Clicktopay' => 'Clicktopay',
      'EPS' => 'EPS',
      'Googlepay' => 'Googlepay',
      'Ideal' => 'Ideal',
//      'InstallmentSecured' => 'Installment - Secured',
//      'Invoice' => 'Invoice',
//      'InvoiceSecured' => 'Invoice - Secured',
//      'OpenbankingPis' => 'OpenbankingPis',
      'PaylaterDirectDebit' => 'PaylaterDirectDebit',
      'PaylaterInstallment' => 'PaylaterInstallment',
      'PaylaterInvoice' => 'PaylaterInvoice',
      'Paypal' => 'Paypal',
      'PayU' => 'PayU',
//      'PIS' => 'PIS',
      'PostFinanceCard' => 'PostFinanceCard',
      'PostFinanceEfinance' => 'PostFinanceEfinance',
      'Prepayment' => 'Prepayment',
      'Przelewy24' => 'Przelewy 24',
      'SepaDirectDebit' => 'Sepa Direct Debit',
//      'SepaDirectDebitSecured' => 'Sepa Direct Debit - Secured',
      'Twint' => 'Twint',
      'Wechatpay' => 'Wechatpay',
    ];

    foreach ($classes as $class => $label) {
      $full_class_name = 'UnzerSDK\\Resources\\PaymentTypes\\' . $class;
      $type_id = $full_class_name::getResourceName();
      // @phpstan-ignore-next-line
      $payment_types[$type_id] = [
        'label' => t($label, [], ['context' => 'Unzer payment type']),
        'class' => $class,
      ];
    }

    return $payment_types;
  }

  /**
   * Processes the notification.
   *
   * @param \Drupal\commerce_order\Entity\OrderInterface $order
   *   The order.
   * @param string $payment_id
   *   The external payment id.
   *
   * @throws \Drupal\commerce_payment\Exception\InvalidRequestException
   */
  protected function processNotification(OrderInterface $order): void {
    $order_custom_data = $order->getData('commerce_unzer');

    if (empty($order_custom_data['paypage_id'])) {
      throw new InvalidRequestException('An error has occurred trying to process the payment data.');
    }
    $paypage_id = $order_custom_data['paypage_id'];

    try {
      $unzer = new Unzer($this->getConfiguration()['private_key']);

      $unzer->setDebugMode($this->configuration['debug_logging'])->setDebugHandler(new DebugHandler($this->logger));

      $paypage = $unzer->fetchPaypageV2($paypage_id);

      $payments = $paypage->getPayments();

      if (empty($payments)) {
        throw new InvalidRequestException('An error has occurred trying to process the payment data (no payments found).');
      }

      /** @var \UnzerSDK\Resources\EmbeddedResources\Paypage\Payment $unzer_payment */
      $unzer_embedded_payment = reset($payments);
      $payment_id = $unzer_embedded_payment->getPaymentId();

      // Get the non-embedded payment.
      $unzer_payment = $unzer->fetchpayment($payment_id);
      $unzer_payment_state = $unzer_payment->getStateName();

      if ($unzer_payment_state !== PaymentState::STATE_NAME_COMPLETED) {
        throw new InvalidRequestException('An error has occurred trying to process the payment data (payment not completed).');
      }

      // Use the short ID of the initial transaction as remote ID because that
      // is the one displayed in Unzer insights. Fall back to the payment ID as
      // a last resort if we cannot get the short ID.
      $initial_transaction = $unzer_payment->getInitialTransaction();
      $remote_id = $payment_id;

      if ($initial_transaction instanceof AbstractTransactionType && !empty($initial_transaction->getShortId())) {
        $remote_id = $initial_transaction->getShortId();
      }

      // Add the Unzer payment ID to the Commerce order data.
      $order_custom_data['payment_id'] = $payment_id;
      $order->setData('commerce_unzer', $order_custom_data);
      $order->save();

      $payment = $this->entityTypeManager->getStorage('commerce_payment')
        ->create([
          'state' => $unzer_payment_state,
          'amount' => $order->getTotalPrice(),
          'payment_gateway' => $this->parentEntity->id(),
          'order_id' => $order->id(),
          'remote_id' => $remote_id,
          'remote_state' => $unzer_payment_state,
          'authorized' => $this->time->getRequestTime(),
        ]);
      $payment->save();
    }
    catch (\Exception $e) {
      throw new InvalidRequestException('An error has occurred trying to process the payment data (internal error), please contact us.', 0, $e);
    }
  }

}
