<?php

declare(strict_types=1);

namespace Drupal\commerce_usaepay\Plugin\Commerce\PaymentGateway;

use Drupal\commerce_payment\CreditCard;
use Drupal\commerce_payment\Entity\PaymentInterface;
use Drupal\commerce_payment\Entity\PaymentMethodInterface;
use Drupal\commerce_payment\Plugin\Commerce\PaymentGateway\OnsitePaymentGatewayBase;
use Drupal\commerce_price\Price;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides the USAePay payment gateway.
 *
 * @CommercePaymentGateway(
 *   id = "usaepay",
 *   label = "USAePay",
 *   display_label = "USAePay",
 *   payment_method_types = {"credit_card"},
 *   credit_card_types = {
 *     "amex", "dinersclub", "discover", "jcb", "mastercard", "visa",
 *   },
 * )
 */
class USAePay extends OnsitePaymentGatewayBase implements USAePayInterface, ContainerFactoryPluginInterface {

  /**
   * The PHP SOAP client.
   *
   * @var \SoapClient
   */
  protected $soapClient;

  /**
   * The private temp store factory.
   *
   * @var \Drupal\Core\TempStore\PrivateTempStoreFactory
   */
  protected $privateTempStore;

  /**
   * The error helper service.
   *
   * @var \Drupal\commerce_usaepay\ErrorHelper
   */
  protected $errorHelper;

  /**
   * Creates an instance of the USAePay payment gateway plugin.
   *
   * Injects private tempstore, ErrorHelper & initializes the SOAP client
   * based on the configured WSDL key and mode (test/live).
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);

    $instance->privateTempStore = $container->get('tempstore.private');
    $instance->errorHelper = $container->get('commerce_usaepay.error_helper');

    if (!empty($instance->configuration['wsdl_key'])) {
      $wsdl = $instance->getMode() === 'test'
        ? 'https://sandbox.usaepay.com/soap/gate/' . $instance->configuration['wsdl_key'] . '/usaepay.wsdl'
        : 'https://usaepay.com/soap/gate/' . $instance->configuration['wsdl_key'] . '/usaepay.wsdl';
      try {
        $instance->soapClient = new \SoapClient($wsdl);
      }
      catch (\SoapFault $e) {
        $instance->errorHelper->handleException($e);
      }
      catch (Exception $e) {
        $instance->errorHelper->handleException($e);
      }
    }

    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'wsdl_key' => '',
      'source_key' => '',
      'pin' => '',
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $form = parent::buildConfigurationForm($form, $form_state);

    $form['wsdl_key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('WSDL Key'),
      '#default_value' => $this->configuration['wsdl_key'],
      '#description' => $this->t("WSDL API endpoint key generated from 'https://sandbox.usaepay.com/_developer/app/login'.  Ex: https://www.usaepay.com/soap/gate/<strong>ABCD1234</strong>/usaepay.wsdl."),
      '#required' => TRUE,
    ];

    $form['source_key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Source Key'),
      '#default_value' => $this->configuration['source_key'],
      '#description' => $this->t('Source key for the merchant account generated by the Merchant Console at www.usaepay.com.'),
      '#required' => TRUE,
    ];

    $form['pin'] = [
      '#type' => 'textfield',
      '#title' => $this->t('PIN for Source Key'),
      '#default_value' => $this->configuration['pin'],
      '#description' => $this->t("While USAePay makes PIN numbers optional it is recommended that you always use a PIN and since the 'sale' method of the USAePay SOAP API requires it, it is a mandatory field for this module."),
      '#required' => TRUE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state) {

  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    parent::submitConfigurationForm($form, $form_state);

    if (!$form_state->getErrors()) {
      $values = $form_state->getValue($form['#parents']);
      $this->configuration['wsdl_key'] = $values['wsdl_key'];
      $this->configuration['source_key'] = $values['source_key'];
      $this->configuration['pin'] = $values['pin'];
    }
  }

  /**
   * {@inheritdoc}
   */
  public function createPayment(PaymentInterface $payment, $capture = TRUE) {

    $this->assertPaymentState($payment, ['new']);
    $payment_method = $payment->getPaymentMethod();
    $this->assertPaymentMethod($payment_method);

    $owner = $payment_method->getOwner();
    if ($owner && $owner->isAuthenticated()) {
      $cust_num = $this->getRemoteCustomerId($owner);
    }
    else {
      $cust_num = $this->privateTempStore->get('commerce_usaepay')->get('cust_num');
    }

    $order = $payment->getOrder();
    $amount = $payment->getAmount();

    if ($capture) {
      $command = 'Sale';
    }
    else {
      $command = 'AuthOnly';
    }

    try {
      $transaction_request = [
        'Command' => $command,
        'Details' => [
          'Invoice' => $order->id(),
          'OrderID' => $order->id() . '-' . $this->time->getRequestTime(),
          'Description' => 'Purchase from website',
          'Amount' => $amount->getNumber(),
        ],
      ];

      $response = $this->soapClient->runCustomerTransaction($this->buildToken(),
        $cust_num, $payment_method->getRemoteId(), $transaction_request);
      $this->errorHelper->handleErrors($response);
    }
    catch (\SoapFault $e) {
      $this->errorHelper->handleException($e);
    }
    catch (Exception $e) {
      $this->errorHelper->handleException($e);
    }

    $next_state = $capture ? 'completed' : 'authorization';
    $payment->setState($next_state);
    $payment->setRemoteId($response->RefNum);
    $payment->save();
  }

  /**
   * {@inheritdoc}
   */
  public function capturePayment(PaymentInterface $payment, Price $amount = NULL) {

    $this->assertPaymentState($payment, ['authorization']);
    $amount = $amount ?: $payment->getAmount();

    try {
      $ref_num = $payment->getRemoteId();
      $number = $amount->getNumber();

      // @todo Add configuration variable so site owner can decide what will
      // happen if the authorization has expired.  Possible options are 'Error',
      // 'ReAuth' and 'Capture'.
      $this->soapClient->captureTransaction($this->buildToken(), $ref_num, $number, 'ReAuth');
    }
    catch (\SoapFault $e) {
      $this->errorHelper->handleException($e);
    }
    catch (Exception $e) {
      $this->errorHelper->handleException($e);
    }

    $payment->setState('completed');
    $payment->setAmount($amount);
    $payment->save();
  }

  /**
   * {@inheritdoc}
   */
  public function voidPayment(PaymentInterface $payment) {

    $this->assertPaymentState($payment, ['authorization']);

    try {
      $ref_num = $payment->getRemoteId();
      $this->soapClient->voidTransaction($this->buildToken(), $ref_num);
    }
    catch (\SoapFault $e) {
      $this->errorHelper->handleException($e);
    }
    catch (Exception $e) {
      $this->errorHelper->handleException($e);
    }

    $payment->setState('authorization_voided');
    $payment->save();
  }

  /**
   * {@inheritdoc}
   */
  public function refundPayment(PaymentInterface $payment, Price $amount = NULL) {

    $this->assertPaymentState($payment, ['completed', 'partially_refunded']);
    $amount = $amount ?: $payment->getAmount();
    $this->assertRefundAmount($payment, $amount);

    try {
      $ref_num = $payment->getRemoteId();
      $number = $amount->getNumber();
      $this->soapClient->refundTransaction($this->buildToken(), $ref_num, $number);
    }
    catch (\SoapFault $e) {
      $this->errorHelper->handleException($e);
    }
    catch (Exception $e) {
      $this->errorHelper->handleException($e);
    }

    $old_refunded_amount = $payment->getRefundedAmount();
    $new_refunded_amount = $old_refunded_amount->add($amount);
    if ($new_refunded_amount->lessThan($payment->getAmount())) {
      $payment->setState('partially_refunded');
    }
    else {
      $payment->setState('refunded');
    }

    $payment->setRefundedAmount($new_refunded_amount);
    $payment->save();
  }

  /**
   * {@inheritdoc}
   */
  public function createPaymentMethod(PaymentMethodInterface $payment_method, array $payment_details) {

    $owner = $payment_method->getOwner();
    $payment_method->card_type = $payment_details['type'];
    $payment_method->card_number = substr($payment_details['number'], -4);
    $payment_method->card_exp_month = $payment_details['expiration']['month'];
    $payment_method->card_exp_year = $payment_details['expiration']['year'];

    if ($owner && $owner->isAuthenticated()) {
      $expires = CreditCard::calculateExpirationTimestamp($payment_details['expiration']['month'], $payment_details['expiration']['year']);
    }
    else {
      $expires = $this->time->getRequestTime() + (3600 * 3);
    }

    $payment_method->setExpiresTime($expires);

    if ($owner && $owner->isAuthenticated()) {
      $cust_num = $this->getRemoteCustomerId($owner);
      if (empty($cust_num)) {
        $cust_num = $this->createRemoteCustomer($payment_method);
        $this->setRemoteCustomerId($owner, $cust_num);
        $owner->save();
      }
      else {
        $this->updateRemoteCustomerBilling($cust_num, $payment_method);
      }
    }
    else {
      $cust_num = $this->createRemoteCustomer($payment_method);
      $this->privateTempStore->get('commerce_usaepay')->set('cust_num', $cust_num);
    }

    if (empty($payment_method->getRemoteId())) {
      $remote_id = $this->createRemotePaymentMethod($cust_num, $payment_details);
      $payment_method->setRemoteId($remote_id);
    }

    $payment_method->save();
  }

  /**
   * {@inheritdoc}
   */
  public function deletePaymentMethod(PaymentMethodInterface $payment_method) {

    // Delete the remote record.
    $owner = $payment_method->getOwner();
    $cust_num = $this->getRemoteCustomerId($owner);

    try {
      $this->soapClient->getCustomerPaymentMethods($this->buildToken(), $cust_num);
      $this->soapClient->deleteCustomerPaymentMethod($this->buildToken(), $cust_num, $payment_method->getRemoteId());
    }
    catch (\SoapFault $e) {
      $this->errorHelper->handleException($e);
    }
    catch (Exception $e) {
      $this->errorHelper->handleException($e);
    }

    // Delete the local entity.
    $payment_method->delete();
  }

  /**
   * Adds customer details to USAePay gateway.
   *
   * @param \Drupal\commerce_payment\Entity\PaymentMethodInterface $payment_method
   *   The payment method.
   *
   * @return int
   *   USAePay customer number.
   */
  private function createRemoteCustomer(PaymentMethodInterface $payment_method) {

    $address = $payment_method->getBillingProfile()->get('address')->first();

    $arr_billing_address = [
      'FirstName' => $address->getGivenName(),
      'LastName' => $address->getFamilyName(),
      'Company' => $address->getOrganization(),
      'Street' => $address->getAddressLine1(),
      'Street2' => $address->getAddressLine2(),
      'City' => $address->getLocality(),
      'State' => $address->getAdministrativeArea(),
      'Zip' => $address->getPostalCode(),
      'Country' => $address->getCountryCode(),
      'Email' => $payment_method->getOwner()->getEmail(),
    ];

    $customer = [
      'CustomerID' => $payment_method->getOwner()->id(),
      'BillingAddress' => $arr_billing_address,
      'Enabled' => FALSE,
      'Schedule' => '',
      'NumLeft' => '',
      'Next' => '',
      'Amount' => '',
      'Description' => '',
      'SendReceipt' => FALSE,
      'ReceiptNote' => '',
      'OrderID' => rand(),
    ];

    try {
      $cust_num = $this->soapClient->addCustomer($this->buildToken(), $customer);
    }
    catch (\SoapFault $e) {
      $this->errorHelper->handleException($e);
    }
    catch (Exception $e) {
      $this->errorHelper->handleException($e);
    }

    return $cust_num;
  }

  /**
   * Adds customer details to USAePay gateway.
   *
   * @param int $cust_num
   *   USAePay customer number.
   * @param \Drupal\commerce_payment\Entity\PaymentMethodInterface $payment_method
   *   The payment method.
   */
  private function updateRemoteCustomerBilling($cust_num, PaymentMethodInterface $payment_method) {

    $address = $payment_method->getBillingProfile()->get('address')->first();

    $arr_billing_address = [
      'FirstName' => $address->getGivenName(),
      'LastName' => $address->getFamilyName(),
      'Company' => $address->getOrganization(),
      'Street' => $address->getAddressLine1(),
      'Street2' => $address->getAddressLine2(),
      'City' => $address->getLocality(),
      'State' => $address->getAdministrativeArea(),
      'Zip' => $address->getPostalCode(),
      'Country' => $address->getCountryCode(),
      'Email' => $payment_method->getOwner()->getEmail(),
    ];

    try {
      $customer = $this->soapClient->getCustomer($this->buildToken(), $cust_num);
      $customer->BillingAddress = $arr_billing_address;
      $this->soapClient->updateCustomer($this->buildToken(), $cust_num, $customer);
    }
    catch (\SoapFault $e) {
      $this->errorHelper->handleException($e);
    }
    catch (Exception $e) {
      $this->errorHelper->handleException($e);
    }
  }

  /**
   * Adds customer payment method to USAePay gateway.
   *
   * @param int $cust_num
   *   USAePay customer number.
   * @param array $payment_details
   *   The gateway-specific payment details.
   *
   * @return int
   *   USAePay ID of payment method.
   */
  private function createRemotePaymentMethod($cust_num, array $payment_details) {

    $arr_payment_method = [
      'MethodType' => 'CreditCard',
      'MethodName' => '',
      'SecondarySort' => 0,
      'CardNumber' => $payment_details['number'],
      'CardExpiration' => $payment_details['expiration']['year'] . '-' . $payment_details['expiration']['month'],
      'CardCode' => $payment_details['security_code'],
    ];

    try {
      $remote_id = $this->soapClient->addCustomerPaymentMethod($this->buildToken(),
        $cust_num, $arr_payment_method, FALSE, FALSE);
    }
    catch (\SoapFault $e) {
      $this->errorHelper->handleException($e);
    }
    catch (Exception $e) {
      $this->errorHelper->handleException($e);
    }

    return $remote_id;
  }

  /**
   * Builds ueSecurityToken object for secure merchant identification.
   *
   * @return array
   *   An array with the security token details.
   */
  private function buildToken() {

    // Generate random seed value.
    $seed = microtime(TRUE) . rand();

    // Assemble prehash data.
    $prehash = $this->configuration['source_key'] . $seed . trim($this->configuration['pin']);

    // Hash the data.
    $hash = sha1($prehash);

    // Assemble ueSecurityToken as an array.
    $token = [
      'SourceKey' => $this->configuration['source_key'],
      'PinHash' => [
        'Type' => 'sha1',
        'Seed' => $seed,
        'HashValue' => $hash,
      ],
      'ClientIP' => \Drupal::request()->getClientIp(),
    ];

    return $token;
  }

  /**
   * Sets the API key after the plugin is unserialized.
   */
  public function __wakeup(): void {
    parent::__wakeup();

    if (!empty($this->configuration['wsdl_key'])) {

      if ($this->getMode() === 'test') {
        $wsdl = 'https://sandbox.usaepay.com/soap/gate/' . $this->configuration['wsdl_key'] . '/usaepay.wsdl';
      }
      else {
        $wsdl = 'https://usaepay.com/soap/gate/' . $this->configuration['wsdl_key'] . '/usaepay.wsdl';
      }

      try {
        $this->soapClient = new \SoapClient($wsdl);
      }
      catch (\SoapFault $e) {
        $this->errorHelper->handleException($e);
      }
      catch (Exception $e) {
        $this->errorHelper->handleException($e);
      }
    }
  }

}
