<?php

namespace Drupal\commerce_usps\Plugin\Commerce\ShippingMethod;

use Drupal\commerce_shipping\Entity\ShipmentInterface;
use Drupal\commerce_shipping\Plugin\Commerce\ShippingMethod\ShippingMethodBase;
use Drupal\commerce_shipping\Plugin\Commerce\ShippingMethod\SupportsTrackingInterface;
use Drupal\commerce_usps\USPSRateRequest;
use Drupal\commerce_usps\USPSSdkFactoryInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\State\StateInterface;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Base class for USPS shipping method plugins.
 */
abstract class USPSBase extends ShippingMethodBase implements SupportsTrackingInterface {

  /**
   * The service for fetching shipping rates from USPS.
   */
  protected USPSRateRequest $uspsRateService;

  /**
   * The USPS SDK factory.
   */
  protected USPSSdkFactoryInterface $sdkFactory;

  /**
   * The state service.
   */
  protected StateInterface $state;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): self {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->uspsRateService = $container->get('commerce_usps.usps_rate_request');
    $instance->uspsRateService->setConfig($configuration);
    $instance->sdkFactory = $container->get('commerce_usps.usps_sdk_factory');
    $instance->state = $container->get('state');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'rate_label' => '',
      'rate_description' => '',
      'api_information' => [
        'client_id' => '',
        'secret' => '',
        'mode' => 'test',
      ],
      'rate_options' => [
        'price_type' => 'retail',
        'account_type' => 'eps',
        'account_number' => '',
        'account_crid' => '',
        'categories' => [],
        'rate_indicators' => [],
        'facility_types' => [],
        'rate_multiplier' => 1.0,
        'round' => PHP_ROUND_HALF_UP,
      ],
      'options' => [
        'tracking_url' => 'https://tools.usps.com/go/TrackConfirmAction?tLabels=[tracking_code]',
        'log' => [],
      ],
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);

    $form['rate_label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Rate label'),
      '#description' => $this->t('Shown to customers when selecting the rate. Leave blank to use the Label provided by USPS'),
      '#default_value' => $this->configuration['rate_label'],
      '#weight' => -10,
    ];
    $form['rate_description'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Rate description'),
      '#description' => $this->t('Provides additional details about the rate to the customer.'),
      '#default_value' => $this->configuration['rate_description'],
      '#weight' => -9,
    ];

    // Select all services by default.
    if (empty($this->configuration['services'])) {
      $service_ids = array_keys($this->services);
      $this->configuration['services'] = array_combine($service_ids, $service_ids);
    }

    $description = $this->t('Update your USPS API information. This is obtained from the <a href="https://developers.usps.com">USPS developer portal</a>.');
    if (!$this->isConfigured()) {
      $description = $this->t('Fill in your USPS API information from the <a href="https://developers.usps.com">USPS developer portal</a>.');
    }

    // API credentials.
    $form['api_information'] = [
      '#type' => 'details',
      '#title' => $this->t('API information'),
      '#description' => $description,
      '#weight' => $this->isConfigured() ? 10 : -10,
      '#open' => !$this->isConfigured(),
    ];
    $form['api_information']['client_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Consumer key'),
      '#default_value' => $this->configuration['api_information']['client_id'],
      '#required' => TRUE,
    ];
    $form['api_information']['secret'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Consumer secret'),
      '#default_value' => $this->configuration['api_information']['secret'],
      '#description' => $this->t('To change the current secret, enter the new one.'),
      '#required' => TRUE,
    ];
    $form['api_information']['mode'] = [
      '#type' => 'select',
      '#title' => $this->t('Mode'),
      '#description' => $this->t('Choose whether to use the test or live mode.'),
      '#options' => [
        'test' => $this->t('Test'),
        'live' => $this->t('Live'),
      ],
      '#default_value' => $this->configuration['api_information']['mode'],
    ];

    // Rate options.
    $form['rate_options'] = [
      '#type' => 'details',
      '#title' => $this->t('Rate options'),
      '#description' => $this->t('Options to pass during rate requests.'),
    ];
    $form['rate_options']['price_type'] = [
      '#type' => 'select',
      '#title' => $this->t('Price type'),
      '#description' => $this->t('The price type to use for shipping rate prices.'),
      '#options' => [
        'retail' => $this->t('Retail'),
        'commercial' => $this->t('Commercial'),
        'contract' => $this->t('Contract'),
      ],
      '#required' => TRUE,
      '#default_value' => $this->configuration['rate_options']['price_type'],
    ];
    $form['rate_options']['account'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Account information'),
      '#description' => $this->t('Fields below is only required when the "Contract" price type is selected.'),
      '#description_display' => 'before',
      '#states' => [
        'visible' => [
          'select[name$="[rate_options][price_type]"]' => ['value' => 'contract'],
        ],
      ],
    ];
    $form['rate_options']['account']['account_type'] = [
      '#type' => 'select',
      '#title' => $this->t('Account type'),
      '#options' => [
        'eps' => $this->t('EPS'),
        'permit' => $this->t('Permit'),
        'meter' => $this->t('Meter'),
        'mid' => $this->t('MID'),
      ],
      '#default_value' => $this->configuration['rate_options']['account_type'],
    ];
    $form['rate_options']['account']['account_number'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Account number'),
      '#default_value' => $this->configuration['rate_options']['account_number'],
      '#states' => [
        'required' => [
          'select[name$="[rate_options][price_type]"]' => ['value' => 'contract'],
        ],
      ],
    ];
    $form['rate_options']['account']['account_crid'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Customer Registration ID'),
      '#default_value' => $this->configuration['rate_options']['account_crid'],
      '#states' => [
        'required' => [
          'select[name$="[rate_options][price_type]"]' => ['value' => 'contract'],
        ],
      ],
    ];
    $form['rate_options']['categories'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Processing categories'),
      '#options' => [
        'CARDS' => $this->t('Cards'),
        'LETTERS' => $this->t('Letters'),
        'FLATS' => $this->t('Flats'),
        'MACHINABLE' => $this->t('Machinable'),
        'NONSTANDARD' => $this->t('Nonstandard parcel'),
        'CATALOGS' => $this->t('Catalogs'),
        'OPEN_AND_DISTRIBUTE' => $this->t('Open and Distribute'),
        'RETURNS' => $this->t('Returns'),
        'SOFT_PACK_MACHINABLE' => $this->t('Soft Pack Machinable'),
        'SOFT_PACK_NON_MACHINABLE' => $this->t('Soft Package Non-machinable'),
      ],
      '#description' => $this->t('Choose the processing categories to include in rates. Leave empty to use all categories.'),
      '#default_value' => $this->configuration['rate_options']['categories'],
    ];
    $form['rate_options']['rate_indicators'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Rate Indicators (Types)'),
      '#options' => $this->getRateIndicatorOptions(),
      '#description' => $this->t('Choose the rate indicators (rate types) to include in rates. Leave empty to use all rate indicators.'),
      '#default_value' => $this->configuration['rate_options']['rate_indicators'],
    ];
    $form['rate_options']['facility_types'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Destination entry facility types'),
      '#options' => $this->getFacilityTypeOptions(),
      '#description' => $this->t('Choose the destination entry facility types to include in rates. Leave empty to use all facility types.'),
      '#default_value' => $this->configuration['rate_options']['facility_types'],
    ];
    $form['rate_options']['rate_multiplier'] = [
      '#type' => 'number',
      '#title' => $this->t('Rate multiplier'),
      '#description' => $this->t('A number that each rate returned from USPS will be multiplied by. For example, enter 1.5 to mark up shipping costs to 150%.'),
      '#min' => 0.1,
      '#step' => 0.1,
      '#size' => 5,
      '#default_value' => $this->configuration['rate_options']['rate_multiplier'],
    ];
    $form['rate_options']['round'] = [
      '#type' => 'select',
      '#title' => $this->t('Round type'),
      '#description' => $this->t('Choose how the shipping rate should be rounded.'),
      '#options' => [
        PHP_ROUND_HALF_UP => 'Half up',
        PHP_ROUND_HALF_DOWN => 'Half down',
        PHP_ROUND_HALF_EVEN => 'Half even',
        PHP_ROUND_HALF_ODD => 'Half odd',
      ],
      '#default_value' => $this->configuration['rate_options']['round'],
    ];

    // Options.
    $form['options'] = [
      '#type' => 'details',
      '#title' => $this->t('USPS Options'),
      '#description' => $this->t('Additional options for USPS'),
    ];
    $form['options']['tracking_url'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Tracking URL base'),
      '#description' => $this->t('The base URL for assembling a tracking URL. If the [tracking_code] token is omitted, the code will be appended to the end of the URL (e.g. "https://tools.usps.com/go/TrackConfirmAction?tLabels=123456789")'),
      '#default_value' => $this->configuration['options']['tracking_url'],
    ];
    $form['options']['log'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Log the following messages for debugging'),
      '#options' => [
        'request' => $this->t('API request messages'),
        'response' => $this->t('API response messages'),
      ],
      '#default_value' => $this->configuration['options']['log'],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state): void {
    parent::validateConfigurationForm($form, $form_state);
    if ($form_state->getErrors()) {
      return;
    }
    $values = $form_state->getValue($form['#parents']);
    $api_values = &$values['api_information'];
    if (empty($api_values['client_id']) || empty($api_values['secret'])) {
      return;
    }

    $api_values['client_id'] = trim($api_values['client_id']);
    $api_values['secret'] = trim($api_values['secret']);
    $sdk = $this->sdkFactory->get($api_values);
    $this->state->delete(USPSSdkFactoryInterface::TOKEN_KEY);
    try {
      $sdk->getAccessToken();
      $form_state->setValue($form['#parents'], $values);
      $this->messenger()->addMessage($this->t('Connectivity to USPS successfully verified.'));
    }
    catch (\Exception $e) {
      $this->messenger()->addError($this->t('Invalid <em>Consumer key</em> or <em>Consumer secret</em> specified.'));
      $form_state->setError($form['api_information']['client_id']);
      $form_state->setError($form['api_information']['secret']);
    }

    // If the "Contract" price type selected we require account information for
    // the proper rate requests.
    if ($values['rate_options']['price_type'] === 'contract') {
      if (empty($values['rate_options']['account']['account_number'])) {
        $form_state->setError($form['rate_options']['account']['account_number'], 'The "Account number" is required when "Contract" price type is selected.');
      }
      if (empty($values['rate_options']['account']['account_crid'])) {
        $form_state->setError($form['rate_options']['account']['account_crid'], 'The "Customer Registration ID" is required when "Contract" price type is selected.');
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    if (!$form_state->getErrors()) {
      $values = $form_state->getValue($form['#parents']);

      $this->configuration['rate_label'] = $values['rate_label'];
      $this->configuration['rate_description'] = $values['rate_description'];
      $this->configuration['api_information']['client_id'] = $values['api_information']['client_id'];
      $this->configuration['api_information']['secret'] = $values['api_information']['secret'];
      $this->configuration['api_information']['mode'] = $values['api_information']['mode'];
      $this->configuration['rate_options']['price_type'] = $values['rate_options']['price_type'];
      $this->configuration['rate_options']['account_type'] = $values['rate_options']['account']['account_type'];
      $this->configuration['rate_options']['account_number'] = $values['rate_options']['account']['account_number'];
      $this->configuration['rate_options']['account_crid'] = $values['rate_options']['account']['account_crid'];
      $this->configuration['rate_options']['categories'] = array_filter($values['rate_options']['categories']);
      $this->configuration['rate_options']['rate_indicators'] = array_filter($values['rate_options']['rate_indicators']);
      $this->configuration['rate_options']['facility_types'] = array_filter($values['rate_options']['facility_types']);
      $this->configuration['rate_options']['rate_multiplier'] = $values['rate_options']['rate_multiplier'];
      $this->configuration['rate_options']['round'] = $values['rate_options']['round'];
      $this->configuration['options']['log'] = $values['options']['log'];
    }
    parent::submitConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function calculateRates(ShipmentInterface $shipment) {
    // Only attempt to collect rates if an address exists on the shipment.
    if ($shipment->getShippingProfile()->get('address')->isEmpty()) {
      return [];
    }
    if ($shipment->getPackageType() === NULL) {
      $shipment->setPackageType($this->getDefaultPackageType());
    }

    return $this->uspsRateService->getRates($shipment, $this->parentEntity);
  }

  /**
   * {@inheritdoc}
   */
  public function getTrackingUrl(ShipmentInterface $shipment) {
    $code = $shipment->getTrackingCode();
    if (!empty($code)) {
      // If the tracking code token exists, replace it with the code.
      if (str_contains($this->configuration['options']['tracking_url'], '[tracking_code]')) {
        $url = str_replace('[tracking_code]', $code, $this->configuration['options']['tracking_url']);
      }
      else {
        // Otherwise, append the tracking code to the end of the URL.
        $url = $this->configuration['options']['tracking_url'] . $code;
      }

      return Url::fromUri($url);
    }
    return FALSE;
  }

  /**
   * Determine if we have the minimum information to connect to USPS.
   *
   * @return bool
   *   TRUE if there is enough information to connect, FALSE otherwise.
   */
  protected function isConfigured() {
    $api_config = $this->configuration['api_information'];

    if (empty($api_config['client_id']) || empty($api_config['secret'])) {
      return FALSE;
    }

    return TRUE;
  }

  /**
   * Gets list of Mail class options.
   */
  abstract public function getMailClass(): string;

  /**
   * Returns the list of allowed facility types.
   */
  abstract public function getFacilityTypeOptions(): array;

  /**
   * Returns the list of allowed rate indicators.
   */
  abstract public function getRateIndicatorOptions(): array;

}
