<?php

namespace Drupal\commerce_usps;

use Drupal\commerce_price\Price;
use Drupal\commerce_price\RounderInterface;
use Drupal\commerce_shipping\Entity\ShipmentInterface;
use Drupal\commerce_shipping\Entity\ShippingMethodInterface;
use Drupal\commerce_shipping\ShippingRate;
use Drupal\commerce_shipping\ShippingService;
use Drupal\Core\Datetime\DrupalDateTime;
use Psr\Log\LoggerInterface;

/**
 * Class for fetching and returning rates using the USPS API.
 *
 * @package Drupal\commerce_usps
 */
class USPSRateRequest implements USPSRateRequestInterface {

  /**
   * The configuration array from a CommerceShippingMethod.
   */
  protected array $configuration;

  /**
   * Constructs a new USPSRateRequest object.
   *
   * @param \Drupal\commerce_usps\USPSSdkFactoryInterface $uspsSdkFactory
   *   The USPS SDK factory.
   * @param \Drupal\commerce_price\RounderInterface $rounder
   *   The price rounder.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger.
   */
  public function __construct(
    protected USPSSdkFactoryInterface $uspsSdkFactory,
    protected RounderInterface $rounder,
    protected LoggerInterface $logger,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function setConfig(array $configuration): void {
    $this->configuration = $configuration;
  }

  /**
   * {@inheritdoc}
   */
  public function getRates(ShipmentInterface $commerce_shipment, ShippingMethodInterface $shipping_method): array {
    $rates = [];
    $rate_options = $this->configuration['rate_options'];

    try {
      /** @var \CommerceGuys\Addressing\AddressInterface $address_to */
      $address_to = $commerce_shipment->getShippingProfile()->get('address')->first();
      /** @var \Drupal\commerce_usps\Plugin\Commerce\ShippingMethod\USPSBase $plugin */
      $plugin = $shipping_method->getPlugin();
      $plugin_id = $plugin->getPluginId();
      $sdk = $this->uspsSdkFactory->get($this->configuration['api_information']);
      if (($sdk->isDomesticCountry($address_to)) !== ($plugin_id === 'usps')) {
        return [];
      }
      $sdk->setShipment($commerce_shipment);
      $rate_options['mail_class'] = $plugin->getMailClass();
      $sdk->setRateOptions($rate_options);
      // Enable logging.
      $sdk->setLogProcess(
        !empty($this->configuration['options']['log']['request']),
        !empty($this->configuration['options']['log']['response'])
      );
      $usps_rates = $sdk->getShipmentRates();
      if (empty($usps_rates)) {
        return [];
      }
    }
    catch (\Exception $e) {
      $this->logger->error($e->getMessage());
      return [];
    }

    // Get rate modifications.
    $multiplier = $rate_options['rate_multiplier'] ?? 1.0;
    $round = $rate_options['round'] ?? PHP_ROUND_HALF_UP;

    // Convert received rates from USPS to ShippingRates.
    $rate_indicators = $rate_options['rate_indicators'] ?: array_keys($plugin->getRateIndicatorOptions());
    $facility_types = $rate_options['facility_types'] ?: array_keys($plugin->getFacilityTypeOptions());
    foreach ($usps_rates as $usps_rate) {
      $rate = reset($usps_rate['rates']);
      // Only add rates that match configured mail service, rate indicator,
      // category, and facility type.
      if ($this->isNotAllowedRate($rate['mailClass'], $this->configuration['services'])
        || $this->isNotAllowedRate($rate['rateIndicator'], $rate_indicators)
        || $this->isNotAllowedRate($rate['destinationEntryFacilityType'], $facility_types)
        || $this->isNotAllowedRate($rate['processingCategory'], $rate_options['categories'])
      ) {
        continue;
      }

      $name = !empty($rate['productName']) ? $rate['productName'] : $rate['description'];
      $price = new Price($rate['price'], 'USD');
      if ($multiplier != 1) {
        $price = $price->multiply((string) $multiplier);
      }
      $price = $this->rounder->round($price, $round);
      $shipping_rate = new ShippingRate([
        'shipping_method_id' => $shipping_method->id(),
        'service' => new ShippingService($rate['SKU'], $name),
        'amount' => $price,
        'description' => !empty($rate['productDefinition']) ? $rate['productDefinition'] : $rate['description'] ?? NULL,
      ]);

      // Set delivery date if possible.
      $delivery_date = $usps_rate['commitment']['scheduleDeliveryDate'] ?? NULL;
      if ($delivery_date) {
        $shipping_rate->setDeliveryDate(DrupalDateTime::createFromFormat('Y-m-d', $delivery_date));
      }
      $rates[] = $shipping_rate;
    }

    return $rates;
  }

  /**
   * Whether the rate is not allowed to use.
   *
   * @param string $rate_property
   *   The value of rate property.
   * @param array $property_options
   *   The list of allowed values.
   */
  private function isNotAllowedRate(string $rate_property, array $property_options): bool {
    return !empty($property_options) && !in_array($rate_property, $property_options);
  }

}
