<?php

namespace Drupal\commerce_usps;

use CommerceGuys\Addressing\AddressInterface;
use Drupal\commerce_shipping\Entity\ShipmentInterface;
use Drupal\commerce_usps\Event\USPSEvents;
use Drupal\commerce_usps\Event\USPSRateRequestEvent;
use Drupal\Component\Serialization\Json;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\physical\LengthUnit;
use Drupal\physical\Measurement;
use Drupal\physical\WeightUnit;
use GuzzleHttp\ClientInterface;
use Psr\Http\Message\ResponseInterface;
use Psr\Log\LoggerInterface;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;

/**
 * Provides a replacement of the USPS SDK.
 */
class USPSSdk implements USPSSdkInterface {

  /**
   * The client.
   */
  protected ClientInterface $client;

  /**
   * The configuration.
   */
  protected array $config = [];

  /**
   * The options for the rate request.
   */
  protected array $rateOptions = [];

  /**
   * The shipment.
   */
  protected ?ShipmentInterface $shipment;

  /**
   * Whether we should write logs on request.
   */
  protected bool $logRequest = FALSE;

  /**
   * Whether we should write logs on response.
   */
  protected bool $logResponse = FALSE;

  /**
   * Constructs a new USPSSdk object.
   *
   * @param \Drupal\Core\Cache\CacheBackendInterface $cacheBackend
   *   The cache backend.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger.
   * @param \Symfony\Component\EventDispatcher\EventDispatcherInterface $eventDispatcher
   *   The event dispatcher.
   */
  public function __construct(
    protected CacheBackendInterface $cacheBackend,
    protected LoggerInterface $logger,
    protected EventDispatcherInterface $eventDispatcher,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function setClient(ClientInterface $client): void {
    $this->client = $client;
  }

  /**
   * {@inheritdoc}
   */
  public function setConfiguration(array $config): void {
    $this->config = $config;
  }

  /**
   * {@inheritdoc}
   */
  public function setRateOptions(array $rate_options): void {
    $this->rateOptions = $rate_options;
  }

  /**
   * {@inheritdoc}
   */
  public function getAccessToken(): ResponseInterface {
    return $this->client->post(
      self::USPS_ACCESS_TOKEN_URL,
      [
        'json' => [
          'grant_type' => 'client_credentials',
          'client_id' => $this->config['client_id'],
          'client_secret' => $this->config['secret'],
        ],
      ],
    );
  }

  /**
   * {@inheritdoc}
   */
  public function setShipment(ShipmentInterface $shipment): void {
    $this->shipment = $shipment;
  }

  /**
   * {@inheritdoc}
   */
  public function getShipmentRates(): array {
    if (!$this->shipment) {
      throw new \InvalidArgumentException('Shipment not provided.');
    }

    $params = $this->prepareRatesRequest();
    $cid = 'rate-request:' . hash('sha256', serialize($params));
    $rate_request_cache = $this->cacheBackend->get($cid);
    if (!empty($rate_request_cache)) {
      return $rate_request_cache->data;
    }
    if ($this->logRequest) {
      $this->log('Sending USPS Rate request with params', $params);
    }
    $response = $this->client->post(self::USPS_API_RATE_URL, ['json' => $params]);
    $usps_rates = Json::decode($response->getBody());
    if ($this->logResponse) {
      $this->log('Received USPS rate response', $usps_rates);
    }
    if (empty($usps_rates['pricingOptions'][0]['shippingOptions'])) {
      return [];
    }

    // Get only rates from the response.
    $rates = [];
    foreach (array_column($usps_rates['pricingOptions'], 'shippingOptions') as $shipping_option) {
      foreach (array_column($shipping_option, 'rateOptions') as $rate_option) {
        $rates = array_merge($rates, $rate_option);
      }
    }

    $this->cacheBackend->set($cid, $rates, time() + 3600);
    return $rates;
  }

  /**
   * {@inheritdoc}
   */
  public function setLogProcess(bool $log_request = FALSE, bool $log_response = FALSE): void {
    $this->logRequest = $log_request;
    $this->logResponse = $log_response;
  }

  /**
   * {@inheritdoc}
   */
  public function isDomesticCountry(AddressInterface $address): bool {
    return in_array($address->getCountryCode(), ['AS', 'GU', 'MP', 'PR', 'US', 'VI']);
  }

  /**
   * Whether the destination address have the correct postal code.
   *
   * @param \CommerceGuys\Addressing\AddressInterface $address
   *   The shipment destination address.
   */
  protected function isDomesticDestination(AddressInterface $address): bool {
    if ($this->isDomesticCountry($address)) {
      if ($this->isValidPostalCode($address->getPostalCode())) {
        return TRUE;
      }
      throw new \InvalidArgumentException('Destination postal code is not valid for domestic shipments');
    }
    return FALSE;
  }

  /**
   * Whether the postal code is valid for rate request.
   *
   * @param string $postal_code
   *   The postal code.
   */
  protected function isValidPostalCode(string $postal_code): bool {
    return !!preg_match('/^\d{5}(?:[-\s]\d{4})?$/', $postal_code);
  }

  /**
   * Returns array of rate request parameters.
   */
  protected function prepareRatesRequest(): array {
    $price_type = $this->rateOptions['price_type'];
    $pricing_option = [
      'priceType' => strtoupper($price_type),
    ];
    if ($price_type === 'contract') {
      $pricing_option['paymentAccount'] = [
        'accountType' => strtoupper($this->rateOptions['account_type']),
        'accountNumber' => $this->rateOptions['account_number'],
        'CRID' => $this->rateOptions['account_crid'],
      ];
    }

    /** @var \CommerceGuys\Addressing\AddressInterface|null $address_from */
    $address_from = $this->shipment->getOrder()?->getStore()?->getAddress();
    if (!$address_from) {
      throw new \InvalidArgumentException('Store or store address is not provided.');
    }
    if (!$this->isValidPostalCode($address_from->getPostalCode())) {
      throw new \InvalidArgumentException('The original postal code is invalid.');
    }

    /** @var \CommerceGuys\Addressing\AddressInterface $address_to */
    $address_to = $this->shipment->getShippingProfile()->get('address')->first();

    $domestic_destination = $this->isDomesticDestination($address_to);
    $params = [
      'pricingOptions' => [$pricing_option],
      'packageDescription' => $this->getPackageDescription(),
      'originZIPCode' => $address_from->getPostalCode(),
      $domestic_destination ? 'destinationZIPCode' : 'foreignPostalCode' => $address_to->getPostalCode(),
    ];
    if (!$domestic_destination) {
      $params['destinationCountryCode'] = $address_to->getCountryCode();
    }

    // Allow modifications in the rate request parameters.
    $rate_request_event = new USPSRateRequestEvent($params, $this->shipment);
    $this->eventDispatcher->dispatch($rate_request_event, USPSEvents::BEFORE_RATE_REQUEST);

    return $rate_request_event->getRateRequestParams();
  }

  /**
   * Returns array for the package description.
   */
  protected function getPackageDescription(): array {
    $package_type = $this->shipment->getPackageType();
    return [
      'weight' => $this->getUnitNumber($this->shipment->getWeight(), WeightUnit::POUND),
      'length' => $this->getUnitNumber($package_type->getLength(), LengthUnit::INCH),
      'height' => $this->getUnitNumber($package_type->getHeight(), LengthUnit::INCH),
      'width' => $this->getUnitNumber($package_type->getWidth(), LengthUnit::INCH),
      'mailClass' => strtoupper($this->rateOptions['mail_class']),
    ];
  }

  /**
   * Logs information from USPS API requests.
   *
   * @param string $message
   *   The message to log.
   * @param mixed|null $data
   *   Message data.
   */
  protected function log(string $message, $data = NULL): void {
    if ($data) {
      $this->logger->info('@message <br /><pre>@data</pre>', [
        '@message' => $message,
        '@data' => Json::encode($data),
      ]);
    }
    else {
      $this->logger->info($message);
    }
  }

  /**
   * Converts measurement to the provided unist and make it bigger than 0.
   *
   * @param \Drupal\physical\Measurement $measurement
   *   The measurement.
   * @param string $convert_to
   *   The conversion unit.
   */
  private function getUnitNumber(Measurement $measurement, string $convert_to): int {
    $number = ceil($measurement->convert($convert_to)->getNumber());
    return !$number ? 1 : $number;
  }

}
