<?php

namespace Drupal\commerce_usps;

use Drupal\Core\Http\ClientFactory;
use Drupal\Core\State\StateInterface;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\HandlerStack;
use Sainsburys\Guzzle\Oauth2\GrantType\GrantTypeBase;
use Sainsburys\Guzzle\Oauth2\Middleware\OAuthMiddleware;

/**
 * Defines a factory for our custom USPS SDK.
 */
class USPSSdkFactory implements USPSSdkFactoryInterface {

  /**
   * Array of all instantiated USPS SDKs.
   *
   * @var \Drupal\commerce_usps\USPSSdkInterface[]
   */
  protected array $instances = [];

  /**
   * Constructs a new USPSSdkFactory object.
   *
   * @param \Drupal\Core\Http\ClientFactory $clientFactory
   *   The client factory.
   * @param \GuzzleHttp\HandlerStack $stack
   *   The handler stack.
   * @param \Drupal\Core\State\StateInterface $state
   *   The state service.
   * @param \Drupal\commerce_usps\USPSSdkInterface $uspsSdk
   *   The USPS SDK service.
   */
  public function __construct(
    protected ClientFactory $clientFactory,
    protected HandlerStack $stack,
    protected StateInterface $state,
    protected USPSSdkInterface $uspsSdk,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function get(array $api_information): USPSSdkInterface {
    $instance_key = md5(serialize($api_information));
    if (!isset($this->instances[$instance_key])) {
      $client = $this->getClient($api_information);
      $sdk_instance = clone $this->uspsSdk;
      $sdk_instance->setConfiguration($api_information);
      $sdk_instance->setClient($client);
      $this->instances[$instance_key] = $sdk_instance;
    }

    return $this->instances[$instance_key];
  }

  /**
   * Gets a preconfigured HTTP client instance for use by the SDK.
   *
   * @param array $api_information
   *   The configurations.
   */
  protected function getClient(array $api_information): ClientInterface {
    $base_uri = $api_information['mode'] === 'live' ? self::USPS_PRODUCTION_BASE_URL : self::USPS_INTEGRATION_BASE_URL;
    $options = ['base_uri' => $base_uri];
    $client = $this->clientFactory->fromOptions($options);
    // Generates a key for storing the OAuth2 token retrieved from USPS.
    // This is useful in case multiple USPS shipping method instances are
    // configured.
    $token_key = self::TOKEN_KEY . '.' . md5($api_information['client_id'] . $api_information['secret']);
    $config = [
      GrantTypeBase::CONFIG_CLIENT_ID => $api_information['client_id'],
      GrantTypeBase::CONFIG_CLIENT_SECRET => $api_information['secret'],
      GrantTypeBase::CONFIG_TOKEN_URL => USPSSdkInterface::USPS_ACCESS_TOKEN_URL,
      'token_key' => $token_key,
    ];
    $grant_type = new ClientCredentials($client, $config, $this->state);
    $middleware = new OAuthMiddleware($client, $grant_type);

    // Check if we've already requested an OAuth2 token, note that we do not
    // need to check for the expired timestamp here as the middleware is already
    // taking care of that.
    $token = $this->state->get($token_key, FALSE);
    if ($token) {
      $middleware->setAccessToken($token['token'], $token['type'], $token['expires']);
    }

    // When multiple shipping methods with the same API credentials
    // are configured the "stack" is overflowed with the OAuth callbacks which
    // leads to the failed request. We should remove those callback before
    // adding again to make it work properly.
    $this->stack->remove('oauth_before');
    $this->stack->remove('oauth_failure');
    $this->stack->push($middleware->onBefore(), 'oauth_before');
    $this->stack->push($middleware->onFailure(2), 'oauth_failure');
    $options['handler'] = $this->stack;
    $options['auth'] = 'oauth2';
    return $this->clientFactory->fromOptions($options);
  }

}
