<?php

namespace Drupal\Tests\commerce_usps\Kernel;

use CommerceGuys\Addressing\Address;
use Drupal\commerce_order\Entity\OrderInterface;
use Drupal\commerce_price\Entity\CurrencyInterface;
use Drupal\commerce_price\Rounder;
use Drupal\commerce_shipping\Entity\ShipmentInterface;
use Drupal\commerce_shipping\Entity\ShippingMethodInterface;
use Drupal\commerce_shipping\Plugin\Commerce\PackageType\PackageTypeInterface;
use Drupal\commerce_shipping\Plugin\Commerce\ShippingMethod\ShippingMethodInterface as ShipmentMethodPluginInterface;
use Drupal\commerce_shipping\ShippingRate;
use Drupal\commerce_store\Entity\StoreInterface;
use Drupal\commerce_usps\Plugin\Commerce\ShippingMethod\USPSBase;
use Drupal\commerce_usps\USPSRateRequest;
use Drupal\commerce_usps\USPSSdk;
use Drupal\commerce_usps\USPSSdkFactory;
use Drupal\commerce_usps\USPSSdkFactoryInterface;
use Drupal\Component\Serialization\Json;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Http\ClientFactory;
use Drupal\Core\State\StateInterface;
use Drupal\KernelTests\KernelTestBase;
use Drupal\physical\Length;
use Drupal\physical\Weight;
use Drupal\profile\Entity\ProfileInterface;
use GuzzleHttp\HandlerStack;
use Psr\Log\LoggerInterface;
use Symfony\Component\EventDispatcher\EventDispatcher;

/**
 * Test creating shipments and shipping methods.
 *
 * @package Drupal\Tests\commerce_usps\Kernel
 *
 * @coversDefaultClass \Drupal\commerce_usps\USPSRateRequest
 * @group commerce_usps
 */
abstract class USPSRateRequestTestBase extends KernelTestBase {

  /**
   * Configuration array.
   *
   * @var array
   */
  protected array $configuration;

  /**
   * The USPS Rate Request class.
   *
   * @var \Drupal\commerce_usps\USPSRateRequest
   */
  protected USPSRateRequest $rateRequest;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    global $_USPS_ACCESS_TOKEN_;
    $this->setConfig();

    // Mock all the objects and set the config.
    $logger = $this->prophesize(LoggerInterface::class);
    $cache_backend = $this->prophesize(CacheBackendInterface::class);
    $state = $this->createMock(StateInterface::class);
    $event_dispatcher = new EventDispatcher();
    $usps_sdk = new USPSSdk($cache_backend->reveal(), $logger->reveal(), $event_dispatcher);
    $sdk_factory = new USPSSdkFactory(
      new ClientFactory(HandlerStack::create()),
      HandlerStack::create(),
      $state,
      $usps_sdk
    );

    // Get access token for requests.
    $config = $this->configuration['api_information'];
    $token_key = USPSSdkFactoryInterface::TOKEN_KEY . '.' . md5($config['client_id'] . $config['secret']);
    if (empty($_USPS_ACCESS_TOKEN_)) {
      $client = $sdk_factory->get($config);
      $response = $client->getAccessToken();
      if ($response->getStatusCode() == '200') {
        $token_data = Json::decode($response->getBody()->getContents());
        $_USPS_ACCESS_TOKEN_ = [
          'token' => $token_data['access_token'],
          'type' => $token_data['token_type'],
          'expires' => time() + (int) $token_data['expires_in'],
        ];
      }
    }

    if (!empty($_USPS_ACCESS_TOKEN_)) {
      $state->method('get')->with($token_key, FALSE)->willReturn($_USPS_ACCESS_TOKEN_);
    }

    $usd_currency = $this->prophesize(CurrencyInterface::class);
    $usd_currency->id()->willReturn('USD');
    $usd_currency->getFractionDigits()->willReturn('2');

    $storage = $this->prophesize(EntityStorageInterface::class);
    $storage->load('USD')->willReturn($usd_currency->reveal());

    $entity_type_manager = $this->prophesize(EntityTypeManagerInterface::class);
    $entity_type_manager->getStorage('commerce_currency')->willReturn($storage->reveal());
    $rounder = new Rounder($entity_type_manager->reveal());
    $this->rateRequest = new USPSRateRequest($sdk_factory, $rounder, $logger->reveal());
  }

  /**
   * Tests getRates().
   *
   * @covers ::getRates
   */
  public function testGetRates() {
    $shipment = $this->mockShipment();
    $shipping_method_plugin = $this->mockShippingMethod();
    $shipping_method = $this->prophesize(ShippingMethodInterface::class);
    $shipping_method->id()->willReturn('123456789');
    $shipping_method->getPlugin()->willReturn($shipping_method_plugin);

    // Fetch rates from the USPS api.
    $this->rateRequest->setConfig($this->getConfig());
    $rates = $this->rateRequest->getRates($shipment, $shipping_method->reveal());

    // Make sure at least one rate was returned.
    $this->assertArrayHasKey(0, $rates);

    /** @var \Drupal\commerce_shipping\ShippingRate $rate */
    foreach ($rates as $rate) {
      $this->assertInstanceOf(ShippingRate::class, $rate);
      $this->assertNotEmpty($rate->getAmount()->getNumber());
      $this->assertNotEmpty($rate->getService()->getLabel());
      $this->assertEquals('123456789', $rate->getShippingMethodId());
    }
  }

  /**
   * Test getRates() with commercial rate response.
   *
   * @throws \Exception
   */
  public function testCommercialRates() {
    $shipping_method_id = '123456789';
    $shipment = $this->mockShipment();
    $shipping_method_plugin = $this->mockShippingMethod();
    $shipping_method = $this->prophesize(ShippingMethodInterface::class);
    $shipping_method->id()->willReturn($shipping_method_id);
    $shipping_method->getPlugin()->willReturn($shipping_method_plugin);
    $config = $this->getConfig();

    // Pass the config to the rate and shipment services.
    $this->rateRequest->setConfig($config);
    $retail_rates = $this->rateRequest->getRates($shipment, $shipping_method->reveal());

    // Get commercial rates.
    $config['rate_options']['price_type'] = 'commercial';
    $this->rateRequest->setConfig($config);
    $commercial_rates = $this->rateRequest->getRates($shipment, $shipping_method->reveal());

    // Make sure both return rates.
    $this->assertGreaterThanOrEqual(count($retail_rates), count($commercial_rates));

    foreach ($commercial_rates as $commercial_rate) {
      $this->assertInstanceOf(ShippingRate::class, $commercial_rate);
      $sku = str_replace(sprintf('%s--', $shipping_method_id), '', $commercial_rate->getId());
      $sku = sprintf('%sR%s', substr($sku, 0, -6), substr($sku, strlen($sku) - 5));
      foreach ($retail_rates as $index => $retail_rate) {
        $this->assertInstanceOf(ShippingRate::class, $retail_rate);
        $retail_sku = str_replace(sprintf('%s--', $shipping_method_id), '', $retail_rate->getId());
        if ($retail_sku === $sku) {
          // Ensure the commercial rate is less or equal to the retail rate.
          $this->assertTrue($retail_rate->getAmount()->greaterThanOrEqual($commercial_rate->getAmount()));
          unset($retail_rates[$index]);
          break;
        }
      }
    }
  }

  /**
   * Test getRates() with filtered rates.
   */
  public function testFilteredRates(string $property_key, array $property_values, ?string $price_type = NULL) {
    $shipment = $this->mockShipment();
    $shipping_method_plugin = $this->mockShippingMethod();
    $shipping_method = $this->prophesize(ShippingMethodInterface::class);
    $shipping_method->id()->willReturn('123456789');
    $shipping_method->getPlugin()->willReturn($shipping_method_plugin);
    $config = $this->getConfig();
    if ($price_type) {
      $config['rate_options']['price_type'] = $price_type;
    }

    // Get all rates without any filtration.
    $this->rateRequest->setConfig($config);
    $rates = $this->rateRequest->getRates($shipment, $shipping_method->reveal());
    $this->assertNotEmpty($rates);

    // Get filtered rates.
    $config['rate_options'][$property_key] = $property_values;
    $this->rateRequest->setConfig($config);
    $filtered_rates = $this->rateRequest->getRates($shipment, $shipping_method->reveal());
    $this->assertNotEmpty($filtered_rates);

    $this->assertGreaterThan(count($filtered_rates), count($rates));
  }

  /**
   * Mocks the configuration array for tests.
   *
   * @param array $config
   *   The shipping method plugin configuration.
   */
  protected function setConfig(array $config = []) {
    $defaults = [
      'api_information' => [
        // cspell:disable
        'client_id' => 'VAEvJu3rGDjuIZGeks7rjB2iJoxgvhZzJGKfnR97ed0BAJZ9',
        'secret' => 'lw6gIna9u8vmmhk7xvZ7WdVAB4pSDrGsgOAT7bopIw3aumOgJmtys6qbqxnxQOjG',
        // cspell:enable
        'mode' => 'test',
      ],
      'rate_options' => [
        'price_type' => 'retail',
        'categories' => [],
        'facility_types' => [],
        'rate_indicators' => [],
      ],
      'services' => $this->getListOfServices(),
    ];

    $this->configuration = array_replace_recursive($defaults, $config);
  }

  /**
   * Get the configuration array.
   *
   * @return array
   *   The config.
   */
  protected function getConfig(): array {
    return $this->configuration;
  }

  /**
   * Creates a mock Drupal Commerce shipment entity.
   *
   * @param array $weight
   *   A weight array keyed by weight and unit.
   * @param array $dimensions
   *   A dimensions array keyed by length, width, height, and unit.
   *
   * @return \Drupal\commerce_shipping\Entity\ShipmentInterface
   *   A mocked commerce shipment object.
   */
  public function mockShipment(array $weight = [], array $dimensions = []) {
    // Ensure default values for weight and dimensions.
    $weight = $weight + [
      'weight' => 10,
      'unit' => 'lb',
    ];

    $dimensions = $dimensions + [
      'length' => 10,
      'width' => 3,
      'height' => 10,
      'unit' => 'in',
    ];

    // Mock a Drupal Commerce Order and associated objects.
    $order = $this->prophesize(OrderInterface::class);
    $store = $this->prophesize(StoreInterface::class);
    $store->getAddress()->willReturn(new Address('US', 'NC', 'Asheville', '', 28806, '', '1025 Brevard Rd'));
    $order->getStore()->willReturn($store->reveal());

    // Mock a Drupal Commerce shipment and associated objects.
    $shipment = $this->prophesize(ShipmentInterface::class);
    $profile = $this->prophesize(ProfileInterface::class);
    $address_list = $this->prophesize(FieldItemListInterface::class);

    // Mock the address list to return a US address.
    $address_list->first()->willReturn($this->getDeliveryAddress());
    $profile->get('address')->willReturn($address_list->reveal());
    $shipment->getShippingProfile()->willReturn($profile->reveal());
    $shipment->getOrder()->willReturn($order->reveal());

    // Mock a package type including dimensions and remote id.
    $package_type = $this->prophesize(PackageTypeInterface::class);
    $package_type->getHeight()->willReturn((new Length($dimensions['height'], 'in'))->convert($dimensions['unit']));
    $package_type->getLength()->willReturn((new Length($dimensions['length'], 'in'))->convert($dimensions['unit']));
    $package_type->getWidth()->willReturn((new Length($dimensions['width'], 'in'))->convert($dimensions['unit']));
    $package_type->getRemoteId()->willReturn('custom');

    // Mock the shipments weight and package type.
    $shipment->getWeight()->willReturn((new Weight($weight['weight'], 'lb'))->convert($weight['unit']));
    $shipment->getPackageType()->willReturn($package_type->reveal());

    // Return the mocked shipment object.
    return $shipment->reveal();
  }

  /**
   * Creates and returns a mock Drupal Commerce shipping method.
   */
  protected function mockShippingMethod(): ShipmentMethodPluginInterface {
    $shipping_method = $this->prophesize(USPSBase::class);
    $package_type = $this->prophesize(PackageTypeInterface::class);
    $package_type->getHeight()->willReturn(new Length(10, 'in'));
    $package_type->getLength()->willReturn(new Length(10, 'in'));
    $package_type->getWidth()->willReturn(new Length(3, 'in'));
    $package_type->getWeight()->willReturn(new Weight(10, 'lb'));
    $shipping_method->getDefaultPackageType()->willReturn($package_type);
    $shipping_method->getMailClass()->willReturn($this->getMailClass());
    $shipping_method->getFacilityTypeOptions()->willReturn([]);
    $shipping_method->getRateIndicatorOptions()->willReturn([]);
    $shipping_method->getPluginId()->willReturn($this->getPluginId());

    return $shipping_method->reveal();
  }

  /**
   * Returns the "mailClass" property for the rate request.
   */
  abstract protected function getMailClass(): string;

  /**
   * Returns the shipping address.
   */
  abstract protected function getDeliveryAddress(): Address;

  /**
   * Returns the shipping method plugin ID.
   */
  abstract protected function getPluginId(): string;

  /**
   * Returns the list of enabled services.
   */
  abstract protected function getListOfServices(): array;

}
