<?php

namespace Drupal\commerce_vicb_mia\Plugin\Commerce\PaymentGateway;

use Drupal\commerce_order\Entity\OrderInterface;
use Drupal\commerce_payment\Annotation\CommercePaymentGateway;
use Drupal\commerce_payment\Entity\PaymentInterface;
use Drupal\commerce_payment\Entity\PaymentMethodInterface;
use Drupal\commerce_payment\Exception\PaymentGatewayException;
use Drupal\commerce_payment\PaymentMethodStorageInterface;
use Drupal\commerce_payment\Plugin\Commerce\PaymentGateway\OffsitePaymentGatewayBase;
use Drupal\commerce_price\Price;
use Drupal\commerce_vicb_mia\VicbMiaQrCodeCheckoutOrderManager;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;

/**
 * Provides the Vicb Mia payment gateway.
 *
 * @CommercePaymentGateway(
 *   id = "vicb_mia_qr_code",
 *   label = @Translation("Victoria Bank Mia Qr Code"),
 *   display_label = @Translation("Victoria Bank Mia Qr Code"),
 *   modes = {
 *     "test" = @Translation("Sandbox"),
 *     "live" = @Translation("Live"),
 *   },
 *   payment_method_types = {"vicb_mia_qr_code"},
 *   payment_type = "vicb_mia_qr_code",
 *   requires_billing_information = FALSE
 * )
 *
 */
class VicbMiaQrCodePaymentGateway extends OffsitePaymentGatewayBase implements VicbMiaQrCodePaymentGatewayInterface {

  private VicbMiaQrCodeCheckoutOrderManager $vicbMiaQrCodeCheckoutOrderManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->vicbMiaQrCodeCheckoutOrderManager = $container->get('commerce_vicb_mia.qr_code.checkout_order_manager');
    return $instance;

  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
        'api_username' => NULL,
        'api_password' => NULL,
        'qr_code_timing' => NULL,
        'company_name' => NULL,
        'creditor_account' => NULL,
      ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $form = parent::buildConfigurationForm($form, $form_state);
    $form['credentials'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('API Credentials'),
      '#weight' => 0,
    ];
    $form['credentials']['api_username'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Username'),
      '#default_value' => $this->configuration['api_username'],
      '#maxlength' => 255,
      '#required' => TRUE,
      '#parents' => array_merge($form['#parents'], ['api_username']),
    ];
    $form['credentials']['api_password'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Password'),
      '#maxlength' => 255,
      '#default_value' => $this->configuration['api_password'],
      '#required' => TRUE,
      '#parents' => array_merge($form['#parents'], ['api_password']),
    ];
    $form['qr_code_configuration'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Configuration'),
      '#weight' => 1,
    ];
    $form['qr_code_configuration']['qr_code_timing'] = [
      '#type' => 'number',
      '#title' => $this->t('Minutes'),
      '#description' => $this->t("Specify the time duration (in minutes) for which the QR code remains valid"),
      '#step' => 1,
      '#min' => 1,
      '#max' => 1440,
      '#default_value' => $this->configuration['qr_code_timing'],
      '#required' => TRUE,
      '#parents' => array_merge($form['#parents'], ['qr_code_timing']),
    ];
    $form['qr_code_configuration']['company_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Company Name'),
      '#maxlength' => 255,
      '#default_value' => $this->configuration['company_name'],
      '#required' => TRUE,
      '#parents' => array_merge($form['#parents'], ['company_name']),
    ];
    $form['qr_code_configuration']['creditor_account'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Creditor Account'),
      '#maxlength' => 255,
      '#default_value' => $this->configuration['creditor_account'],
      '#required' => TRUE,
      '#parents' => array_merge($form['#parents'], ['creditor_account']),
    ];

    return $form;
  }

  /**
   * @inheritDoc
   */
  public function onReturn(OrderInterface $order, Request $request) {
    $vicb_mia_payload = $order->getData('vicb_mia_payload', []);

    $payment = $vicb_mia_payload['payment'] ?? [];
    $reference = $payment['reference'] ?? NULL;
    $amount = $payment['amount'] ?? NULL;
    if (is_null($reference) || is_null($amount)) {
      throw new PaymentGatewayException('Count fields reference or amount is null');
    }
    $payment_method = NULL;
    // If a payment method is already referenced by the order, no need to create
    // a new one.
    if (!$order->get('payment_method')->isEmpty()) {
      /** @var \Drupal\commerce_payment\Entity\PaymentMethodInterface $payment_method */
      $payment_method = $order->get('payment_method')->entity;
    }

    // If the order doesn't reference a payment method yet, or if the payment
    // method doesn't reference the right gateway, create a new one.
    if (!$payment_method || $payment_method->getPaymentGatewayId() !== $this->parentEntity->id()) {
      // Create a payment method.
      $payment_method_storage = $this->entityTypeManager->getStorage('commerce_payment_method');
      assert($payment_method_storage instanceof PaymentMethodStorageInterface);
      $payment_method = $payment_method_storage->createForCustomer(
        'vicb_mia_qr_code',
        $this->parentEntity->id(),
        $order->getCustomerId(),
        $order->getBillingProfile()
      );
    }
    [, , , $remoteId] = explode('|', $reference);
    $payment_method->setRemoteId($remoteId);
    $payment_method->setReusable(FALSE);
    $payment_method->save();
    $order->set('payment_method', $payment_method);


    $payment_storage = $this->entityTypeManager->getStorage('commerce_payment');
    /** @var \Drupal\commerce_payment\Entity\PaymentInterface $payment */
    $payment = $payment_storage->create([
      'state' => 'new',
      'amount' => $order->getBalance(),
      'payment_gateway' => $this->parentEntity->id(),
      'payment_method' => $payment_method->id(),
      'order_id' => $order->id(),
    ]);
    $payment->setRemoteId($reference);
    $this->createPayment($payment);

    parent::onReturn($order, $request);
  }

  /**
   * @inheritDoc
   */
  public function createPayment(PaymentInterface $payment, $capture = TRUE) {
    $this->assertPaymentState($payment, ['new']);
    $payment->state = 'completed';
    $payment->save();
  }

  /**
   * @inheritDoc
   */
  public function deletePaymentMethod(PaymentMethodInterface $payment_method) {
    $payment_method->delete();
  }


  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    parent::submitConfigurationForm($form, $form_state);
    if ($form_state->getErrors()) {
      return;
    }
    $values = $form_state->getValue($form['#parents']);

    $keys = [
      'api_username',
      'api_password',
      'qr_code_timing',
      'company_name',
      'creditor_account',
    ];

    foreach ($keys as $key) {
      if (!isset($values[$key])) {
        continue;
      }
      $this->configuration[$key] = $values[$key];
    }
  }

  public function refundPayment(PaymentInterface $payment, Price $amount = NULL) {

    $status = $this->vicbMiaQrCodeCheckoutOrderManager->reverseQrCode($this, $payment);
    if ($status) {
      $payment->state = 'refund';
      $payment->save();
    }
  }

  /**
   * @inheritDoc
   */
  public function apiUrl(): string {
    if ($this->getMode() === 'test') {
      return self::URL_TEST;
    }
    return self::URL_LIVE;
  }

  /**
   * @inheritDoc
   */
  public function apiUsername(): string {
    return $this->configuration['api_username'];
  }

  /**
   * @inheritDoc
   */
  public function apiPassword(): string {
    return $this->configuration['api_password'];
  }

  /**
   * @inheritDoc
   */
  public function qrCodeTiming(): int {
    return $this->configuration['qr_code_timing'];
  }

  /**
   * @inheritDoc
   */
  public function creditorAccount(): string {
    return $this->configuration['creditor_account'];
  }

  /**
   * @inheritDoc
   */
  public function companyName(): string {
    return $this->configuration['company_name'];
  }

}
