<?php

namespace Drupal\commerce_vicb_mia\Controller;

use Drupal\commerce_payment\Exception\PaymentGatewayException;
use Drupal\commerce_vicb_mia\Plugin\Commerce\PaymentGateway\VicbMiaQrCodePaymentGateway;
use Drupal\commerce_vicb_mia\VicbMiaQrCodeCheckoutOrderManager;
use Drupal\commerce_vicb_mia\VicbMiaQrCodeClient;
use Drupal\Core\Access\AccessException;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Url;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;

class CheckoutQrCodeCallbackController extends ControllerBase {

  private VicbMiaQrCodeCheckoutOrderManager $vicbMiaQrCodeCheckoutOrderManager;

  private LoggerInterface $logger;

  /**
   *
   * @param \Drupal\commerce_vicb_mia\VicbMiaQrCodeCheckoutOrderManager $vicbMiaQrCodeCheckoutOrderManager
   * @param \Psr\Log\LoggerInterface $logger
   */
  public function __construct(VicbMiaQrCodeCheckoutOrderManager $vicbMiaQrCodeCheckoutOrderManager, LoggerInterface $logger) {
    $this->vicbMiaQrCodeCheckoutOrderManager = $vicbMiaQrCodeCheckoutOrderManager;

    $this->logger = $logger;
  }

  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('commerce_vicb_mia.qr_code.checkout_order_manager'),
      $container->get('logger.factory')->get('commerce_vicb_mia'),
    );
  }


  public function index(Request $request) {
    $context = json_decode($request->getContent(), TRUE);

    if (!$context) {
      return new JsonResponse("No data");
    }

    $jwt = $context;

    if (commerce_vicb_mia_verify_jwt($jwt, '')) {
      return new JsonResponse("Invalid JWT Format");
    }

    $decoded = commerce_vicb_mia_decode_jwt($jwt);
    $payload = $decoded['payload'];
    $qrExtensionUUID = $payload['qrExtensionUUID'];
    $qrCodeHistory = $this->vicbMiaQrCodeCheckoutOrderManager->findByQrExtensionUUid($qrExtensionUUID);

    if (!$qrCodeHistory || $qrCodeHistory['qr_extension_uuid'] != $qrExtensionUUID) {
      return new JsonResponse('Unsupported payment gateway provided.');
    }
    $uid = $qrCodeHistory['order_id'];

    /** @var \Drupal\commerce_order\Entity\OrderInterface $order */
    $order = $this->entityTypeManager()->getStorage('commerce_order')->load($uid);
    if ($order->getState()->getId() != 'draft') {
      $this->logger->error('Order is not in draft state.');
      return new JsonResponse('Unsupported payment gateway provided.');
    }
    /** @var \Drupal\commerce_payment\Entity\PaymentGatewayInterface $payment_gateway */
    $payment_gateway = $order->get('payment_gateway')->entity;
    $payment_gateway_plugin = $payment_gateway->getPlugin();
    if (!$payment_gateway_plugin instanceof VicbMiaQrCodePaymentGateway) {
      return new JsonResponse('Unsupported payment gateway provided.');
    }

    $qrCodeStatus = $this->vicbMiaQrCodeCheckoutOrderManager->statusQrCodeByQrHeaderUuid($qrCodeHistory['qr_header_uuid'], $payment_gateway_plugin);

    if ($qrCodeStatus['status'] != VicbMiaQrCodeClient::STATUS_PAID) {
      return new JsonResponse('Qr code is not paid');
    }

    $order->setData('vicb_mia_payload', $payload);
    $order->setData('vicb_mia_status', $qrCodeStatus);

    try {
      $payment_gateway_plugin->onReturn($order, $request);
      $step_id = $order->get('checkout_step')->value;
      if (!empty($step_id)) {
        /** @var \Drupal\commerce_checkout\Entity\CheckoutFlowInterface $checkout_flow */
        $checkout_flow = $order->get('checkout_flow')->entity;
        $checkout_flow_plugin = $checkout_flow->getPlugin();
        if (!$checkout_flow_plugin->getOrder()) {
          $checkout_flow_plugin->setOrder($order);
        }
        $step_id = $checkout_flow_plugin->getNextStepId($step_id);
        $order->set('checkout_step', $step_id);

      }
      $order->save();

      return new JsonResponse('ok');
    } catch (PaymentGatewayException $e) {

      $this->logger->error($e->getMessage());
      $message = $this->t('Payment failed at the payment server. Please review your information and try again.');
      return new JsonResponse(['message' => $message], Response::HTTP_BAD_REQUEST);
    } catch (\Exception $exception) {

      $this->logger->error($exception->getMessage());
      $message = $this->t('Payment failed. Please review your information and try again.');
      return new JsonResponse(['message' => $message], Response::HTTP_BAD_REQUEST);
    }

  }

}
