<?php

namespace Drupal\commerce_vicb_mia\EventSubscriber;

use Drupal\commerce_cart\Event\CartEvents;
use Drupal\commerce_cart\Event\CartOrderItemAddEvent;
use Drupal\commerce_cart\Event\CartOrderItemUpdateEvent;
use Drupal\commerce_order\Entity\OrderInterface;
use Drupal\commerce_vicb_mia\Plugin\Commerce\PaymentGateway\VicbMiaQrCodePaymentGateway;
use Drupal\commerce_vicb_mia\VicbMiaQrCodeCheckoutOrderManager;
use Drupal\commerce_vicb_mia\VicbMiaQrCodeClient;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Subscribes to cart events for adding and updating order items.
 * This class provides event handling methods for cart operations such as
 * adding and updating order items in the shopping cart.
 * Implements the EventSubscriberInterface to define event subscriptions.
 */
class VicbMiaAddToCartSubscriber implements EventSubscriberInterface {

  /**
   *
   */
  private VicbMiaQrCodeCheckoutOrderManager $vicbMiaQrCodeCheckoutOrderManager;

  /**
   * Constructor method for initializing the VicbMiaQrCodeCheckoutOrderManager.
   *
   * @param VicbMiaQrCodeCheckoutOrderManager $vicbMiaQrCodeCheckoutOrderManager
   *     Instance of VicbMiaQrCodeCheckoutOrderManager.
   *
   * @return void
   */
  public function __construct(VicbMiaQrCodeCheckoutOrderManager $vicbMiaQrCodeCheckoutOrderManager) {
    $this->vicbMiaQrCodeCheckoutOrderManager = $vicbMiaQrCodeCheckoutOrderManager;
  }


  /**
   * Retrieves the events to which this class is subscribed and their corresponding handlers.
   *
   * @return array An associative array where the keys are event names and the values are arrays containing the handler method names and their priorities.
   */
  public static function getSubscribedEvents() {
    return [
      CartEvents::CART_ORDER_ITEM_ADD => ['onCartOrderItemAdd', 100],
      CartEvents::CART_ORDER_ITEM_UPDATE => ['onCartOrderItemUpdate', 100],
    ];
  }

  /**
   * Handles the event triggered when an order item is added to the cart.
   *
   * @param CartOrderItemAddEvent $event
   *   The event object containing information about the cart and the added order item.
   *
   * @return void
   */
  public function onCartOrderItemAdd(CartOrderItemAddEvent $event) {
    $order = $event->getCart();
    $this->deleteMiaQrCode($order);
  }

  /**
   * Handles the cart order item update event.
   *
   * @param CartOrderItemUpdateEvent $event
   *   The event triggered when an order item in the cart is updated.
   *
   * @return void
   */
  public function onCartOrderItemUpdate(CartOrderItemUpdateEvent $event) {
    $order = $event->getCart();
    $this->deleteMiaQrCode($order);
  }

  /**
   * Deletes the Mia QR code associated with the given order.
   *
   * @param OrderInterface $order
   *   The order entity from which the Mia QR code will be deleted.
   *
   * @return void
   */
  private function deleteMiaQrCode(OrderInterface $order): void {

    /** @var \Drupal\commerce_payment\Entity\PaymentGatewayInterface $payment_gateway */
    $payment_gateway = $order->get('payment_gateway')->entity;

    $payment_gateway_plugin = $payment_gateway?->getPlugin();

    // Ensure that is draft and is not locked.
    if ($payment_gateway_plugin instanceof VicbMiaQrCodePaymentGateway && $order->getState()->getId() === 'draft' && !$order->isLocked()) {

      $this->vicbMiaQrCodeCheckoutOrderManager->deleteQrCode($order, $payment_gateway_plugin);
    }
  }

}
