<?php

namespace Drupal\commerce_vicb_mia;

use Drupal\commerce_order\Entity\OrderInterface;
use Drupal\commerce_payment\Entity\PaymentInterface;
use Drupal\commerce_vicb_mia\Plugin\Commerce\PaymentGateway\VicbMiaQrCodePaymentGatewayInterface;
use Drupal\Core\Database\Connection;

class VicbMiaQrCodeCheckoutOrderManager {

  private ?VicbMiaQrCodeClient $vicbMiaQrCodeClient = NULL;

  private Connection $connection;

  public function __construct(Connection $connection) {
    $this->connection = $connection;
  }

  /**
   * Generates a QR code for the given order using the specified payment gateway.
   *
   * @param OrderInterface $order The order object for which the QR code should be generated.
   * @param VicbMiaQrCodePaymentGatewayInterface $payment_gateway The payment gateway used to generate the QR code.
   *
   * @return array The generated QR code data.
   */
  public function generateQrCode(OrderInterface $order, VicbMiaQrCodePaymentGatewayInterface $payment_gateway): array {
    $vicbMiaQrCodeData = $order->getData('vicb_mia_qr_code');
    if ($order->isLocked()) {
      return $vicbMiaQrCodeData;
    }
    $data = [
      "header" => [
        "qrType" => "DYNM",
        "amountType" => "Fixed",
        "pmtContext" => "o",
      ],
      "extension" => [
        "creditorAccount" => [
          "iban" => $payment_gateway->creditorAccount(),
        ],
        "amount" => [
          "sum" => $order->getTotalPrice()->getNumber(),
          "currency" => "MDL",
        ],

        "dba" => $payment_gateway->companyName(),
        "remittanceInfo4Payer" => "Commerce order: " . $order->id(),
        "creditorRef" => "order:" . $order->id(),
        "ttl" => [
          "length" => $payment_gateway->qrCodeTiming(),
          "units" => "mm",
        ],
      ],
    ];


    // Exist code, always revalidate
    $vicbMiaQrCodeData = $this->vicbMiaQrCodeClient($payment_gateway)->generateQrCode($data);
    $this->insertQrCodeHistory($vicbMiaQrCodeData, $order);
    $order->setData('vicb_mia_qr_code', $vicbMiaQrCodeData);
    $order->save();
    return $vicbMiaQrCodeData;
  }

  public function vicbMiaQrCodeClient($payment_gateway): VicbMiaQrCodeClient {
    if (!$this->vicbMiaQrCodeClient) {
      $this->vicbMiaQrCodeClient = new VicbMiaQrCodeClient($payment_gateway->apiUrl(), $payment_gateway->apiUsername(), $payment_gateway->apiPassword());
    }
    return $this->vicbMiaQrCodeClient;
  }

  public function statusQrCode(OrderInterface $order, $payment_gateway_plugin): ?array {
    $vicbMiaQrCodeData = $order->getData('vicb_mia_qr_code');
    if (!$vicbMiaQrCodeData) {
      return NULL;
    }
    $qrHeaderUUID = $vicbMiaQrCodeData['qrHeaderUUID'] ?? NULL;
    if (!$qrHeaderUUID) {
      return NULL;
    }
    return $this->vicbMiaQrCodeClient($payment_gateway_plugin)->statusQrCode($qrHeaderUUID);
  }

  public function statusQrCodeByQrHeaderUuid($qrHeaderUUID, $payment_gateway_plugin): ?array {
    return $this->vicbMiaQrCodeClient($payment_gateway_plugin)->statusQrCode($qrHeaderUUID);
  }

  /**
   * Deletes a QR code associated with the given order.
   *
   * @param OrderInterface $order The order object containing QR code data.
   * @param mixed $payment_gateway_plugin The payment gateway plugin instance needed to perform the deletion.
   *
   * @return void
   */
  public function deleteQrCode(OrderInterface $order, $payment_gateway_plugin): void {
    $vicbMiaQrCodeData = $order->getData('vicb_mia_qr_code');
    if (!$vicbMiaQrCodeData) {
      return;
    }
    $qrHeaderUUID = $vicbMiaQrCodeData['qrHeaderUUID'] ?? NULL;
    if (!$qrHeaderUUID) {
      return;
    }
    $this->vicbMiaQrCodeClient($payment_gateway_plugin)->deleteQrCode($qrHeaderUUID);
  }


  public function isPaidQrCode(OrderInterface $order, $payment_gateway_plugin): bool {
    $status = $this->statusQrCode($order, $payment_gateway_plugin);

    if (!$status) {
      return FALSE;
    }

    return $status['status'] === VicbMiaQrCodeClient::STATUS_PAID;
  }


  public function reverseQrCode($payment_gateway_plugin, PaymentInterface $payment): bool {
    return $this->vicbMiaQrCodeClient($payment_gateway_plugin)->revers($payment->getRemoteId());
  }

  private function insertQrCodeHistory(array $vicbMiaQrCodeData, OrderInterface $order) {
    // Check if qrHeaderUUID already exists.
    $exists = $this->connection->select('commerce_vicb_mia_qr_codes', 'q')
      ->fields('q', ['qr_header_uuid'])
      ->condition('qr_header_uuid', $vicbMiaQrCodeData['qrHeaderUUID'])
      ->range(0, 1)
      ->execute()
      ->fetchField();

    if (!$exists) {
      $this->connection->insert('commerce_vicb_mia_qr_codes')
        ->fields([
          'qr_header_uuid' => $vicbMiaQrCodeData['qrHeaderUUID'],
          'qr_extension_uuid' => $vicbMiaQrCodeData['qrExtensionUUID'],
          'order_id' => $order->id(),
          'created' => \Drupal::time()->getCurrentTime(),
        ])
        ->execute();
    }
  }

  public function findByQrExtensionUUid($qrExtensionUUID) {
    return $this->connection->select('commerce_vicb_mia_qr_codes', 'q')
      ->fields('q')
      ->condition('qr_extension_uuid', $qrExtensionUUID)
      ->range(0, 1)
      ->execute()
      ->fetchAssoc();
  }

}
