<?php

namespace Drupal\commerce_vicb_mia;

use Drupal\Core\State\StateInterface;
use GuzzleHttp\Client;

class VicbMiaQrCodeClient {

  const STATUS_PAID = 'Paid';

  const STATUS_EXPIRED = 'Expired';

  const STATUS_CANCELLED = 'Cancelled';

  const STATUS_ACTIVE = 'Active';

  private string $apiUrl;

  private string $apiUsername;

  private string $apiPassword;

  private ?StateInterface $state = NULL;

  private ?Client $httpClient = NULL;

  public function __construct($url, $user, $password) {
    $this->apiUrl = $url;
    $this->apiUsername = $user;
    $this->apiPassword = $password;
  }

  private function makeTokenRequest(): array {
    $response = $this->httpClient()->post($this->apiUrl . 'identity/token', [
      'form_params' => [
        'grant_type' => 'password',
        'username' => $this->apiUsername,
        'password' => $this->apiPassword,
      ],
    ]);
    $data = json_decode($response->getBody()->getContents(), TRUE);
    if (isset($data['errorCode'])) {
      throw new \Exception($data['description']);
    }
    return $data;
  }

  private function getStoredToken(): ?array {
    return $this->state()->get('vicb_mia.token');
  }

  private function storeToken(array $token): void {
    $this->state()->set('vicb_mia.token', $token);
  }

  private function isTokenExpired(?array $token): bool {
    if (!$token) {
      return TRUE;
    }
    return ($token['expiresIn'] + $token['created']) < (time() + 3600);
  }

  public function accessToken(): string {
    $token = $this->getStoredToken();
    if ($this->isTokenExpired($token)) {
      $token = $this->makeTokenRequest();
      $token['created'] = time();
      $this->storeToken($token);
    }

    return $token['accessToken'];
  }

  public function deleteQrCode($qrHeaderUUID): void {
    $accessToken = $this->accessToken();
    try {
      $this->httpClient()->delete($this->apiUrl . 'api/v1/qr/' . $qrHeaderUUID, [
        'headers' => [
          'Authorization' => 'Bearer ' . $accessToken,
        ],
      ]);
    } catch (\Exception $e) {
      // Do nothing
    }
  }

  /**
   * @param $data
   *
   * @return array
   * @throws \GuzzleHttp\Exception\GuzzleException
   */
  public function generateQrCode($data): array {
    $accessToken = $this->accessToken();

    $response = $this->httpClient()->post($this->apiUrl . 'api/v1/qr', [
      'headers' => [
        'Authorization' => 'Bearer ' . $accessToken,
      ],
      'json' => $data,
    ]);

    $data = json_decode($response->getBody()->getContents(), TRUE);

    if (isset($data['errorCode'])) {
      throw new \Exception($data['description']);
    }
    return $data;
  }

  public function statusQrCode($qrHeaderUUID) {
    $accessToken = $this->accessToken();
    $response = $this->httpClient()->get($this->apiUrl . 'api/v1/qr/' . $qrHeaderUUID . '/status', [
      'headers' => [
        'Authorization' => 'Bearer ' . $accessToken,
      ],
    ]);
    $data = json_decode($response->getBody()->getContents(), TRUE);
    if (isset($data['errorCode'])) {
      throw new \Exception($data['description']);
    }
    return $data;
  }

  private function state() {
    if (!$this->state) {
      $this->state = \Drupal::state();
    }
    return $this->state;
  }

  private function httpClient() {
    if (!$this->httpClient) {
      $this->httpClient = \Drupal::httpClient();
    }
    return $this->httpClient;
  }

  public function revers($reference) {
    $accessToken = $this->accessToken();
    $this->httpClient()->delete($this->apiUrl . 'api/v1/transaction/' . $reference, [
      'headers' => [
        'Authorization' => 'Bearer ' . $accessToken,
      ],
    ]);
    if (isset($data['errorCode'])) {
      return FALSE;
    }
    return TRUE;
  }

  public function getReconciliationTransactions($dateFrom, $dateTo){
    $accessToken = $this->accessToken();
    $response = $this->httpClient()->get($this->apiUrl . 'api/v1/reconciliation/transactions', [
      'headers' => [
        'Authorization' => 'Bearer ' . $accessToken,
      ],
      'query' => [
        'dateFrom' => $dateFrom,
        'dateTo' => $dateTo,
      ],
    ]);
    $data = json_decode($response->getBody()->getContents(), TRUE);
    if (isset($data['errorCode'])) {
      throw new \Exception($data['description']);
    }
    return $data;
  }


}
