<?php declare(strict_types=1);

namespace Drupal\commerce_view_receipt\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Access\AccessResult;
use Drupal\commerce_order\Entity\OrderInterface;
use Drupal\commerce_order\OrderTotalSummaryInterface;
use Drupal\user\UserInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Receipt Controller.
 */
final class ReceiptController extends ControllerBase {

  /**
   * Commerce Order total Summary.
   *
   * @var \Drupal\commerce_order\OrderTotalSummaryInterface
   */
  protected $orderTotalSummary;

  /**
   * Constructor.
   */
  public function __construct(OrderTotalSummaryInterface $order_total_summary) {
    $this->orderTotalSummary = $order_total_summary;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('commerce_order.order_total_summary')
    );
  }

  /**
   * View receipt in the browser.
   */
  public function viewReceipt(OrderInterface $commerce_order) {
    $billing_profile = $commerce_order->getBillingProfile();
    return [
      '#theme' => 'commerce_order_receipt',
      '#order_entity' => $commerce_order,
      '#billing_information' => $billing_profile ? $this
        ->entityTypeManager()
        ->getViewBuilder('profile')
        ->view($billing_profile) : NULL,
      '#totals' => $this
        ->orderTotalSummary
        ->buildTotals($commerce_order),
    ];
  }

  /**
   * Access check.
   * Ensure route parameters {user} matches {commerce_order} owner.
   */
  public function userAccess(UserInterface $user, OrderInterface $commerce_order) {
    return AccessResult::allowedIf($user->id() === $commerce_order->getCustomer()->id());
  }

}
