<?php

namespace Drupal\commerce_wstack\Controller;

use Drupal\commerce_wstack\PaymentSdk\Model\TokenVersion;
use Drupal\commerce_wstack\EventSubscriber\WebhookEvent;
use Drupal\commerce_wstack\Plugin\Commerce\PaymentGateway\OffsiteRedirect;
use Drupal\commerce_wstack\Services\PaymentSdk;
use Drupal\commerce_payment\Entity\PaymentMethod;
use Drupal\commerce_payment\Entity\EntityWithPaymentGatewayInterface;
use Drupal\commerce_payment\Entity\PaymentGatewayInterface;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Psr\Log\LoggerInterface;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException;

/**
 * Class WebhookController for payments.
 */
class WebhookController extends ControllerBase {

  /**
   * The logger.
   *
   * @var \Psr\Log\LoggerInterface
   */
  private LoggerInterface $logger;

  /**
   * The payment SDK.
   *
   * @var \Drupal\commerce_wstack\Services\PaymentSdk
   */
  protected PaymentSdk $paymentSdk;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The event dispatcher.
   *
   * @var \Symfony\Component\EventDispatcher\EventDispatcherInterface
   */
  protected $eventDispatcher;

  /**
   * Constructor.
   *
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger.
   * @param \Drupal\commerce_wstack\Services\PaymentSdk $paymentSdk
   *   The payment SDK.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Symfony\Component\EventDispatcher\EventDispatcherInterface $event_dispatcher
   *   The event dispatcher.
   */
  public function __construct(LoggerInterface $logger, PaymentSdk $paymentSdk, EntityTypeManagerInterface $entity_type_manager, EventDispatcherInterface $event_dispatcher) {
    $this->logger = $logger;
    $this->paymentSdk = $paymentSdk;
    $this->entityTypeManager = $entity_type_manager;
    $this->eventDispatcher = $event_dispatcher;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('logger.channel.commerce_wstack'),
      $container->get('commerce_wstack.payment_sdk'),
      $container->get('entity_type.manager'),
      $container->get('event_dispatcher'));
  }

  /**
   * Handles the webhook request.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   The response for payments.
   */
  public function content(Request $request): Response {
    // Get data from webhook.
    $data_post = $request->getContent();
    
    // Check if webhook URL is in use.
    if (!$data_post) {
      $this->logger->warning('Webhook request is empty.');
      throw new AccessDeniedHttpException('No payload');
    }
    else {
      $this->logger->info('request.content: ' . $data_post);
    }

    // Decode data.
    $data = json_decode($data_post, TRUE);
    if (isset($data['listenerEntityTechnicalName']) && isset($data['entityId']) && isset($data['spaceId'])) {

      $event = new WebhookEvent($data);
      $this->eventDispatcher->dispatch($event);
      if ($event->isProcessed()) {
        return new Response('OK');
      }

      // Get the relevant local entity based on the remote id.
      switch ($data['listenerEntityTechnicalName']) {
        case 'TokenVersion':
          $payment_gateways = $this->entityTypeManager()->getStorage('commerce_payment_gateway')
            ->loadByProperties([
            'configuration.space_id' => $data['spaceId'],
            'plugin' => 'commerce_wstack_offsite_redirect',
          ]);

          // It is possible to have multiple gateways with the same space id,
          // that should not be a problem as this is just needed to get the
          // token.
          if (count($payment_gateways) > 0 and current($payment_gateways) instanceof PaymentGatewayInterface) {
            $payment_gateway = current($payment_gateways);

            try {
              // Get token version.
              $version = $this->paymentSdk->getTokenVersion($payment_gateway, $data['entityId']);
            }
            catch (\Exception $e) {
              throw new AccessDeniedHttpException('No active version.');
            }

            if (!$version instanceof TokenVersion) {
              throw new AccessDeniedHttpException('No active version.');
            }

            $entity = $this->loadLocalEntity('commerce_payment_method', $version->getToken()->getId());
            if ($entity instanceof PaymentMethod) {
              $this->paymentSdk->webhookTokenVersion($entity->getPaymentGateway(), $data, $version);
            }
          }
          break;

        case 'Token':
          // In order of security reasons we only process tokens which exists on our system.
          // We don't add new tokens over webhook (not possible to be sure if user is the right one).
          $entity = $this->loadLocalEntity('commerce_payment_method', $data['entityId']);
          if ($entity instanceof PaymentMethod) {
            $this->paymentSdk->webhookToken($entity->getPaymentGateway(), $data);
          }
          break;

        case 'Transaction':
          // @todo is used to wait for the transaction to be created.
          sleep(20);
          $entity = $this->loadLocalEntity('commerce_payment', $data['entityId']);
          if ($entity instanceof Payment) {
            $this->paymentSdk->webhookTransaction($entity->getPaymentGateway(), $data);
          }
          break;
      }
    }
    else {
      throw new AccessDeniedHttpException('Missing parameters.');
    }

    $response = new Response();
    $response->setContent('Ok');
    return $response;
  }

  /**
   * Loads the local entity that matches the remote id.
   *
   * @param string $listener_name
   *   The listener technical name as specified in backoffice.
   * @param int $remote_id
   *   The remote id.
   *
   * @return \Drupal\commerce_payment\Entity\EntityWithPaymentGatewayInterface
   *   The local entity.
   */
  protected function loadLocalEntity(string $entity_type, int $remote_id) {
    /** @var \Drupal\commerce_payment\Entity\EntityWithPaymentGatewayInterface[] $entity_candidates */
    $entity_candidates = $this->entityTypeManager()
      ->getStorage($entity_type)
      ->loadByProperties(['remote_id' => $remote_id]);

    foreach ($entity_candidates as $entity) {
      if ($entity->getPaymentGateway()->getPlugin() instanceof OffsiteRedirect) {
        return $entity;
      }
    }

    return FALSE;
  }

}
