<?php

namespace Drupal\commerce_wstack\Plugin\Commerce\CheckoutPane;

use Drupal\commerce_payment\Plugin\Commerce\CheckoutPane\PaymentInformation as BasePaymentInformation;
use Drupal\commerce_payment\Entity\PaymentGateway;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Component\Utility\NestedArray;
use Drupal\address\AddressInterface;
use Drupal\profile\Entity\ProfileInterface;

/**
 * Provides a custom payment information pane.
 */
class PaymentInformation extends BasePaymentInformation {

  /**
   * AJAX callback: refreshes the pane form.
   */
  public static function ajaxRefresh(array $form, FormStateInterface $form_state) {
    $parents = $form_state->getTriggeringElement()['#parents'];
    array_pop($parents);
    return NestedArray::getValue($form, $parents);
  }

  /**
   * {@inheritdoc}
   */
  public function buildPaneForm(array $pane_form, FormStateInterface $form_state, array &$complete_form) {
    $pane_form = parent::buildPaneForm($pane_form, $form_state, $complete_form);

    // Get payment information value only for authenticated users.
    if ($this->order->getCustomerId() == 0) {
      return $pane_form;
    }

    // Get payment method.
    $payment_method = '';
    if (isset($pane_form['payment_method']['#default_value'])) {
      $payment_method = $pane_form['payment_method']['#default_value'];
    }

    $payment_information = $form_state->getValue('payment_information');
    if (isset($payment_information['payment_method'])) {
      $payment_method = $payment_information['payment_method'];
    }

    if ($payment_method != '') {
      // Load payment method configuration.
      $payment_gateway = \Drupal::entityTypeManager()
        ->getStorage('commerce_payment_gateway')
        ->load($payment_method);
      if ($payment_gateway and $payment_gateway instanceof PaymentGateway) {
        $plugin_configuration = $payment_gateway->getPluginConfiguration();

        // Add billing address fields for payment gateway.
        if ($payment_gateway->getPluginId() === 'commerce_wstack_offsite_redirect') {
          /** @var \Drupal\profile\ProfileStorageInterface $profile_storage */
          $profile_storage = \Drupal::entityTypeManager()->getStorage('profile');

          $pane_form['billing_address'] = [
            '#type' => 'fieldset',
            '#title' => $this->t('Billing Address'),
            '#weight' => 10,
          ];

          // Get customer's stored addresses.
          $customer = $this->order->getCustomer();
          $stored_addresses = [];
          $default_address = 'new';
          if (!$customer->isAnonymous()) {
            $profiles = $profile_storage->loadMultipleByUser($customer, 'customer');
            if (count($profiles) > 0 and current($profiles) instanceof ProfileInterface) {
              foreach ($profiles as $profile) {
                $address = $profile->get('address')->first();
                if ($address) {
                  $stored_addresses[$profile->id()] = $address->getAddressLine1() . ', ' . $address->getLocality();
                  // Use the first address as default if we haven't set one yet.
                  if ($default_address === 'new') {
                    $default_address = $profile->id();
                  }
                }
              }
            }
          }

          // Add "Enter a new address" option.
          $stored_addresses['new'] = $this->t('+ Enter a new address');

          $pane_form['billing_address']['address_selection'] = [
            '#type' => 'select',
            '#title' => $this->t('Select an address'),
            '#options' => $stored_addresses,
            '#default_value' => $default_address,
          ];

          // Display area for selected stored address.
          $pane_form['billing_address']['selected_address_display'] = [
            '#type' => 'container',
            '#attributes' => ['class' => ['selected-address-display']],
            '#states' => [
              'visible' => [
                ':input[name="payment_information[billing_address][address_selection]"]' => ['!value' => 'new'],
              ],
            ],
          ];

          // Add address details display for each stored address.
          if (!$customer->isAnonymous()) {
            $profiles = $profile_storage->loadMultipleByUser($customer, 'customer');
            if (count($profiles) > 0 and current($profiles) instanceof ProfileInterface) {
              foreach ($profiles as $profile) {
                $address = $profile->get('address')->first();
                if ($address) {
                  $pane_form['billing_address']['selected_address_display'][$profile->id()] = [
                    '#type' => 'container',
                    '#attributes' => ['class' => ['address-details']],
                    '#states' => [
                      'visible' => [
                        ':input[name="payment_information[billing_address][address_selection]"]' => ['value' => $profile->id()],
                      ],
                    ],
                    'address_info' => [
                      '#type' => 'html_tag',
                      '#tag' => 'div',
                      '#attributes' => ['class' => ['address-info']],
                      '#value' => $this->formatAddressDisplay($address),
                    ],
                  ];
                }
              }
            }
          }

          // Container for address fields.
          $pane_form['billing_address']['address_fields'] = [
            '#type' => 'container',
            '#attributes' => ['id' => 'billing-address-fields'],
            '#states' => [
              'visible' => [
                ':input[name="payment_information[billing_address][address_selection]"]' => ['value' => 'new'],
              ],
            ],
          ];

          // Show address fields by default (for new address).
          $pane_form['billing_address']['address_fields']['country_code'] = [
            '#type' => 'select',
            '#title' => $this->t('Land'),
            '#options' => $this->getCountryOptions(),
            '#required' => FALSE,
            '#states' => [
              'required' => [
                ':input[name="payment_information[billing_address][address_selection]"]' => ['value' => 'new'],
              ],
            ],
          ];

          $pane_form['billing_address']['address_fields']['company'] = [
            '#type' => 'textfield',
            '#title' => $this->t('Company'),
            '#maxlength' => 255,
            '#required' => FALSE,
          ];

          $pane_form['billing_address']['address_fields']['given_name'] = [
            '#type' => 'textfield',
            '#title' => $this->t('First name'),
            '#required' => FALSE,
            '#maxlength' => 255,
            '#states' => [
              'required' => [
                ':input[name="payment_information[billing_address][address_selection]"]' => ['value' => 'new'],
              ],
            ],
          ];

          $pane_form['billing_address']['address_fields']['family_name'] = [
            '#type' => 'textfield',
            '#title' => $this->t('Last name'),
            '#required' => FALSE,
            '#maxlength' => 255,
            '#states' => [
              'required' => [
                ':input[name="payment_information[billing_address][address_selection]"]' => ['value' => 'new'],
              ],
            ],
          ];

          $pane_form['billing_address']['address_fields']['address_line1'] = [
            '#type' => 'textfield',
            '#title' => $this->t('Street address'),
            '#required' => FALSE,
            '#maxlength' => 255,
            '#states' => [
              'required' => [
                ':input[name="payment_information[billing_address][address_selection]"]' => ['value' => 'new'],
              ],
            ],
          ];

          $pane_form['billing_address']['address_fields']['address_line2'] = [
            '#type' => 'textfield',
            '#title' => $this->t('Street address line 2'),
            '#maxlength' => 255,
            '#required' => FALSE,
          ];

          $pane_form['billing_address']['address_fields']['postal_code'] = [
            '#type' => 'textfield',
            '#title' => $this->t('Postal code'),
            '#required' => FALSE,
            '#maxlength' => 20,
            '#states' => [
              'required' => [
                ':input[name="payment_information[billing_address][address_selection]"]' => ['value' => 'new'],
              ],
            ],
          ];

          $pane_form['billing_address']['address_fields']['locality'] = [
            '#type' => 'textfield',
            '#title' => $this->t('City'),
            '#required' => FALSE,
            '#maxlength' => 255,
            '#states' => [
              'required' => [
                ':input[name="payment_information[billing_address][address_selection]"]' => ['value' => 'new'],
              ],
            ],
          ];
        }

        if (isset($plugin_configuration['payment_reusable']) && $plugin_configuration['payment_reusable'] == 'ask') {
          // Check if user has set a value on last checkout.
          $tempstore = \Drupal::service('tempstore.private');
          $store = $tempstore->get('commerce_wstack');

          $pane_form['save_payment'] = [
            '#type' => 'checkbox',
            '#title' => $this->t('Save payment information'),
            '#description' => $this->t('Only for Visa, Master and Paypal'),
            '#default_value' => $store->get('save_payment') ?? FALSE,
          ];
        }
      }
    }

    return $pane_form;
  }

  /**
   * Get country options for the billing address form.
   *
   * @return array
   *   Array of country options.
   */
  protected function getCountryOptions() {
    // Get the current store's supported billing countries.
    $store = $this->order->getStore();
    $supported_countries = [];

    if ($store && $store->hasField('billing_countries')) {
      $billing_countries = $store->get('billing_countries')->getValue();
      foreach ($billing_countries as $country_item) {
        $supported_countries[] = $country_item['value'];
      }
    }

    // If no supported countries configured, fall back to all countries.
    if (empty($supported_countries)) {
      $countries = \Drupal::service('address.country_repository')->getList();
      return $countries;
    }

    // Get only the supported countries.
    $country_repository = \Drupal::service('address.country_repository');
    $all_countries = $country_repository->getList();
    $filtered_countries = [];

    foreach ($supported_countries as $country_code) {
      if (isset($all_countries[$country_code])) {
        $filtered_countries[$country_code] = $all_countries[$country_code];
      }
    }

    return $filtered_countries;
  }

  /**
   * {@inheritdoc}
   */
  public function submitPaneForm(array &$pane_form, FormStateInterface $form_state, array &$complete_form) {
    parent::submitPaneForm($pane_form, $form_state, $complete_form);
    $payment_information = $form_state->getValue('payment_information');

    // Check if save payment information is checked.
    if (isset($payment_information['save_payment'])) {
      $save_payment = FALSE;

      if ($payment_information['save_payment']) {
        $save_payment = TRUE;
        $tempstore = \Drupal::service('tempstore.private');
        $store = $tempstore->get('commerce_wstack');
        $store->set('save_payment', TRUE);
      }

      $this->order->setData('commerce_wstack_save_payment_method', $save_payment);
      $this->order->save();
    }

    // Handle billing address submission for payment gateway.
    if (isset($payment_information['payment_method'])) {
      $payment_gateway = \Drupal::entityTypeManager()
        ->getStorage('commerce_payment_gateway')
        ->load($payment_information['payment_method']);

      if ($payment_gateway instanceof PaymentGateway and $payment_gateway->getPluginId() === 'commerce_wstack_offsite_redirect') {
        $billing_address = $form_state->getValue(['payment_information', 'billing_address']);

        // Check if new address is selected.
        if ($billing_address && $billing_address['address_selection'] === 'new') {
          // Save billing address to user's address book.
          $profile = $this->saveBillingAddress($billing_address['address_fields']);
        }
        else {
          // User selected stored address.
          $profile_storage = \Drupal::entityTypeManager()->getStorage('profile');
          $profile = $profile_storage->load($billing_address['address_selection']);
        }

        // Do a copy from the profile.
        \Drupal::service('commerce_order.address_book')->copy($profile, $this->order->getCustomer());

        // Add profile to order (commerce change this to uid 0).
        $this->order->setBillingProfile($profile);
      }
    }
  }

  /**
   * Save billing address to user's address book.
   *
   * @param array $address_data
   *   The address data to save.
   */
  protected function saveBillingAddress(array $address_data) {
    $customer = $this->order->getCustomer();
    if ($customer->isAnonymous()) {
      return FALSE;
    }

    try {
      /** @var \Drupal\profile\ProfileStorageInterface $profile_storage */
      $profile_storage = \Drupal::entityTypeManager()->getStorage('profile');

      // Create profile.
      $profile = $profile_storage->create([
        'type' => 'customer',
        'uid' => $customer->id(),
        'address' => [
          'address_line1' => $address_data['address_line1'] ?? '',
          'address_line2' => $address_data['address_line2'] ?? '',
          'locality' => $address_data['locality'] ?? '',
          'postal_code' => $address_data['postal_code'] ?? '',
          'country_code' => $address_data['country_code'] ?? '',
          'given_name' => $address_data['given_name'] ?? '',
          'family_name' => $address_data['family_name'] ?? '',
          'organization' => $address_data['company'] ?? '',
        ],
        'is_default' => FALSE,
      ]);

      // Save the profile.
      $profile->save();

      return $profile;
    }
    catch (\Exception $e) {
      \Drupal::logger('commerce_wstack')->error('Failed to save billing address: @error', ['@error' => $e->getMessage()]);
    }

    return FALSE;
  }

  /**
   * Formats an address for display.
   *
   * @param \Drupal\address\AddressInterface $address
   *   The address object.
   *
   * @return string
   *   The formatted address string.
   */
  protected function formatAddressDisplay(AddressInterface $address) {
    $address_parts = [];

    $name = '';
    if ($address->getGivenName()) {
      $name .= $address->getGivenName();
    }
    if ($address->getFamilyName()) {
      $name .= ' ' . $address->getFamilyName();
    }
    if ($name) {
      $address_parts[] = $name;
    }

    if ($address->getOrganization()) {
      $address_parts[] = $address->getOrganization();
    }

    if ($address->getAddressLine1()) {
      $address_parts[] = $address->getAddressLine1();
    }

    if ($address->getAddressLine2()) {
      $address_parts[] = $address->getAddressLine2();
    }

    $city = '';
    if ($address->getPostalCode()) {
      $city .= $address->getPostalCode();
    }
    if ($address->getLocality()) {
      $city .= ' ' . $address->getLocality();
    }
    if ($address->getCountryCode()) {
      $city .= ' (' . $address->getCountryCode() . ')';
    }
    if ($city) {
      $address_parts[] = $city;
    }

    return implode('<br>', array_filter($address_parts));
  }

}
