<?php

namespace Drupal\commerce_wstack\Plugin\Commerce\PaymentGateway;

use Drupal\commerce_wstack\PluginForm\OffsiteRedirect\PaymentOffsiteForm;
use Drupal\commerce_wstack\PaymentSdk\ApiClient;
use Drupal\commerce_wstack\PaymentSdk\ApiException;
use Drupal\commerce_wstack\PaymentSdk\Model\EntityQuery;
use Drupal\commerce_wstack\PaymentSdk\Service\PaymentMethodConfigurationService;
use Drupal\commerce_wstack\PaymentSdk\Model\Token;
use Drupal\commerce_payment\Entity\PaymentGateway;
use Drupal\commerce_payment\Entity\PaymentInterface;
use Drupal\commerce_price\Price;
use Drupal\commerce_order\Entity\OrderInterface;
use Drupal\commerce_payment\Entity\PaymentMethodInterface;
use Drupal\commerce_payment\Attribute\CommercePaymentGateway;
use Drupal\commerce_payment\Exception\PaymentGatewayException;
use Drupal\commerce_payment\Plugin\Commerce\PaymentGateway\OffsitePaymentGatewayBase;
use Drupal\commerce_payment\Plugin\Commerce\PaymentGateway\SupportsStoredPaymentMethodsInterface;
use Drupal\commerce_payment\Plugin\Commerce\PaymentGateway\SupportsRefundsInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides the Off-site Redirect payment gateway.
 */
#[CommercePaymentGateway(
  id: "commerce_wstack_offsite_redirect",
  label: new TranslatableMarkup("WStack"),
  display_label: new TranslatableMarkup("WStack"),
  forms: [
    "offsite-payment" => PaymentOffsiteForm::class,
  ],
  payment_method_types: ["wtype"],
  requires_billing_information: FALSE,
)]
class OffsiteRedirect extends OffsitePaymentGatewayBase implements SupportsStoredPaymentMethodsInterface, SupportsRefundsInterface {

  /**
   * The payment sdk.
   *
   * @var \Drupal\commerce_wstack\Services\PaymentSdk
   */
  private $paymentSdk;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The minor units converter.
   *
   * @var \Drupal\commerce_price\MinorUnitsConverter
   */
  protected $minorUnitsConverter;

  /**
   * The logger channel factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  private $loggerChannelFactory;

  /**
   * The event dispatcher.
   *
   * @var \Symfony\Component\EventDispatcher\EventDispatcherInterface
   */
  private $eventDispatcher;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);

    $instance->paymentSdk = $container->get('commerce_wstack.payment_sdk');
    $instance->entityTypeManager = $container->get('entity_type.manager');
    $instance->loggerChannelFactory = $container->get('logger.factory');
    $instance->eventDispatcher = $container->get('event_dispatcher');

    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getLibraries(): array {
    return [];
  }

  /**
   * {@inheritdoc}
   */
  public function collectsBillingInformation() {
    // Disable billing collection to allow stored payment methods to work.
    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function getDisplayLabel() {
    return $this->configuration['display_label'] ?? $this->t('Payment Gateway');
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return parent::defaultConfiguration() + [
      'space_id' => '',
      'user_id' => '',
      'secret' => '',
      'payment_configuration' => 'all',
      'payment_reusable' => 'ask',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $form = parent::buildConfigurationForm($form, $form_state);

    $route_name = \Drupal::routeMatch()->getRouteName();

    $form['collect_billing_information']['#access'] = FALSE;

    $form['space_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('API Space ID'),
      '#description' => $this->t('This is the space id from the PHP SDK.'),
      '#default_value' => $this->configuration['space_id'],
      '#required' => TRUE,
    ];

    $form['user_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('API User ID'),
      '#description' => $this->t('The user id from the PHP SDK.'),
      '#default_value' => $this->configuration['user_id'],
      '#required' => TRUE,
    ];

    $form['secret'] = [
      '#type' => 'textfield',
      '#title' => $this->t('API Secret'),
      '#description' => $this->t('The secret from the PHP SDK.'),
      '#default_value' => $this->configuration['secret'],
      '#required' => TRUE,
    ];

    // Check if gateway is setup complete.
    if (isset($this->configuration['user_id']) and $this->configuration['user_id'] != '') {
      // Get all payment methods.
      $methods = [];
      $methods['all'] = 'All';

      // Call PSP API for payment methods.
      try {
        $client = new ApiClient($this->configuration['user_id'], $this->configuration['secret']);
        $configuration = new PaymentMethodConfigurationService($client);
        $query = new EntityQuery();
        $query->setLanguage('en');
        $configurations = $configuration->search($this->configuration['space_id'], $query);
        foreach ($configurations as $config) {
          $methods[$config->getId()] = $config->getName();
        }
      }
      catch (ApiException $e) {
        $this->messenger()->addError($this->t('Error on Communication with PSP, please check your settings.'));
      }

      if (!empty($methods)) {
        // List all payment types they are possible from PSP.
        $form['payment_configuration'] = [
          '#type' => 'radios',
          '#options' => $methods,
          '#title' => $this->t('Payment Types'),
          '#description' => $this->t('If no selected we will use all.'),
          '#default_value' => $this->configuration['payment_configuration'],
        ];
      }

      // Allow user to save token.
      $form['payment_reusable'] = [
        '#type' => 'select',
        '#title' => $this->t('Request token and create reusable payment method'),
        '#description' => $this->t('Allows to reuse the payment information for additional payments. PSP allows right now only Visa, Master and Paypal.'),
        '#default_value' => $this->configuration['payment_reusable'],
        '#options' => [
          'always' => $this->t('Always, payment methods are created for logged in users only'),
          'ask' => $this->t('Ask the user'),
          'never' => $this->t('Never'),
        ],
      ];

      // Webhook configuration.
      $webhook_url = $this->createWebhookUrl();
      if ($this->configuration['webhook_url'] != '' and $route_name != 'entity.commerce_payment_gateway.duplicate_form') {
        $webhook_url = $this->configuration['webhook_url'];
      }
      $form['webhook_url'] = [
        '#type' => 'textfield',
        '#title' => $this->t('Webhook URL'),
        '#description' => $this->t('Please setup webhooks as described bellow on PSP Backend.'),
        '#default_value' => $webhook_url,
        '#attributes' => ['readonly' => 'readonly'],
        '#required' => TRUE,
      ];

      // Info.
      $webhook_info = [];
      $webhook_info[] = '- Token (name it "Token")';
      $webhook_info[] = '- Token Version (name it "TokenVersion")';
      $webhook_info[] = '- Transaction (name it "Transaction" with states "Completed" and "Failed")';

      $form['webhook_url_info'] = [
        '#markup' => '<b>' . $this->t('You need to setup following webhooks in PSP Backend') . '</b><br>' . implode('<br>', $webhook_info),
      ];

      // Create a new button to migrate payment methods.
      $form['migrate_payment_methods_info'] = [
        '#markup' => '<div class="form-item"><b>' . $this->t('You can migrate the old payment methods (from version 2) to our new payment method entity.') . '</b>',
      ];
      $form['migrate_payment_methods'] = [
        '#type' => 'submit',
        '#prefix' => '<div>',
        '#suffix' => '</div></div>',
        '#value' => $this->t('Migrate payment methods'),
        '#submit' => ['\Drupal\commerce_wstack\Plugin\Commerce\PaymentGateway\OffsiteRedirect::submitPaymentMethodForm'],
      ];
    }
    else {
      $form['payment_configuration'] = [
        '#markup' => '<b>' . $this->t('After first saving, you can edit the settings again and set the payment methods you want.') . '</b>',
      ];
    }

    return $form;
  }

  /**
   * Migrate payment methods.
   */
  public static function submitPaymentMethodForm(array &$form, FormStateInterface $form_state) {
    $values = $form_state->getValue($form['#parents']);
    if (isset($values['id'])) {
      $service = \Drupal::service('commerce_wstack.payment_sdk');
      $service->migratePaymentMethods($values['id']);

      // Output message.
      \Drupal::messenger()->addStatus(t('Payment methods migrated successfully.'));
    }
  }

  /**
   * Create the webhook URL for this gateway.
   */
  private function createWebhookUrl() {
    $current_page = parse_url((isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https' : 'http') . '://' . $_SERVER['HTTP_HOST']);
    return $current_page['scheme'] . '://' . $current_page['host'] . '/commerce_wstack/webhook?gate=' . md5(random_bytes(50));
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    parent::submitConfigurationForm($form, $form_state);

    $values = $form_state->getValue($form['#parents']);
    $this->configuration['type'] = 'commerce_wstack';
    $this->configuration['space_id'] = $values['space_id'];
    $this->configuration['user_id'] = $values['user_id'];
    $this->configuration['secret'] = $values['secret'];
    $this->configuration['payment_reusable'] = ($values['payment_reusable'] ?? FALSE);
    $this->configuration['webhook_url'] = ($values['webhook_url'] ?? '');
    $this->configuration['webhook_url_gate'] = (isset($values['webhook_url']) ? explode('?gate=', $values['webhook_url'])[1] : '');
    $this->configuration['payment_configuration'] = ($values['payment_configuration'] ?? 'all');
  }

  /**
   * {@inheritdoc}
   */
  public function onReturn(OrderInterface $order, Request $request) {
    parent::onReturn($order, $request);

    // If order is free only create the payment method but not the payment.
    if ($order->getBalance()->isZero()) {
      return;
    }

    /** @var \Drupal\commerce_payment\PaymentStorage $paymentStorage */
    $paymentStorage = $this->entityTypeManager->getStorage('commerce_payment');
    $payments = $paymentStorage->loadMultipleByOrder($order);

    $success = FALSE;
    foreach ($payments as $payment) {
      // Check if gateway is paymentgateway entity.
      if ($payment->payment_gateway->entity instanceof PaymentGateway) {
        $configuration = $payment->getPaymentGateway()->getPluginConfiguration();
        if (isset($configuration['type']) and $configuration['type'] == 'commerce_wstack') {
          $transaction_id = $payment->getRemoteId();
          $transation = $this->paymentSdk->readTransaction($configuration['user_id'], $configuration['secret'], $configuration['space_id'], $transaction_id);

          // Check state of transaction.
          if (isset($transation['state'])) {
            // Transform PSP state.
            $state = $this->paymentSdk->transformTransactionState($transation['state']);
            if ($state != NULL and $state != 'void') {
              $success = TRUE;
            }

            // Update payment.
            if ($state != NULL) {
              $payment->set('state', $state);
              $payment->set('remote_state', $state);
              $payment->save();
            }
          }

          // Check token on transaction.
          if (isset($transation['token']) and $transation['token'] instanceof Token) {
            // Anonymous cannot have tokens.
            if ($order->getCustomerId() != 0) {
              $user = $order->getCustomer();
              $billing_profile = $order->getBillingProfile() ?? NULL;

              // Create payment method.
              $this->paymentSdk->createPaymentMethod($payment->getPaymentGateway(), $transation['token'], $user, $billing_profile);
            }
          }
        }
      }
    }

    if ($success == FALSE) {
      // Error on payment.
      throw new PaymentGatewayException('Payment incomplete or declined');
    }
  }

  /**
   * {@inheritdoc}
   */
  public function onCancel(OrderInterface $order, Request $request) {
    parent::onCancel($order, $request);
  }

  /**
   * {@inheritdoc}
   */
  public function onNotify(Request $request) {
    parent::onNotify($request);
  }

  /**
   * {@inheritdoc}
   */
  public function canRefundPayment(PaymentInterface $payment) {
    // Check if the payment is in a state that allows refunds.
    $refundable_states = ['completed', 'partially_refunded'];
    return in_array($payment->getState()->value, $refundable_states);
  }

  /**
   * {@inheritdoc}
   */
  public function refundPayment(PaymentInterface $payment, ?Price $amount = NULL) {
    $this->assertPaymentState($payment, ['completed', 'partially_refunded']);

    // If not specified, refund the entire amount.
    $amount = $amount ?: $payment->getAmount();

    // Validate the requested amount.
    $this->assertRefundAmount($payment, $amount);

    $old_refunded_amount = $payment->getRefundedAmount();
    $new_refunded_amount = $old_refunded_amount->add($amount);
    if ($new_refunded_amount->lessThan($payment->getAmount())) {
      $payment->state = 'partially_refunded';
    }
    else {
      $payment->state = 'refunded';
    }

    // Check if gateway is paymentgateway entity.
    if ($payment->payment_gateway->entity instanceof PaymentGateway) {
      // Call remote refund on PSP.
      $configuration = $payment->getPaymentGateway()->getPluginConfiguration();
      try {
        $refund = $this->paymentSdk->createRefund($configuration['user_id'], $configuration['secret'], $configuration['space_id'], $amount->getNumber(), $payment->getRemoteId());
        if ($refund->getState() == 'SUCCESSFUL') {
          // Save refund.
          $payment->setRefundedAmount($new_refunded_amount);
          $payment->save();

          $this->messenger()->addStatus($this->t('Refund on payment %payment', [
            '%payment' => $payment->id(),
          ]));
        }
        else {
          // Error on refund.
          $this->messenger()->addError($this->t('Error refund on payment %payment', [
            '%payment' => $payment->id(),
          ]));
        }
      }
      catch (\Exception $e) {
        throw new PaymentGatewayException($e->getMessage());
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function createPayment(PaymentInterface $payment, $capture = TRUE) {
    // Check if gateway is paymentgateway entity.
    if ($payment->payment_gateway->entity instanceof PaymentGateway) {
      // Get plugin configuration.
      $configuration = $payment->payment_gateway->entity->getPluginConfiguration();

      // Payment data.
      $currency = $payment->getAmount()->getCurrencyCode();
      $amount = $payment->getAmount()->getNumber();

      /** @var \Drupal\commerce_order\Entity\OrderInterface $order */
      $order = $payment->getOrder();
      $profiles = $order->collectProfiles();
      $mail = $order->getEmail();

      // Create addresses.
      $billingAddress = NULL;
      if (isset($profiles['billing'])) {
        $billingAddress = $this->paymentSdk->createAddress($profiles['billing'], $order);
      }

      // Create addresses.
      $shippingAddress = NULL;
      if (isset($profiles['shipping'])) {
        $shippingAddress = $this->paymentSdk->createAddress($profiles['shipping'], $order);
      }
      $token = $payment->payment_method->entity->remote_id->value;

      // Set environment.
      if ($configuration['mode'] == 'test') {
        $environment = 'PREVIEW';
      }
      else {
        $environment = 'LIVE';
      }

      // Create transaction with no interaction.
      $transaction = $this->paymentSdk->createTransactionNoUserInteraction($configuration['user_id'], $configuration['secret'], $configuration['space_id'], $currency, $order, $amount, $mail, $billingAddress, $shippingAddress, $token, $environment);

      // Error handling for transaction.
      if (!isset($transaction['id']) or $transaction['id'] == '') {
        $this->messenger()->addError($this->t('Error on payment gateway, no transaction found.'));
        throw new PaymentGatewayException('Error on payment gateway, no transaction found.');
      }
      if (!isset($transaction['state']) or $transaction['state'] == '') {
        $this->messenger()->addError($this->t('Error on payment gateway, no transaction found.'));
        throw new PaymentGatewayException('Error on payment gateway, no transaction found.');
      }
      if ($transaction['state'] == 'FAILED') {
        $message = 'Error on payment gateway, failure on transaction.';

        // Check if token is disabled.
        if (isset($transaction['token'])) {
          if ($transaction['token']->getState() == 'INACTIVE') {
            $message = 'Error on payment gateway, payment token is inactive.';
          }
        }

        $this->messenger()->addError($this->t($message));
        throw new PaymentGatewayException($message);
      }

      // Check state of transaction.
      if (isset($transaction['state'])) {
        // Transform PSP state.
        $state = $this->paymentSdk->transformTransactionState($transaction['state']);

        // Update payment.
        if ($state != NULL) {
          $payment->set('remote_id', $transaction['id']);
          $payment->set('state', $state);
          $payment->set('remote_state', $state);
          $payment->save();
        }
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function deletePaymentMethod(PaymentMethodInterface $payment_method) {
    // Delete token on PSP backend.
    $this->paymentSdk->deletePaymentMethod($payment_method);
    $payment_method->delete();
  }

  /**
   * {@inheritdoc}
   */
  public function updatePaymentMethod(PaymentMethodInterface $payment_method) {
    // @todo not working right now on PSP.
  }

  /**
   * Converts the given amount to its minor units.
   *
   * For example, 9.99 USD becomes 999.
   *
   * @param \Drupal\commerce_price\Price $amount
   *   The amount.
   *
   * @return int
   *   The amount in minor units, as an integer.
   */
  public function toMinorUnits(Price $amount): int {
    return $this->minorUnitsConverter->toMinorUnits($amount);
  }

}
