<?php

namespace Drupal\commerce_wstack\Services;

use Psr\Log\LoggerInterface;
use Drupal\commerce_payment\Entity\PaymentMethod;
use Drupal\commerce_payment\Entity\PaymentGateway;
use Drupal\commerce_payment\Exception\PaymentGatewayException;
use Drupal\commerce_order\Entity\OrderInterface;
use Drupal\commerce_payment\Entity\PaymentInterface;
use Drupal\commerce_payment\Entity\PaymentMethodInterface;
use Drupal\commerce_payment\Entity\PaymentGatewayInterface;
use Drupal\profile\Entity\ProfileInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\TempStore\PrivateTempStoreFactory;
use Drupal\Core\Session\AccountProxy;

use Drupal\commerce_wstack\PaymentSdk\ApiClient;
use Drupal\commerce_wstack\PaymentSdk\Model\TokenVersion;
use Drupal\commerce_wstack\PaymentSdk\Model\Transaction;
use Drupal\commerce_wstack\PaymentSdk\Model\Refund;
use Drupal\commerce_wstack\PaymentSdk\Model\Token;
use Drupal\commerce_wstack\PaymentSdk\Model\TokenizationMode;
use Drupal\commerce_wstack\PaymentSdk\Model\TransactionCreate;
use Drupal\commerce_wstack\PaymentSdk\Model\LineItemType;
use Drupal\commerce_wstack\PaymentSdk\Model\LineItemCreate;
use Drupal\commerce_wstack\PaymentSdk\Model\AddressCreate;
use Drupal\commerce_wstack\PaymentSdk\Service\TokenService;
use Drupal\commerce_wstack\PaymentSdk\Service\TokenVersionService;
use Drupal\commerce_wstack\PaymentSdk\Service\TransactionService;

/**
 * Event handler for Payment SDK.
 */
class PaymentSdk {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The private tempstore service.
   *
   * @var \Drupal\Core\TempStore\PrivateTempStoreFactory
   */
  protected $privateTempStore;

  /**
   * The current user account.
   *
   * @var \Drupal\Core\Session\AccountProxy
   */
  protected $currentUser;

  /**
   * The logger service.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactory
   */
  protected $logger;

  /**
   * PaymentSdk constructor.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\TempStore\PrivateTempStoreFactory $privateTempStore
   *   The private tempstore service.
   * @param \Drupal\Core\Session\AccountProxy $currentUser
   *   The current user account.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger service.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager, PrivateTempStoreFactory $privateTempStore, AccountProxy $currentUser, LoggerInterface $logger) {
    $this->entityTypeManager = $entity_type_manager;
    $this->privateTempStore = $privateTempStore;
    $this->currentUser = $currentUser;
    $this->logger = $logger;
  }

  /**
   * Create Payment SDK Client.
   *
   * @param string $user_id
   *   The user id.
   * @param string $secret
   *   The secret.
   *
   * @return \Drupal\commerce_wstack\PaymentSdk\ApiClient
   *   The client.
   */
  public function apiClient(string $user_id, string $secret) {
    return new ApiClient($user_id, $secret);
  }

  /**
   * Create Payment line item.
   *
   * @param string $order_id
   *   The order id.
   * @param string $amount
   *   The amount.
   *
   * @return \Drupal\commerce_wstack\PaymentSdk\LineItemCreate
   *   The line item.
   */
  private function createLineItem(string $order_id, string $amount) {
    $lineItem = new LineItemCreate();
    $lineItem->setName('Order ' . $order_id);
    $lineItem->setUniqueId('orderid-' . $order_id);
    $lineItem->setQuantity(1);
    $lineItem->setAmountIncludingTax($amount);
    $lineItem->setType(LineItemType::FEE);

    return $lineItem;
  }

  /**
   * {@inheritdoc}
   */
  public function createAddress(ProfileInterface $profile, OrderInterface $order) {
    $payment_address = new AddressCreate();

    /** @var \Drupal\address\AddressInterface $address */
    $address = $profile->hasField('address') ? $profile->get('address')->first() : NULL;
    if ($address) {
      $payment_address->setOrganizationName($address->getOrganization());
      $payment_address->setFamilyName($address->getFamilyName());
      $payment_address->setGivenName($address->getGivenName());
      $payment_address->setStreet($address->getAddressLine1());
      $payment_address->setPostCode($address->getPostalCode());
      $payment_address->setCity($address->getLocality());
      $payment_address->setCountry($address->getCountryCode());
    }
    $payment_address->setEmailAddress($order->getEmail());

    // Allow alter address over hook.
    \Drupal::moduleHandler()->invokeAll('commerce_wstack_alter_address', [&$payment_address, $profile, $order]);

    return $payment_address;
  }

  /**
   * Create Payment transaction.
   *
   * @param string $user_id
   *   The user id.
   * @param string $secret
   *   The secret.
   * @param string $space_id
   *   The space id.
   * @param string $return_url
   *   The return url.
   * @param string $cancel_url
   *   The cancel url.
   * @param string $currency
   *   The currency.
   * @param string $order_id
   *   The order id.
   * @param string $amount
   *   The amount.
   * @param string $mail
   *   The mail.
   * @param Drupal\commerce_wstack\PaymentSdk\Model\AddressCreate|null $billingAddress
   *   The billing address.
   * @param Drupal\commerce_wstack\PaymentSdk\Model\AddressCreate|null $shippingAddress
   *   The shipping address.
   * @param string $environment
   *   The environment.
   * @param array $pluginConfiguration
   *   The plugin configuration.
   *
   * @return array
   *   The transaction.
   *
   * @throws \Drupal\commerce_wstack\PaymentSdk\ApiException
   * @throws \Drupal\commerce_wstack\PaymentSdk\Http\ConnectionException
   * @throws \Drupal\commerce_wstack\PaymentSdk\VersioningException
   */
  public function createTransaction(string $user_id, string $secret, string $space_id, string $return_url, string $cancel_url, string $currency, $order_id, $amount, $mail, ?AddressCreate $billingAddress, ?AddressCreate $shippingAddress, string $environment = 'PREVIEW', array $pluginConfiguration = []) {
    // Set user language.
    $language = \Drupal::languageManager()->getCurrentLanguage()->getId();
    $language_string = strtolower($language) . '-' . strtoupper($language);

    $transactionPayload = new TransactionCreate();
    $transactionPayload->setMerchantReference($order_id);
    $transactionPayload->setLanguage($language_string);
    $transactionPayload->setSuccessUrl($return_url);
    $transactionPayload->setFailedUrl($cancel_url);
    $transactionPayload->setEnvironment($environment);
    if ($billingAddress) {
      $transactionPayload->setBillingAddress($billingAddress);
    }
    if ($shippingAddress) {
      $transactionPayload->setShippingAddress($shippingAddress);
    }
    $transactionPayload->setCurrency($currency);
    $transactionPayload->setLineItems([$this->createLineItem($order_id, $amount)]);
    $transactionPayload->setAutoConfirmationEnabled(TRUE);

    if (isset($pluginConfiguration['payment_configuration']) and $pluginConfiguration['payment_configuration'] != 'all') {
      $transactionPayload->setAllowedPaymentMethodConfigurations($pluginConfiguration['payment_configuration']);
    }

    if ($this->shouldRequestToken($pluginConfiguration)) {
      // Force token generation on payment.
      $transactionPayload->setTokenizationMode(TokenizationMode::FORCE_CREATION_WITH_ONE_CLICK_PAYMENT);
      $transactionPayload->setCustomerEmailAddress($mail);
      if ($this->currentUser->id() != 0) {
        $transactionPayload->setCustomerId($this->currentUser->id());
      }
    }

    // Allow alter transaction over hook.
    \Drupal::moduleHandler()->invokeAll('commerce_wstack_alter_transaction', [&$transactionPayload, $order_id]);

    // Create transaction.
    $client = $this->apiClient($user_id, $secret);
    $transaction = $client->getTransactionService()->create($space_id, $transactionPayload);

    return [
      'transaction_id' => $transaction->getId(),
      'redirect_url' => $client->getTransactionPaymentPageService()->paymentPageUrl($space_id, $transaction->getId()),
    ];
  }

  /**
   * Create Payment transaction with no user interaction.
   *
   * @param string $user_id
   *   The user id.
   * @param string $secret
   *   The secret.
   * @param string $space_id
   *   The space id.
   * @param string $currency
   *   The currency.
   * @param Drupal\commerce_order\Entity\OrderInterface $order
   *   The order.
   * @param string $amount
   *   The amount.
   * @param string $mail
   *   The mail.
   * @param Drupal\commerce_wstack\PaymentSdk\Model\AddressCreate|null $billingAddress
   *   The billing address.
   * @param Drupal\commerce_wstack\PaymentSdk\Model\AddressCreate|null $shippingAddress
   *   The shipping address.
   * @param string $token
   *   The token.
   * @param string $environment
   *   The environment.
   *
   * @return array
   *   The transaction id and the redirect url or throw an exception if token is not active.
   *
   * @throws \Drupal\commerce_wstack\PaymentSdk\ApiException
   * @throws \Drupal\commerce_wstack\PaymentSdk\Http\ConnectionException
   * @throws \Drupal\commerce_wstack\PaymentSdk\VersioningException
   */
  public function createTransactionNoUserInteraction($user_id, $secret, $space_id, $currency, OrderInterface $order, $amount, $mail, ?AddressCreate $billingAddress, ?AddressCreate $shippingAddress, $token, $environment = 'PREVIEW') {
    // Check if token is valid.
    $client = $this->apiClient($user_id, $secret);
    $tokenVersionService = new TokenVersionService($client);
    $version = $tokenVersionService->activeVersion($space_id, $token);
    if ($version instanceof TokenVersion) {
      if ($version->getState() != 'ACTIVE') {
        // Allow alter token transaction over hook.
        \Drupal::moduleHandler()->invokeAll('commerce_wstack_alter_transaction_token', [$space_id, $order, $token, $version]);

        $this->logger->error('Token version is not active: @token (user_id: @user_id)', ['@token' => $token, '@user_id' => $user_id]);
        throw new PaymentGatewayException('Token version is not active: ' . $token);
      }
    }

    // Prepare transaction payload.
    $transactionPayload = new TransactionCreate();
    $transactionPayload->setMerchantReference($order->id());
    $transactionPayload->setEnvironment($environment);
    if ($billingAddress) {
      $transactionPayload->setBillingAddress($billingAddress);
    }
    if ($shippingAddress) {
      $transactionPayload->setShippingAddress($shippingAddress);
    }
    $transactionPayload->setCurrency($currency);
    $transactionPayload->setLineItems([$this->createLineItem($order->id(), $amount)]);
    $transactionPayload->setAutoConfirmationEnabled(TRUE);

    // Set token to transaction.
    $transactionPayload->setToken($token);
    $transactionPayload->setCustomerEmailAddress($mail);

    try {
      // Create transaction.
      $client = $this->apiClient($user_id, $secret);
      $transaction = $client->getTransactionService()->create($space_id, $transactionPayload);
      $transaction_service = new TransactionService($client);
      $transaction_return = $transaction_service->processWithoutUserInteraction($space_id, $transaction->getId());
    }
    catch (\Exception $e) {
      throw new PaymentGatewayException($e->getMessage());
    }

    return [
      'id' => $transaction_return->getId(),
      'authorization_amount' => $transaction_return->getAuthorizationAmount(),
      'authorization_environment' => $transaction_return->getAuthorizationEnvironment(),
      'authorized_on' => $transaction_return->getAuthorizedOn(),
      'completed_amount' => $transaction_return->getCompletedAmount(),
      'completed_on' => $transaction_return->getCompletedOn(),
      'currency' => $transaction_return->getCurrency(),
      'state' => $transaction_return->getState(),
      'token' => $transaction_return->getToken(),
    ];
  }

  /**
   * Transform Payment transaction state to drupal commerce state.
   *
   * @param string $payment_state
   *   Payment state.
   *
   * @return string|null
   *   Commerce state.
   */
  public function transformTransactionState(string $payment_state) {
    // State is authorized.
    $commerce_sate = NULL;
    if ($payment_state == 'VOIDED') {
      $commerce_sate = 'void';
    }
    if ($payment_state == 'AUTHORIZED') {
      $commerce_sate = 'authorize';
    }
    if ($payment_state == 'COMPLETED') {
      $commerce_sate = 'completed';
    }
    if ($payment_state == 'FULFILL') {
      $commerce_sate = 'completed';
    }

    return $commerce_sate;
  }

  /**
   * Read transaction.
   *
   * @param string $user_id
   *   User id.
   * @param string $secret
   *   Secret.
   * @param string $space_id
   *   Space id.
   * @param string $transaction_id
   *   Transaction id.
   *
   * @return array
   *   Transaction data.
   *
   * @throws \Drupal\commerce_wstack\PaymentSdk\ApiException
   * @throws \Drupal\commerce_wstack\PaymentSdk\Http\ConnectionException
   * @throws \Drupal\commerce_wstack\PaymentSdk\VersioningException
   */
  public function readTransaction(string $user_id, string $secret, string $space_id, string $transaction_id) {
    $client = $this->apiClient($user_id, $secret);
    $transaction = $client->getTransactionService()->read($space_id, $transaction_id);

    return [
      'id' => $transaction_id,
      'authorization_amount' => $transaction->getAuthorizationAmount(),
      'authorization_environment' => $transaction->getAuthorizationEnvironment(),
      'authorized_on' => $transaction->getAuthorizedOn(),
      'completed_amount' => $transaction->getCompletedAmount(),
      'completed_on' => $transaction->getCompletedOn(),
      'currency' => $transaction->getCurrency(),
      'state' => $transaction->getState(),
      'token' => $transaction->getToken(),
    ];
  }

  /**
   * Create refund for transaction.
   */
  public function createRefund($user_id, $secret, $space_id, $amount, $remote_id) {
    // Create refund object.
    $trans = new Transaction();
    $trans->setId($remote_id);

    $refund = new Refund();
    $refund->setAmount($amount);
    $refund->setExternalId(time());
    $refund->setMerchantReference('Drupal Commerce');
    $refund->setTransaction($trans);
    $refund->setType('MERCHANT_INITIATED_ONLINE');

    // Setup API client and call refund.
    $client = $this->apiClient($user_id, $secret);
    return $client->getRefundService()->refund($space_id, $refund);
  }

  /**
   * Create or update Payment method.
   *
   * @param \Drupal\commerce_payment\Entity\PaymentGatewayInterface $payment_gateway
   *   Payment gateway.
   * @param \Drupal\commerce_wstack\PaymentSdk\Model\Token $token
   *   Token.
   * @param \Drupal\user\UserInterface $user
   *   User.
   * @param \Drupal\profile\Entity\ProfileInterface|null $billing_profile
   *   Billing profile.
   *
   * @return \Drupal\commerce_payment\Entity\PaymentMethodInterface|false
   *   Payment method.
   *
   * @throws \Drupal\commerce_wstack\PaymentSdk\ApiException
   * @throws \Drupal\commerce_wstack\PaymentSdk\Http\ConnectionException
   * @throws \Drupal\commerce_wstack\PaymentSdk\VersioningException
   */
  public function createPaymentMethod(PaymentGatewayInterface $payment_gateway, Token $token, $user, $billing_profile = NULL) {
    // Get token version.
    $configuration = $payment_gateway->getPluginConfiguration();
    $client = $this->apiClient($configuration['user_id'], $configuration['secret']);
    try {
      $tokenVersionService = new TokenVersionService($client);
      $version = $tokenVersionService->activeVersion($configuration['space_id'], $token['id']);
    }
    catch (\Exception $e) {
      throw new PaymentGatewayException($e->getMessage());
    }

    if ($version instanceof TokenVersion) {
      if ($version->getState() != 'ACTIVE') {
        throw new PaymentGatewayException('Token version is not active: ' . $token['id']);
      }

      if ($version->getPaymentConnectorConfiguration() == NULL) {
        throw new PaymentGatewayException('Payment connector configuration is not set: ' . $token['id']);
      }

      // Get token information.
      $token_id = $token['id'];
      $card_endingnumber = $version->getName();
      $connector_name = $version->getPaymentConnectorConfiguration()->getName();

      // Get user timezone.
      $owner_timezone = $user->getTimeZone();
      if ($owner_timezone == NULL) {
        $owner_timezone = 'UTC';
      }

      // Default to no expiration (0 means never expires)
      $expires_timestamp = 0;

      // Get expires timestamp.
      if ($version->getExpiresOn() instanceof \DateTime) {
        $expires_timestamp = $version->getExpiresOn()->setTimezone(new \DateTimeZone($owner_timezone));
        $expires_timestamp = $expires_timestamp->getTimestamp();
      }

      // Get payment method storage.
      /** @var \Drupal\commerce_payment\Entity\PaymentMethodInterface $payment_method */
      $payment_method_storage = $this->entityTypeManager->getStorage('commerce_payment_method');

      // Check if payment method already exists.
      $payment_methods = $payment_method_storage->loadByProperties([
        'remote_id' => $token_id,
        'payment_gateway' => $payment_gateway->id(),
      ]);
      if (count($payment_methods) > 0 and current($payment_methods) instanceof PaymentMethod) {
        // Update existing payment method.
        /** @var \Drupal\commerce_payment\Entity\PaymentMethodInterface $payment_method */
        $payment_method = current($payment_methods);
        $payment_method->card_name = $connector_name;
        $payment_method->card_version_id = $version->getId();
        $payment_method->card_number = $card_endingnumber;
        // Only set card expiration if we have a valid expiration timestamp.
        if ($expires_timestamp > 0) {
          $payment_method->card_exp_month = date('m', $expires_timestamp);
          $payment_method->card_exp_year = date('y', $expires_timestamp);
        }
        $payment_method->setRemoteId($token_id);
        $payment_method->setExpiresTime($expires_timestamp);
        $payment_method->save();
      }
      else {
        // Create new payment method.
        $payment_method = $payment_method_storage->createForCustomer(
          'wtype',
          $payment_gateway->id(),
          $user->id(),
          $billing_profile
        );
        $payment_method->card_type = 'wtype';
        $payment_method->card_name = $connector_name;
        $payment_method->card_version_id = $version->getId();
        $payment_method->card_number = $card_endingnumber;
        // Only set card expiration if we have a valid expiration timestamp.
        if ($expires_timestamp > 0) {
          $payment_method->card_exp_month = date('m', $expires_timestamp);
          $payment_method->card_exp_year = date('y', $expires_timestamp);
        }
        $payment_method->setRemoteId($token_id);
        $payment_method->setExpiresTime($expires_timestamp);
        $payment_method->save();
      }

      return $payment_method;
    }

    return FALSE;
  }

  /**
   * Delete Payment method.
   *
   * @param \Drupal\commerce_payment\Entity\PaymentMethodInterface $payment_method
   *   The payment method.
   *
   * @return bool
   *   TRUE if payment method is deleted, FALSE otherwise.
   */
  public function deletePaymentMethod(PaymentMethodInterface $payment_method) {
    // Check if gateway is paymentgateway entity.
    if ($payment_method->payment_gateway->entity instanceof PaymentGateway) {
      $configuration = $payment_method->payment_gateway->entity->getPluginConfiguration();
      $client = $this->apiClient($configuration['user_id'], $configuration['secret']);

      try {
        return $client->getTokenService()->delete($configuration['space_id'], $payment_method->getRemoteId());
      }
      catch (\Exception $e) {
        throw new PaymentGatewayException($e->getMessage());
      }
    }

    return FALSE;
  }

  /**
   * Webhook for Token update.
   *
   * @param \Drupal\commerce_payment\Entity\PaymentGateway $paymentGateway
   *   Payment gateway.
   * @param array $data
   *   Webhook data.
   *
   * @throws \Drupal\commerce_wstack\PaymentSdk\ApiException
   * @throws \Drupal\commerce_wstack\PaymentSdk\Http\ConnectionException
   * @throws \Drupal\commerce_wstack\PaymentSdk\VersioningException
   */
  public function webhookToken(PaymentGateway $paymentGateway, array $data) {
    // Get configuration and create client.
    $configuration = $paymentGateway->getPluginConfiguration();
    $client = $this->apiClient($configuration['user_id'], $configuration['secret']);

    // Get token entity.
    $payment_method = $this->entityTypeManager->getStorage('commerce_payment_method')->loadByProperties(
      [
        'remote_id' => $data['entityId'],
        'payment_gateway.target_id' => $paymentGateway->id(),
      ]);
    if (count($payment_method) > 0 and current($payment_method) instanceof PaymentMethod) {
      // Get token with entity id.
      $payment_method = current($payment_method);

      try {
        $tokenService = new TokenService($client);
        $token = $tokenService->read($configuration['space_id'], $data['entityId']);
        if ($token instanceof Token) {
          // Check token state: DELETE.
          if ($token->getState() == 'DELETED') {
            $payment_method->delete();
          }
          // Check token state: INACTIVE.
          if ($token->getState() == 'INACTIVE') {
            $payment_method->delete();
          }
        }
      }
      catch (\Exception $e) {
        throw new PaymentGatewayException($e->getMessage());
      }
    }
  }

  /**
   * Webhook for Token version update.
   *
   * @param \Drupal\commerce_payment\Entity\PaymentGateway $paymentGateway
   *   Payment gateway.
   * @param array $data
   *   Webhook data.
   *
   * @throws \Drupal\commerce_wstack\PaymentSdk\ApiException
   * @throws \Drupal\commerce_wstack\PaymentSdk\Http\ConnectionException
   * @throws \Drupal\commerce_wstack\PaymentSdk\VersioningException
   */
  public function webhookTokenVersion(PaymentGateway $paymentGateway, array $data) {
    // Get configuration and create client.
    $configuration = $paymentGateway->getPluginConfiguration();
    $client = $this->apiClient($configuration['user_id'], $configuration['secret']);

    // Get token version details.
    try {
      $tokenVersionService = new TokenVersionService($client);
      $version = $tokenVersionService->read($configuration['space_id'], $data['entityId']);
    }
    catch (\Exception $e) {
      throw new PaymentGatewayException($e->getMessage());
    }

    if ($version instanceof TokenVersion) {
      if ($version->getPaymentConnectorConfiguration() == NULL) {
        throw new PaymentGatewayException('Payment connector configuration is not set: ' . $version->getToken()->getId());
      }

      // Get token entity.
      $token_id = $version->getToken()->getId();
      $card_endingnumber = $version->getName();
      $connector_name = $version->getPaymentConnectorConfiguration()->getName();
      $state = $version->getState();

      // Get payment method storage.
      /** @var \Drupal\commerce_payment\Entity\PaymentMethodInterface $payment_method */
      $payment_method_storage = $this->entityTypeManager->getStorage('commerce_payment_method');

      // Check if payment method already exists.
      $payment_methods = $payment_method_storage->loadByProperties([
        'remote_id' => $token_id,
        'payment_gateway' => $paymentGateway->id(),
      ]);
      if (count($payment_methods) > 0 and current($payment_methods) instanceof PaymentMethod) {
        // Update existing payment method.
        /** @var \Drupal\commerce_payment\Entity\PaymentMethodInterface $payment_method */
        $payment_method = current($payment_methods);

        // Load user.
        $user = $payment_method->uid->entity;

        // Get user timezone.
        $owner_timezone = $user->getTimeZone();
        if ($owner_timezone == NULL) {
          $owner_timezone = 'UTC';
        }

        // Get expires timestamp.
        $expires_timestamp = 0;
        if ($version->getExpiresOn() instanceof \DateTime) {
          $expires_timestamp = $version->getExpiresOn()->setTimezone(new \DateTimeZone($owner_timezone));
          $expires_timestamp = $expires_timestamp->getTimestamp();
        }

        // Override expires timestamp.
        if ($state == 'OBSOLETE') {
          $expires_timestamp = time();
        }

        // Update payment method.
        $payment_method->card_name = $connector_name;
        $payment_method->card_version_id = $version->getId();
        $payment_method->card_number = $card_endingnumber;
        // Only set card expiration if we have a valid expiration timestamp.
        if ($expires_timestamp > 0) {
          $payment_method->card_exp_month = date('m', $expires_timestamp);
          $payment_method->card_exp_year = date('y', $expires_timestamp);
        }
        $payment_method->setRemoteId($token_id);
        $payment_method->setExpiresTime($expires_timestamp);
        $payment_method->save();
      }
    }
  }

  /**
   * Webhook for Transaction update.
   *
   * @param \Drupal\commerce_payment\Entity\PaymentGateway $paymentGateway
   *   The payment gateway.
   * @param array $data
   *   The webhook data.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  public function webhookTransaction(PaymentGateway $paymentGateway, array $data) {
    // Get configuration and create client.
    $configuration = $paymentGateway->getPluginConfiguration();

    // Get token entity.
    $payment_storage = $this->entityTypeManager->getStorage('commerce_payment');
    $payments = $payment_storage->loadByProperties(
      [
        'remote_id' => $data['entityId'],
        'payment_gateway' => $paymentGateway->id(),
      ]);
    if (count($payments) > 0 and current($payments) instanceof PaymentInterface) {
      /** @var \Drupal\commerce_payment\Entity\PaymentInterface $payment */
      $payment = current($payments);

      $transaction = $this->readTransaction($configuration['user_id'], $configuration['secret'], $configuration['space_id'], $data['entityId']);
      // Check state of transaction.
      if (isset($transaction['state'])) {
        $this->logger->notice('Webhook transaction state: @state', ['@state' => $transaction['state']]);

        // Transform payment state.
        $state = $this->transformTransactionState($transaction['state']);

        // Update payment.
        if ($state != NULL) {
          // This does not need to update the order, but use the loadForUpdate()
          // method of the order storage, this locking system requires
          // https://www.drupal.org/project/commerce/issues/3043180.
          $order_storage = $this->entityTypeManager->getStorage('commerce_order');
          if (\method_exists($order_storage, 'loadForUpdate')) {
            $order_storage->loadForUpdate($payment->getOrderId());

            // Reload the payment entity.
            $payment_storage->resetCache([$payment->id()]);
            $payment = $payment_storage->load($payment->id());
          }

          // Update payment.
          $payment->state->value = $state;
          $payment->remote_state->value = $state;
          $payment->completed->value = time();
          $payment->save();

          // The order storage does not have a public method to release the
          // lock. The current implementation uss the lock backend with
          // a given lock id.
          if (\method_exists($order_storage, 'loadForUpdate')) {
            $lock_id = 'commerce_order_update:' . $payment->getOrderId();
            \Drupal::lock()->release($lock_id);
          }
        }
        else {
          $this->logger->error('State is null on transaction: @remoteid on payment_gateway: @paymentgateway',
            [
              '@remoteid' => $data['entityId'],
              '@paymentgateway' => $paymentGateway->id(),
            ]);
        }
      }
      else {
        $this->logger->error('No state on transaction: @remoteid on payment_gateway: @paymentgateway',
          [
            '@remoteid' => $data['entityId'],
            '@paymentgateway' => $paymentGateway->id(),
          ]);
      }
    }
    else {
      $this->logger->error('No payment found with remote_id: @remoteid on payment_gateway: @paymentgateway',
        [
          '@remoteid' => $data['entityId'],
          '@paymentgateway' => $paymentGateway->id(),
        ]);
    }
  }

  /**
   * Returns whether a token should be requested.
   *
   * @param array $plugin_configuration
   *   The payment gateway plugin configuration.
   *
   * @return bool
   *   TRUE if a token should be requested.
   */
  protected function shouldRequestToken(array $plugin_configuration) {
    if ($plugin_configuration['payment_reusable'] == 'always') {
      return TRUE;
    }

    if ($plugin_configuration['payment_reusable'] == 'ask' && $this->currentUser->id() != 0) {
      // Check if the current order has the consent flag set.
      $current_route = \Drupal::routeMatch();
      if ($order = $current_route->getParameter('commerce_order')) {
        return (bool) $order->getData('commerce_wstack_save_payment_method');
      }
      return FALSE;
    }

    return FALSE;
  }

  /**
   * Get token version.
   *
   * @param \Drupal\commerce_payment\Entity\PaymentGatewayInterface $paymentGateway
   *   Payment gateway.
   * @param string $token_version_id
   *   Token version ID.
   *
   * @return \Drupal\commerce_wstack\PaymentSdk\Model\TokenVersion
   *   Token version.
   *
   * @throws \Drupal\commerce_wstack\PaymentSdk\ApiException
   * @throws \Drupal\commerce_wstack\PaymentSdk\Http\ConnectionException
   * @throws \Drupal\commerce_wstack\PaymentSdk\VersioningException
   */
  public function getTokenVersion(PaymentGatewayInterface $paymentGateway, $token_version_id) {
    // Get configuration and create client.
    $configuration = $paymentGateway->getPluginConfiguration();
    $client = $this->apiClient($configuration['user_id'], $configuration['secret']);

    try {
      // Get token version details.
      $tokenVersionService = $client->getTokenVersionService();
      return $tokenVersionService->read($configuration['space_id'], $token_version_id);
    }
    catch (\Exception $e) {
      throw new PaymentGatewayException($e->getMessage());
    }
  }

  /**
   * Get active token version.
   *
   * @param \Drupal\commerce_payment\Entity\PaymentGatewayInterface $paymentGateway
   *   Payment gateway.
   * @param string $token_id
   *   Token ID.
   *
   * @return \Drupal\commerce_wstack\PaymentSdk\Model\TokenVersion
   *   Token version.
   *
   * @throws \Drupal\commerce_wstack\PaymentSdk\ApiException
   * @throws \Drupal\commerce_wstack\PaymentSdk\Http\ConnectionException
   * @throws \Drupal\commerce_wstack\PaymentSdk\VersioningException
   */
  public function getActiveTokenVersion(PaymentGatewayInterface $paymentGateway, $token_id) {
    // Get configuration and create client.
    $configuration = $paymentGateway->getPluginConfiguration();
    $client = $this->apiClient($configuration['user_id'], $configuration['secret']);

    try {
      // Get token version details.
      $tokenVersionService = $client->getTokenVersionService();
      return $tokenVersionService->activeVersion($configuration['space_id'], $token_id);
    }
    catch (\Exception $e) {
      throw new PaymentGatewayException($e->getMessage());
    }
  }

  /**
   * Migrate payment methods.
   */
  public function migratePaymentMethods(string $payment_gateway_id) {
    // Load payment gateway.
    $payment_gateway_storage = \Drupal::entityTypeManager()->getStorage('commerce_payment_gateway');
    $payment_gateway = $payment_gateway_storage->load($payment_gateway_id);
    if (!$payment_gateway instanceof PaymentGateway) {
      return FALSE;
    }

    // Load all payment methods.
    $payment_method_storage = \Drupal::entityTypeManager()->getStorage('commerce_payment_method');
    $payment_methods = $payment_method_storage->loadMultiple();
    foreach ($payment_methods as $payment_method) {
      /** @var \Drupal\commerce_payment\Entity\PaymentMethodInterface $payment_method */
      if ($payment_method->type->value == 'credit_card' or $payment_method->type->value == 'paypal') {
        // Get active token version.
        $version = $this->getActiveTokenVersion($payment_gateway, $payment_method->remote_id->value);
        if ($version instanceof TokenVersion) {
          // Get billing profile.
          $billing_profile = $payment_method->getBillingProfile();

          // Get token entity.
          $token_id = $version->getToken()->getId();
          $card_endingnumber = $version->getName();
          $connector_name = $version->getPaymentConnectorConfiguration()->getName();

          // Load user.
          $user = $payment_method->uid->entity;

          // Get user timezone.
          $owner_timezone = $user->getTimeZone();
          if ($owner_timezone == NULL) {
            $owner_timezone = 'UTC';
          }

          // Get expires timestamp.
          $expires_timestamp = 0;
          if ($version->getExpiresOn() instanceof \DateTime) {
            $expires_timestamp = $version->getExpiresOn()->setTimezone(new \DateTimeZone($owner_timezone));
            $expires_timestamp = $expires_timestamp->getTimestamp();
          }

          // Create new payment method.
          $payment_method_wtype = $payment_method_storage->createForCustomer(
            'wtype',
            $payment_gateway->id(),
            $user->id(),
            $billing_profile
          );
          $payment_method_wtype->card_type = 'wtype';
          $payment_method_wtype->card_name = $connector_name;
          $payment_method_wtype->card_version_id = $version->getId();
          $payment_method_wtype->card_number = $card_endingnumber;
          // Only set card expiration if we have a valid expiration timestamp.
          if ($expires_timestamp > 0) {
            $payment_method_wtype->card_exp_month = date('m', $expires_timestamp);
            $payment_method_wtype->card_exp_year = date('y', $expires_timestamp);
          }
          $payment_method_wtype->setRemoteId($token_id);
          $payment_method_wtype->setExpiresTime($expires_timestamp);
          $payment_method_wtype->save();

          $this->logger->notice('Migrated payment method: @card_endingnumber', ['@card_endingnumber' => $card_endingnumber]);

          $payment_method->delete();
        }
      }
    }
  }

}
