<?php

namespace Drupal\commerce_webform_order\Plugin\WebformElement;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Utility\Error;
use Drupal\state_machine\WorkflowManagerInterface;
use Drupal\webform\Plugin\WebformElement\WebformCompositeBase;
use Drupal\webform\Plugin\WebformElement\WebformDisplayOnTrait;
use Drupal\webform\Plugin\WebformElementDisplayOnInterface;
use Drupal\webform\WebformSubmissionInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a form element for synchronizing the order status.
 *
 * @WebformElement(
 *   id = "commerce_webform_order_state",
 *   label = @Translation("Order State"),
 *   description = @Translation("Provides a form element for synchronizing the order state."),
 *   category = @Translation("Commerce"),
 *   multiline = TRUE,
 *   composite = TRUE,
 *   states_wrapper = FALSE,
 *   dependencies = {
 *     "commerce_order",
 *   },
 * )
 */
class OrderState extends WebformCompositeBase implements WebformElementDisplayOnInterface {

  use WebformDisplayOnTrait;

  /**
   * The workflow manager.
   *
   * @var \Drupal\state_machine\WorkflowManagerInterface
   */
  protected WorkflowManagerInterface $workflowManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->workflowManager = $container->get('plugin.manager.workflow');

    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  protected function defineDefaultProperties(): array {
    $properties = [
      'display_on' => WebformElementDisplayOnInterface::DISPLAY_ON_NONE,
      'default_value' => '',
      'workflow' => '',
      'previous' => '',
      'current' => '',
    ] + parent::defineDefaultProperties();

    unset(
      $properties['required'],
      $properties['required_error'],
      $properties['multiple'],
      $properties['multiple__add'],
      $properties['multiple__add_more'],
      $properties['multiple__add_more_button_label'],
      $properties['multiple__add_more_input'],
      $properties['multiple__add_more_input_label'],
      $properties['multiple__add_more_items'],
      $properties['multiple__empty_items'],
      $properties['multiple__header'],
      $properties['multiple__header_label'],
      $properties['multiple__item_label'],
      $properties['multiple__min_items'],
      $properties['multiple__no_items_message'],
      $properties['multiple__operations'],
      $properties['multiple__remove'],
      $properties['multiple__sorting'],
      $properties['format_items'],
      $properties['format_items_html'],
      $properties['format_items_text'],
    );

    return $properties;
  }

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state): array {
    $form = parent::form($form, $form_state);

    $form['order_state'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Order state settings'),
    ];
    $form['order_state']['display_on'] = [
      '#type' => 'select',
      '#title' => $this->t('Display on'),
      '#options' => $this->getDisplayOnOptions(TRUE),
      '#required' => TRUE,
    ];

    $display_on_form_states = [
      'visible' => [
        [':input[name="properties[display_on]"]' => ['value' => WebformElementDisplayOnInterface::DISPLAY_ON_FORM]],
        'or',
        [':input[name="properties[display_on]"]' => ['value' => WebformElementDisplayOnInterface::DISPLAY_ON_BOTH]],
      ],
    ];
    $form['element_description']['#states'] = $display_on_form_states;
    $form['form']['#states'] = $display_on_form_states;
    $form['wrapper_attributes']['#states'] = $display_on_form_states;
    $form['element_attributes']['#states'] = $display_on_form_states;
    $form['label_attributes']['#states'] = $display_on_form_states;

    $display_on_view_states = [
      'visible' => [
        [':input[name="properties[display_on]"]' => ['value' => WebformElementDisplayOnInterface::DISPLAY_ON_VIEW]],
        'or',
        [':input[name="properties[display_on]"]' => ['value' => WebformElementDisplayOnInterface::DISPLAY_ON_BOTH]],
      ],
    ];
    $form['display']['#states'] = $display_on_view_states;

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function postLoad(array &$element, WebformSubmissionInterface $webform_submission): void {
    // Replace the previous state with the current state, so we ensure that
    // the previous state is always the last state when the form is
    // submitted.
    $key = $element['#webform_key'];
    $data = $webform_submission->getData();
    if (!empty($data[$key]) && !empty($data[$key]['current'])) {
      $data[$key]['previous'] = $data[$key]['current'];
      $webform_submission->setData($data);
    }
  }

  /**
   * {@inheritdoc}
   */
  protected function formatComposite($type, array $element, WebformSubmissionInterface $webform_submission, array $options = []): array|string {
    $value = $this->getValue($element, $webform_submission);
    if (empty($value) || empty($value['workflow'])) {
      return '';
    }

    // Display order workflow or state label.
    try {
      $workflow_definition = $this->workflowManager->createInstance($value['workflow']);
      switch ($options['composite_key']) {
        case 'workflow':
          return $workflow_definition->getLabel();

        case 'previous':
        case 'current':
          return $workflow_definition->getState($value[$options['composite_key']])?->getLabel();
      }
    }
    catch (\Exception $exception) {
      $variables = Error::decodeException($exception);
      $this->logger->error('%type: @message in %function (line %line of %file).', $variables);
    }

    return '';
  }

}
