<?php

namespace Drupal\commerce_webform_order\Plugin\WebformElement;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Utility\Error;
use Drupal\webform\Plugin\WebformElement\WebformCompositeBase;
use Drupal\webform\Plugin\WebformElement\WebformDisplayOnTrait;
use Drupal\webform\Plugin\WebformElementDisplayOnInterface;
use Drupal\webform\WebformSubmissionInterface;

/**
 * Provides a form element for synchronizing the payment status.
 *
 * @WebformElement(
 *   id = "commerce_webform_order_payment_status",
 *   label = @Translation("Payment Status"),
 *   description = @Translation("Provides a form element for synchronizing the payment status."),
 *   category = @Translation("Commerce"),
 *   multiline = TRUE,
 *   composite = TRUE,
 *   states_wrapper = FALSE,
 *   dependencies = {
 *     "commerce_order",
 *     "commerce_payment",
 *   },
 * )
 */
class PaymentStatus extends WebformCompositeBase implements WebformElementDisplayOnInterface {

  use WebformDisplayOnTrait;

  /**
   * {@inheritdoc}
   */
  protected function defineDefaultProperties(): array {
    $properties = [
      'display_on' => WebformElementDisplayOnInterface::DISPLAY_ON_NONE,
      'default_value' => '',
      'current' => '',
      'previous' => '',
      'total_paid_number' => '',
      'total_paid_currency_code' => '',
    ] + parent::defineDefaultProperties();

    unset(
      $properties['required'],
      $properties['required_error'],
      $properties['multiple'],
      $properties['multiple__add'],
      $properties['multiple__add_more'],
      $properties['multiple__add_more_button_label'],
      $properties['multiple__add_more_input'],
      $properties['multiple__add_more_input_label'],
      $properties['multiple__add_more_items'],
      $properties['multiple__empty_items'],
      $properties['multiple__header'],
      $properties['multiple__header_label'],
      $properties['multiple__item_label'],
      $properties['multiple__min_items'],
      $properties['multiple__no_items_message'],
      $properties['multiple__operations'],
      $properties['multiple__remove'],
      $properties['multiple__sorting'],
      $properties['format_items'],
      $properties['format_items_html'],
      $properties['format_items_text'],
    );

    return $properties;
  }

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state): array {
    $form = parent::form($form, $form_state);

    $form['payment_status'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Payment status settings'),
    ];
    $form['payment_status']['display_on'] = [
      '#type' => 'select',
      '#title' => $this->t('Display on'),
      '#options' => $this->getDisplayOnOptions(TRUE),
      '#required' => TRUE,
    ];

    $display_on_form_states = [
      'visible' => [
        [':input[name="properties[display_on]"]' => ['value' => WebformElementDisplayOnInterface::DISPLAY_ON_FORM]],
        'or',
        [':input[name="properties[display_on]"]' => ['value' => WebformElementDisplayOnInterface::DISPLAY_ON_BOTH]],
      ],
    ];
    $form['element_description']['#states'] = $display_on_form_states;
    $form['form']['#states'] = $display_on_form_states;
    $form['wrapper_attributes']['#states'] = $display_on_form_states;
    $form['element_attributes']['#states'] = $display_on_form_states;
    $form['label_attributes']['#states'] = $display_on_form_states;

    $display_on_view_states = [
      'visible' => [
        [':input[name="properties[display_on]"]' => ['value' => WebformElementDisplayOnInterface::DISPLAY_ON_VIEW]],
        'or',
        [':input[name="properties[display_on]"]' => ['value' => WebformElementDisplayOnInterface::DISPLAY_ON_BOTH]],
      ],
    ];
    $form['display']['#states'] = $display_on_view_states;

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function postLoad(array &$element, WebformSubmissionInterface $webform_submission): void {
    // Replace the previous state with the current state, so we ensure that
    // the previous state is always the last state when the form is
    // submitted.
    $key = $element['#webform_key'];
    $data = $webform_submission->getData();
    if (!empty($data[$key]) && !empty($data[$key]['current'])) {
      $data[$key]['previous'] = $data[$key]['current'];
      $webform_submission->setData($data);
    }
  }

  /**
   * {@inheritdoc}
   */
  protected function formatComposite($type, array $element, WebformSubmissionInterface $webform_submission, array $options = []): array|string {
    $value = $this->getValue($element, $webform_submission);
    if (empty($value)) {
      return '';
    }

    // Display order workflow or state label.
    try {
      switch ($options['composite_key']) {
        case 'previous':
        case 'current':
          return !empty($value[$options['composite_key']]) ? $this->t('Fully paid') : $this->t('Not fully paid');

        case 'total_paid_number':
        case 'total_paid_currency_code':
          return $value[$options['composite_key']];
      }
    }
    catch (\Exception $exception) {
      $variables = Error::decodeException($exception);
      $this->logger->error('%type: @message in %function (line %line of %file).', $variables);
    }

    return '';
  }

}
