<?php

namespace Drupal\commerce_webform_order;

use Drupal\commerce_order\Entity\OrderInterface;
use Drupal\webform\Entity\WebformSubmission;
use Drupal\webform\WebformSubmissionInterface;

/**
 * Provides helper methods to update the webform element values.
 */
trait WebformSubmissionUpdaterTrait {

  /**
   * Updates the webform submissions with the order state.
   *
   * @param \Drupal\commerce_order\Entity\OrderInterface $order
   *   The order.
   * @param bool $save
   *   Whether to save the webform submission.
   *
   * @throws \Drupal\Core\Entity\EntityStorageException
   */
  protected function updateWebformSubmissionsOrderStateFromOrder(OrderInterface $order, bool $save = TRUE): void {
    $submissions = $this->collectAllWebformSubmissionsFromOrder($order);

    // If there are no webform submissions, we don't need to update anything.
    if (empty($submissions)) {
      return;
    }

    $this->updateWebformSubmissionsOrderState(
      $submissions,
      $order->getState()->getWorkflow()->getId(),
      $order->getState()->getOriginalId(),
      $order->getState()->getId(),
      $save
    );
  }

  /**
   * Updates the webform submissions with the order state.
   *
   * @param \Drupal\webform\WebformSubmissionInterface[] $webform_submissions
   *   The webform submissions.
   * @param string|null $workflow_id
   *   The workflow ID.
   * @param string|null $previous_state
   *   The previous state.
   * @param string|null $current_state
   *   The current state.
   * @param bool $save
   *   Whether to save the webform submission.
   *
   * @throws \Drupal\Core\Entity\EntityStorageException
   */
  protected function updateWebformSubmissionsOrderState(array $webform_submissions, ?string $workflow_id, ?string $previous_state, ?string $current_state, bool $save = TRUE): void {
    foreach ($webform_submissions as $webform_submission) {
      $this->updateWebformSubmissionOrderState($webform_submission, $workflow_id, $previous_state, $current_state, $save);
    }
  }

  /**
   * Updates the webform submission with the order state.
   *
   * @param \Drupal\webform\WebformSubmissionInterface $webform_submission
   *   The webform submission.
   * @param string|null $workflow_id
   *   The workflow ID.
   * @param string|null $previous_state
   *   The previous state.
   * @param string|null $current_state
   *   The current state.
   * @param bool $save
   *   Whether to save the webform submission.
   *
   * @throws \Drupal\Core\Entity\EntityStorageException
   */
  protected function updateWebformSubmissionOrderState(WebformSubmissionInterface $webform_submission, ?string $workflow_id, ?string $previous_state, ?string $current_state, bool $save = TRUE): void {
    $updated = FALSE;

    $elements = $webform_submission->getWebform()->getElementsDecodedAndFlattened();
    foreach ($elements as $element_key => $element) {
      // Search for the order status element(s) and update them.
      if ($element['#type'] === 'commerce_webform_order_state') {
        $updated = TRUE;
        $webform_submission->setElementData($element_key, [
          'workflow' => $workflow_id,
          'previous' => $previous_state,
          'current' => $current_state,
        ]);
      }
    }

    if ($updated && $save) {
      $webform_submission->save();
    }
  }

  /**
   * Updates the webform submissions with the payment status.
   *
   * @param \Drupal\commerce_order\Entity\OrderInterface $order
   *   The order.
   * @param bool $save
   *   Whether to save the webform submission.
   *
   * @throws \Drupal\Core\Entity\EntityStorageException
   */
  protected function updateWebformSubmissionsPaymentStatusFromOrder(OrderInterface $order, bool $save = TRUE): void {
    $submissions = $this->collectAllWebformSubmissionsFromOrder($order);

    // If there are no webform submissions, we don't need to update anything.
    if (empty($submissions)) {
      return;
    }

    $this->updateWebformSubmissionsPaymentStatus(
      $submissions,
      $order->isPaid(),
      $order->getTotalPaid()->getNumber(),
      $order->getTotalPaid()->getCurrencyCode(),
      $save
    );
  }

  /**
   * Updates the webform submissions with the payment status.
   *
   * @param \Drupal\webform\WebformSubmissionInterface[] $webform_submissions
   *   The webform submissions.
   * @param bool $current_state
   *   The current state.
   * @param string|null $total_paid_number
   *   The total paid amount.
   * @param string|null $total_paid_currency_code
   *   The total paid currency code.
   * @param bool $save
   *   Whether to save the webform submission.
   *
   * @throws \Drupal\Core\Entity\EntityStorageException
   */
  protected function updateWebformSubmissionsPaymentStatus(array $webform_submissions, bool $current_state, ?string $total_paid_number, ?string $total_paid_currency_code, bool $save = TRUE): void {
    foreach ($webform_submissions as $webform_submission) {
      $this->updateWebformSubmissionPaymentStatus($webform_submission, $current_state, $total_paid_number, $total_paid_currency_code, $save);
    }
  }

  /**
   * Updates the webform submission with the payment status.
   *
   * @param \Drupal\webform\WebformSubmissionInterface $webform_submission
   *   The webform submission.
   * @param bool $current_state
   *   If the order is fully paid.
   * @param string|null $total_paid_number
   *   The total paid amount.
   * @param string|null $total_paid_currency_code
   *   The total paid currency code.
   * @param bool $save
   *   Whether to save the webform submission.
   *
   * @throws \Drupal\Core\Entity\EntityStorageException
   */
  protected function updateWebformSubmissionPaymentStatus(WebformSubmissionInterface $webform_submission, bool $current_state, ?string $total_paid_number, ?string $total_paid_currency_code, bool $save = TRUE): void {
    $updated = FALSE;

    $elements = $webform_submission->getWebform()->getElementsDecodedAndFlattened();
    foreach ($elements as $element_key => $element) {
      // Search for the order status element(s) and update them.
      if ($element['#type'] === 'commerce_webform_order_payment_status') {
        $updated = TRUE;
        $data = $webform_submission->getElementData($element_key) ?: [];
        $data['current'] = $current_state;
        $data['total_paid_number'] = $total_paid_number;
        $data['total_paid_currency_code'] = $total_paid_currency_code;
        $webform_submission->setElementData($element_key, $data);
      }
    }

    if ($updated && $save) {
      $webform_submission->save();
    }
  }

  /**
   * Collects all webform submissions from the order items.
   *
   * @param \Drupal\commerce_order\Entity\OrderInterface $order
   *   The order.
   *
   * @return \Drupal\webform\WebformSubmissionInterface[]
   *   An array of webform submissions keyed by their IDs.
   */
  protected function collectAllWebformSubmissionsFromOrder(OrderInterface $order): array {
    // Collect all webform submissions from the order items.
    $storage = $this->entityTypeManager->getStorage('webform_submission');
    /** @var \Drupal\webform\WebformSubmissionInterface[] $submissions */
    $submissions = [];
    foreach ($order->getItems() as $item) {
      if ($item->hasField('commerce_webform_order_submission') &&
        !$item->get('commerce_webform_order_submission')->isEmpty()) {

        $submission = $item->get('commerce_webform_order_submission')->entity;
        if ($submission instanceof WebformSubmission) {
          // Ensure any changes to the submission during the curring request
          // are reflected in the returned array.
          $storage->resetCache([$submission->id()]);
          $submissions[$submission->id()] = $storage->load($submission->id());
        }
      }
    }

    return $submissions;
  }

}
