<?php

/**
 * @file
 * Builds placeholder replacement tokens for commerce webform order.
 */

use Drupal\commerce_order\Entity\OrderItemInterface;
use Drupal\Core\Render\BubbleableMetadata;

/**
 * Implements hook_token_info().
 */
function commerce_webform_order_token_info() {
  $types['webform_submission'] = [
    'name' => t('Webform submissions'),
    'description' => t('Tokens related to webform submission.'),
    'needs-data' => 'webform_submission',
  ];
  $webform_submission['commerce_order'] = [
    'name' => t('Commerce Order'),
    'description' => t('The order associated with this submission'),
    'type' => 'commerce_order',
  ];
  $webform_submission['commerce_order_item'] = [
    'name' => t('Commerce Order Item'),
    'description' => t('The order item associated with this submission'),
    'type' => 'commerce_order_item',
  ];

  return [
    'types' => $types,
    'tokens' => [
      'webform_submission' => $webform_submission,
    ],
  ];
}

/**
 * Implements hook_tokens().
 */
function commerce_webform_order_tokens($type, $tokens, array $data, array $options, BubbleableMetadata $bubbleable_metadata) {
  $token_service = \Drupal::token();

  $replacements = [];

  if ($type === 'webform_submission' && !empty($data['webform_submission'])) {
    /** @var \Drupal\commerce_webform_order\OrderItemRepositoryInterface $entity_repository */
    $entity_repository = \Drupal::service('commerce_webform_order.order_item_repository');

    /** @var \Drupal\webform\WebformSubmissionInterface $webform_submission */
    $webform_submission = $data['webform_submission'];

    $commerce_order_item = $entity_repository->getLastByWebformSubmission($webform_submission);
    $commerce_order = $commerce_order_item?->getOrder();

    // If the order item is not an instance of OrderItemInterface, we cannot
    // generate tokens for it, so we return early.
    if (!$commerce_order_item instanceof OrderItemInterface) {
      return $replacements;
    }

    foreach ($tokens as $name => $original) {
      switch ($name) {
        case 'commerce_order_item':
          $replacements[$original] = $commerce_order_item?->label();
          break;

        case 'commerce_order':
          $replacements[$original] = $commerce_order?->label();
          break;
      }
    }

    $entity_types = [
      'commerce_order',
      'commerce_order_item',
    ];
    foreach ($entity_types as $entity_type) {
      if ($$entity_type && ($entity_tokens = $token_service->findWithPrefix($tokens, $entity_type))) {
        $replacements += $token_service->generate($entity_type, $entity_tokens, [$entity_type => $$entity_type], $options, $bubbleable_metadata);
        $bubbleable_metadata->addCacheableDependency($$entity_type);
      }
    }
  }

  return $replacements;
}
