<?php

namespace Drupal\commerce_webform_order\Element;

use Drupal\Component\Utility\Html;
use Drupal\Component\Utility\NestedArray;
use Drupal\Core\Form\FormStateInterface;
use Drupal\commerce_payment\Entity\PaymentMethod as PaymentMethodEntity;
use Drupal\commerce_payment\Entity\PaymentMethodInterface;
use Drupal\commerce_payment\PaymentOption;
use Drupal\commerce_payment\Plugin\Commerce\PaymentGateway\SupportsCreatingPaymentMethodsInterface;
use Drupal\Core\Render\Element;
use Drupal\profile\Entity\ProfileInterface;
use Drupal\webform\Element\WebformCompositeBase;

/**
 * Provides a form element for embedding the payment gateway forms.
 *
 * @FormElement("commerce_webform_order_payment_method")
 */
class PaymentMethod extends WebformCompositeBase {

  /**
   * {@inheritdoc}
   */
  public static function getCompositeElements(array $element): array {
    $elements = [];
    $elements['payment_gateway'] = [
      '#type' => $element['#payment_gateway__type'] ?? 'radios',
      '#title' => t('Payment gateway'),
      '#options' => [],
    ];
    $elements['payment_method'] = [
      '#type' => 'value',
      '#title' => t('Payment method'),
    ];
    $elements['billing_profile'] = [
      '#type' => 'value',
      '#title' => t('Billing profile'),
    ];

    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  public static function processWebformComposite(&$element, FormStateInterface $form_state, &$complete_form): array {
    // Only process composite elements that are visible.
    if (isset($element['#initialize']) || !Element::isVisibleElement($element)) {
      return $element;
    }

    $element = parent::processWebformComposite($element, $form_state, $complete_form);
    $wrapper_id = Html::getUniqueId($element['#id']);
    $supports_stored_payment_methods = empty($element['#disable_stored_payments']);
    $element['#attributes']['id'] = $wrapper_id;
    $element['#attributes']['class'][] = 'commerce-webform-order--payment-method';
    $element['#attributes']['class'][] = 'js-commerce-webform-order--payment-method';

    /** @var \Drupal\commerce_payment\PaymentGatewayStorageInterface $payment_gateway_storage */
    $payment_gateway_storage = \Drupal::entityTypeManager()->getStorage('commerce_payment_gateway');
    $gateway_query = $payment_gateway_storage->getQuery()
      ->condition('status', TRUE);

    if (!empty($element['#allowed_payment_gateways'])) {
      $gateway_query->condition('id', $element['#allowed_payment_gateways'], 'IN');
    }

    $gateway_ids = $gateway_query->execute();

    /** @var \Drupal\commerce_payment\Entity\PaymentGatewayInterface[] $payment_gateways */
    $payment_gateways = $payment_gateway_storage->loadMultiple($gateway_ids);

    // Core bug #1988968 doesn't allow the payment method add form JS to depend
    // on an external library, so the libraries need to be preloaded here.
    foreach ($payment_gateways as $payment_gateway) {
      if (method_exists($payment_gateway, 'getLibraries')) {
        $js_libraries = $payment_gateway->getPlugin()->getLibraries();
      }
      // @todo Remove once Commerce Core <= 2.x support is dropped.
      else {
        $js_libraries = $payment_gateway->getPlugin()->getJsLibrary();
      }

      if (!empty($js_libraries)) {
        $element['#attached']['library'][] = $js_libraries;
      }
    }

    /** @var \Drupal\commerce_payment\PaymentOption[] $options */
    $options = \Drupal::service('commerce_webform_order.options_builder')->buildOptions($payment_gateways, $supports_stored_payment_methods);
    $option_labels = array_map(function (PaymentOption $option) {
      return $option->getLabel();
    }, $options);

    $default_option_id = $element['payment_gateway']['#default_value'] ?? NULL;
    // Use the default option if it is set.
    if ($default_option_id && isset($options[$default_option_id])) {
      $default_option = $options[$default_option_id];
    }
    // If not exists, try to use the first gateway ID coincidence.
    elseif ($default_option_id) {
      foreach ($options as $option) {
        if ($option->getPaymentGatewayId() == $default_option_id) {
          $default_option = $option;
          break;
        }
      }
    }
    // Always use de first value as fallback value.
    if (empty($default_option_id)) {
      $default_option = reset($options);
    }

    $element['#after_build'][] = [get_called_class(), 'clearValues'];
    $element['payment_gateway']['#options'] = $option_labels;
    $element['payment_gateway']['#ajax'] = [
      'callback' => [get_called_class(), 'ajaxRefresh'],
      'wrapper' => $wrapper_id,
    ];

    // Add a class to each individual radio, to help themers.
    if ($element['payment_gateway']['#type'] === 'radios') {
      foreach ($options as $option) {
        $class_name = $option->getPaymentMethodId() ? 'stored' : 'new';
        $element['payment_gateway'][$option->getId()]['#attributes']['class'][] = "payment-method--$class_name";
      }
    }

    if (!empty($default_option)) {
      // Update the default value.
      $element['payment_gateway']['#default_value'] = $default_option->getId();
      $element['payment_method']['#default_value'] = $default_option->getPaymentMethodId();
      if ($default_option->getPaymentMethodId()) {
        try {
          if ($payment_method = PaymentMethodEntity::load($default_option->getPaymentMethodId())) {
            $element['billing_profile']['#default_value'] = $payment_method->getBillingProfile()?->id();
          }
        }
        catch (\Exception $exception) {
          // Just try to load the stored payment method.
        }
      }

      $default_payment_gateway_id = $default_option->getPaymentGatewayId();
      $payment_gateway = $payment_gateways[$default_payment_gateway_id];
      $payment_gateway_plugin = $payment_gateway->getPlugin();

      // If this payment gateway plugin supports creating tokenized payment
      // methods before processing payment, we build the "add-payment-method"
      // plugin form.
      $empty_default_option = empty($default_option->getPaymentMethodId());
      if ($empty_default_option && $payment_gateway_plugin instanceof SupportsCreatingPaymentMethodsInterface) {
        $element = self::buildPaymentMethodForm($element, $form_state, $default_option);
      }
      // Check if the billing profile form should be rendered for the payment
      // gateway to collect billing information.
      elseif (!empty($element['#billing_profile__access']) && $empty_default_option && $payment_gateway_plugin->collectsBillingInformation()) {
        $element = self::buildBillingProfileForm($element, $form_state);
      }
    }

    return $element;
  }

  /**
   * Validates a composite element.
   */
  public static function validateWebformComposite(&$element, FormStateInterface $form_state, &$complete_form): void {
    parent::validateWebformComposite($element, $form_state, $complete_form);

    $value = NestedArray::getValue($form_state->getValues(), $element['#parents']);

    // When the user selects an existing payment method,
    // we must update the values, since the payment gateway selector uses
    // the method key instead of the gateway key.
    // @see \Drupal\commerce_webform_order\PaymentOptionsBuilder::buildOptions()
    if (!empty($value['payment_gateway']) && is_numeric($value['payment_gateway'])) {
      /** @var \Drupal\commerce_payment\PaymentMethodStorageInterface $payment_method_storage */
      $payment_method_storage = \Drupal::entityTypeManager()->getStorage('commerce_payment_method');

      /** @var \Drupal\commerce_payment\Entity\PaymentMethodInterface $payment_method */
      $payment_method = $payment_method_storage->load($value['payment_gateway']);
      $value = [
        'payment_gateway' => $payment_method->getPaymentGatewayId(),
        'payment_method' => $payment_method->id(),
        'billing_profile' => $payment_method?->getBillingProfile(),
      ];
      $element['#value'] = $value;
      $form_state->setValueForElement($element, $value);
    }
  }

  /**
   * Builds the payment method form for the selected payment option.
   *
   * @param array $element
   *   The element.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state of the parent form.
   * @param \Drupal\commerce_payment\PaymentOption $payment_option
   *   The payment option.
   *
   * @return array
   *   The modified form.
   */
  protected static function buildPaymentMethodForm(array $element, FormStateInterface $form_state, PaymentOption $payment_option): array {
    if (!empty($element['#states'])) {
      /** @var \Drupal\webform\WebformSubmissionInterface $webform_submission */
      $webform_submission = $form_state->getFormObject()->getEntity();
      $conditions_validator = \Drupal::service('webform_submission.conditions_validator');
      // Validation and submission should be occurred only when the element is
      // accessible, in multistep forms this element is not accessible in hidden
      // steps.
      if (!$conditions_validator->isElementEnabled($element, $webform_submission) || !$conditions_validator->isElementVisible($element, $webform_submission)) {
        return $element;
      }
    }

    /** @var \Drupal\commerce_payment\PaymentMethodStorageInterface $payment_method_storage */
    $payment_method_storage = \Drupal::entityTypeManager()->getStorage('commerce_payment_method');

    $payment_method = $payment_method_storage->createForCustomer(
      $payment_option->getPaymentMethodTypeId(),
      $payment_option->getPaymentGatewayId(),
      \Drupal::currentUser()->id()
    );

    /** @var \Drupal\commerce\InlineFormManager $inline_form_manager */
    $inline_form_manager = \Drupal::service('plugin.manager.commerce_inline_form');
    $inline_form = $inline_form_manager->createInstance('payment_gateway_form', [
      'operation' => 'add-payment-method',
    ], $payment_method);

    $element['payment_method'] = [
      '#parents' => array_merge($element['#parents'], ['payment_method']),
      '#inline_form' => $inline_form,
    ];
    $element['payment_method'] = $inline_form->buildInlineForm($element['payment_method'], $form_state);

    $element['#element_validate'][] = [get_called_class(), 'validatePaymentMethodForm'];
    unset(
      $element['payment_method']['#element_validate'],
      $element['payment_method']['billing_information']['#element_validate']
    );
    $element['payment_method']['#commerce_element_submit'] = [
      [get_called_class(), 'submitPaymentMethodForm'],
    ];

    return $element;
  }

  /**
   * Runs the inline form validation.
   *
   * @param array $element
   *   The form element.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   */
  public static function validatePaymentMethodForm(array &$element, FormStateInterface $form_state): void {
    /** @var \Drupal\commerce\Plugin\Commerce\InlineForm\InlineFormInterface $plugin */
    $plugin = $element['payment_method']['#inline_form'];
    $plugin->validateInlineForm($element['payment_method'], $form_state);

    if (!empty($element['payment_method']['billing_information'])) {
      /** @var \Drupal\commerce\Plugin\Commerce\InlineForm\InlineFormInterface $plugin */
      $plugin = $element['payment_method']['billing_information']['#inline_form'];
      $plugin->validateInlineForm($element['payment_method']['billing_information'], $form_state);
    }
  }

  /**
   * Runs the inline form submission.
   *
   * @param array $element
   *   The form element.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   */
  public static function submitPaymentMethodForm(array &$element, FormStateInterface $form_state): void {
    /** @var \Drupal\commerce\Plugin\Commerce\InlineForm\InlineFormInterface $plugin */
    $plugin = $element['#inline_form'];
    $plugin->submitInlineForm($element, $form_state);

    /** @var \Drupal\commerce_payment\Entity\PaymentMethodInterface $payment_method */
    $payment_method = $plugin->getEntity();
    $payment_method_id = NULL;
    if ($payment_method instanceof PaymentMethodInterface) {
      $payment_method_id = $payment_method->id();
    }

    $billing_profile_id = NULL;
    if (!empty($element['billing_information'])) {
      /** @var \Drupal\commerce\Plugin\Commerce\InlineForm\InlineFormInterface $plugin */
      $plugin = $element['billing_information']['#inline_form'];
      $plugin->submitInlineForm($element['billing_information'], $form_state);

      /** @var \Drupal\profile\Entity\ProfileInterface $billing_profile */
      $billing_profile = $plugin->getEntity();
      if ($billing_profile instanceof ProfileInterface) {
        $billing_profile_id = $billing_profile->id();

        if ($payment_method instanceof PaymentMethodInterface) {
          $payment_method->setBillingProfile($billing_profile);
        }
      }
    }

    // Ensure payment gateway and method are stored.
    $parents = array_slice($element['#array_parents'], -2, 1);
    $values = $form_state->getValue($parents);
    $values['payment_gateway'] = $payment_method->getPaymentGatewayId();
    $values['payment_method'] = $payment_method_id;
    $values['billing_profile'] = $billing_profile_id;
    $form_state->setValue($parents, $values);
  }

  /**
   * Builds the billing profile form.
   *
   * @param array $element
   *   The element.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state of the parent form.
   *
   * @return array
   *   The modified form.
   */
  protected static function buildBillingProfileForm(array $element, FormStateInterface $form_state): array {
    if (!empty($element['#states'])) {
      /** @var \Drupal\webform\WebformSubmissionInterface $webform_submission */
      $webform_submission = $form_state->getFormObject()->getEntity();
      $conditions_validator = \Drupal::service('webform_submission.conditions_validator');
      // Validation and submission should be occurred only when the element is
      // accessible, in multistep forms this element is not accessible in hidden
      // steps.
      if (!$conditions_validator->isElementEnabled($element, $webform_submission) || !$conditions_validator->isElementVisible($element, $webform_submission)) {
        return $element;
      }
    }

    /** @var \Drupal\profile\ProfileStorageInterface $profile_storage */
    $profile_storage = \Drupal::entityTypeManager()->getStorage('profile');

    try {
      $billing_profile = $element['#value']['billing_profile'] ?
      $profile_storage->load($element['#value']['billing_profile']) :
      NULL;
    }
    catch (\Exception $exception) {
      $billing_profile = NULL;
    }

    if (empty($billing_profile)) {
      $billing_profile = $profile_storage->create([
        'type' => 'customer',
        'uid' => 0,
      ]);
    }

    $store = \Drupal::service('commerce_store.current_store')->getStore();
    /** @var \Drupal\commerce\InlineFormManager $inline_form_manager */
    $inline_form_manager = \Drupal::service('plugin.manager.commerce_inline_form');
    $inline_form = $inline_form_manager->createInstance('customer_profile', [
      'profile_scope' => 'billing',
      'available_countries' => $store->getBillingCountries(),
      'address_book_uid' => \Drupal::currentUser()->id(),
      // Don't copy the profile to address book until the order is placed.
      'copy_on_save' => FALSE,
    ], $billing_profile);

    $element['billing_profile'] = [
      '#parents' => array_merge($element['#parents'], ['billing_profile']),
      '#inline_form' => $inline_form,
    ];
    $element['billing_profile'] = $inline_form->buildInlineForm($element['billing_profile'], $form_state);

    $element['#element_validate'][] = [get_called_class(), 'validateBillingProfileForm'];
    unset($element['billing_profile']['#element_validate']);

    return $element;
  }

  /**
   * Runs the inline form validation.
   *
   * @param array $element
   *   The form element.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   */
  public static function validateBillingProfileForm(array &$element, FormStateInterface $form_state): void {
    /** @var \Drupal\commerce\Plugin\Commerce\InlineForm\InlineFormInterface $plugin */
    $plugin = $element['billing_profile']['#inline_form'];
    $plugin->validateInlineForm($element['billing_profile'], $form_state);
  }

  /**
   * Runs the inline form submission.
   *
   * @param array $element
   *   The form element.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   */
  public static function submitBillingProfileForm(array &$element, FormStateInterface $form_state): void {
    /** @var \Drupal\commerce\Plugin\Commerce\InlineForm\InlineFormInterface $plugin */
    $plugin = $element['#inline_form'];
    $plugin->submitInlineForm($element, $form_state);

    /** @var \Drupal\profile\Entity\ProfileInterface $billing_profile */
    $billing_profile = $plugin->getEntity();
    if ($billing_profile instanceof ProfileInterface) {
      $billing_profile_id = $billing_profile->id();
    }

    // Ensure billing profile stored.
    $parents = array_slice($element['#array_parents'], -2, 1);
    $values = $form_state->getValue($parents);
    $values['billing_profile'] = $billing_profile_id;
    $form_state->setValue($parents, $values);
  }

  /**
   * Ajax callback.
   */
  public static function ajaxRefresh(array $form, FormStateInterface $form_state): array {
    $parents = array_slice($form_state->getTriggeringElement()['#array_parents'], 0, -2);

    return NestedArray::getValue($form, $parents);
  }

  /**
   * Clears dependent form input when the payment_method changes.
   *
   * Without this Drupal considers the rebuilt form to already be submitted,
   * ignoring default values.
   */
  public static function clearValues(array $element, FormStateInterface $form_state): array {
    $triggering_element = $form_state->getTriggeringElement();
    if (!$triggering_element) {
      return $element;
    }

    $parents = array_slice($triggering_element['#array_parents'], 0, -1);
    $triggering_element_name = end($parents);
    if ($triggering_element_name == 'payment_gateway') {
      $user_input = &$form_state->getUserInput();
      $element_input = NestedArray::getValue($user_input, $element['#array_parents']);
      unset($element_input['payment_method']);
      NestedArray::setValue($user_input, $element['#array_parents'], $element_input);
    }

    return $element;
  }

}
