<?php

namespace Drupal\yookassa\Controller;

use Drupal;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Url;
use Drupal\yookassa\Helpers\YooKassaLoggerHelper;
use Drupal\yookassa\Helpers\YooKassaPaymentMethodHelper;
use Exception;
use Drupal\yookassa\Oauth\YooKassaClientFactory;
use Drupal\yookassa\Oauth\YooKassaWebhookSubscriber;
use Drupal\yookassa\Oauth\YooKassaOauth;
use Symfony\Component\HttpFoundation\Request;

class YooKassaOauthController extends ControllerBase
{
    /**
     * Делает запрос к OAuth приложению для получения ссылки на авторизацию
     *
     * @param Request $request
     * @return AjaxResponse
     */
    public function getOauthUrl(Request $request): AjaxResponse
    {
        $machineName = $request->get('name') ?: null;
        $formData = $request->get('form') ?: null;

        $response = new AjaxResponse();
        $is_ajax = $request->isXmlHttpRequest();

        if (!$is_ajax) {
            $response->setData(['status' => 'error', 'error' => 'Unknown', 'code' => 'unknown']);
            return $response;
        }

        try {
            $config = YooKassaPaymentMethodHelper::check($machineName)
                ? Drupal::configFactory()->getEditable(
                    'commerce_payment.commerce_payment_gateway.' . $machineName
                )->getOriginal('configuration')
                : YooKassaPaymentMethodHelper::savePaymentMethod($formData, $machineName)->toArray()['configuration'];
        } catch (Exception $e) {
            $this->errorExit('Error while saving payment method: ' . $e->getMessage(), $e->getMessage(), 500);
        }

        if (!$config) {
            $this->errorExit('Payment method configuration is empty');
        }

        $kassaLogger = $this->getKassaLogger($config['shop_id'] ?? null);
        $kassaLogger->sendHeka(['oauth.process.init']);

        try {
            $oauth = new YooKassaOauth($machineName);
            $result = $oauth->generateOauthUrl();
        } catch (Exception $e) {
            $kassaLogger->sendHeka(['oauth.process.fail']);
            $this->errorExit('Exception: ' . $e->getMessage(), $e->getMessage(), 500);
        }
        $code = $result['code'];

        if ($code != 200) {
            $kassaLogger->sendHeka(['oauth.process.fail']);
            $this->errorExit('Got error while getting OAuth link.');
        }

        $body = json_decode($result['response'], true);

        if (!isset($body['oauth_url'])) {
            $kassaLogger->sendHeka(['oauth.process.fail']);
            $error = empty($body['error']) ? 'OAuth URL not found' : $body['error'];
            $this->errorExit('Got error while getting OAuth link. Response body: ' . json_encode($body), $error);
        }

        $response->setData(['oauth_url' => $body['oauth_url']]);
        $kassaLogger->sendHeka(['oauth.process.success']);
        return $response;
    }

    /**
     * Функция обработки ajax запроса на получение OAuth токена через OAuth-приложение
     *
     * @return void
     * @throws Exception
     */
    public function getOauthToken(Request $request): AjaxResponse
    {
        $machineName = $request->get('name') ?: null;

        $response = new AjaxResponse();
        $is_ajax = $request->isXmlHttpRequest();

        if (!$is_ajax) {
            $response->setData(['status' => 'error', 'error' => 'Unknown', 'code' => 'unknown']);
            return $response;
        }

        $oauth = new YooKassaOauth($machineName);
        $config = $oauth->paymentMethodEditConfig->getOriginal('configuration');
        $kassaLogger = $this->getKassaLogger($config['shop_id'] ?? null);
        $kassaLogger->sendHeka(['oauth.callback.init']);

        try {
            $kassaLogger->sendHeka(['oauth.get-token.init']);
            $result = $oauth->generateOauthToken();
        } catch (Exception $e) {
            $kassaLogger->sendHeka(['oauth.callback.fail', 'oauth.get-token.fail']);
            $this->errorExit('Exception: ' . $e->getMessage(), $e->getMessage(), 500);
        }

        $code = $result['code'];

        if ($code != 200) {
            if ($code == 422) {
                $kassaLogger->sendHeka(['oauth.callback.fail', 'oauth.get-token.fail']);
                $error = empty($body['error']) ? 'Access token not found' : $body['error'];
                $this->errorExit($error, 'Авторизация не пройдена');
            }
            $kassaLogger->sendHeka(['oauth.callback.fail', 'oauth.get-token.fail']);
            $this->errorExit('Got error while getting OAuth token.');
        }

        $body = json_decode($result['response'], true);

        if (!isset($body['access_token'])) {
            $kassaLogger->sendHeka(['oauth.callback.fail', 'oauth.get-token.fail']);
            $error = empty($body['error']) ? 'Access token not found' : $body['error'];
            $this->errorExit('Got error while getting OAuth token. Key access_token not found. Response body: ' . json_encode($body), $error);
        }

        if (!isset($body['expires_in'])) {
            $kassaLogger->sendHeka(['oauth.callback.fail', 'oauth.get-token.fail']);
            $error = empty($body['error']) ? 'Expires_in parameter not found' : $body['error'];
            $this->errorExit('Got error while getting OAuth token. Key expires_in not found. Response body: ' . json_encode($body), $error);
        }

        $token = !empty($config['access_token']) ? $config['access_token'] : null;

        $config['access_token'] = $body['access_token'];
        $config['token_expires_in'] = $body['expires_in'];

        $oauth->saveConfigurationPayment([
            'configuration.access_token' => $body['access_token'],
            'configuration.token_expires_in' => $body['expires_in'],
            'status' => 1
        ]);
        $kassaLogger->sendHeka(['oauth.get-token.success']);

        if ($token) {
            try {
                $kassaLogger->sendHeka(['token.revoke.init']);
                $oauth->revokeOldToken($token);
                $kassaLogger->sendHeka(['token.revoke.success']);
            } catch (Exception $e) {
                $kassaLogger->sendHeka(['oauth.callback.fail', 'token.revoke.fail']);
                $this->log('error', 'Error when revoking old token: ' . $e->getMessage());
            }
        }

        try {
            $client = YooKassaClientFactory::getYooKassaClient($config);
        } catch (Exception $e) {
            $kassaLogger->sendHeka(['oauth.callback.fail']);
            $this->errorExit('Error when creating a client: ' . $e->getMessage(), $e->getMessage(), 500);
        }

        try {
            $kassaLogger->sendHeka(['oauth.get-shop.init']);
            $oauth->saveShopInfoByOauth();
            $kassaLogger->sendHeka(['oauth.get-shop.success']);
        } catch (Exception $e) {
            $kassaLogger->sendHeka(['oauth.callback.fail', 'oauth.get-shop.fail']);
            $this->errorExit('Error to get information about shop: ' . $e->getMessage(), $e->getMessage(), 500);
        }

        try {
            $kassaLogger->sendHeka(['webhooks.subscribe.init']);
            YooKassaWebhookSubscriber::subscribe($client, $config);
            $kassaLogger->sendHeka(['webhooks.subscribe.success']);
        } catch (Exception $e) {
            $kassaLogger->sendHeka(['oauth.callback.fail', 'webhooks.subscribe.fail']);
            $this->errorExit('Error occurred during creating webhooks: ' . $e->getMessage(), $e->getMessage(), 500);
        }

        $url = Url::fromRoute('entity.commerce_payment_gateway.edit_form', ['commerce_payment_gateway' => $machineName])->toString();

        $response->setData(['url' => $url]);
        $kassaLogger->sendHeka(['oauth.callback.success']);
        return $response;
    }

    /**
     * Вывод сообщений об ошибках и остановка скрипта
     *
     * @param string $errorLog
     * @param string|null $errorFront
     * @param int $code
     * @return void
     */
    function errorExit(string $errorLog, string $errorFront = null, int $code = 502)
    {
        $errorFront = $errorFront ?: $errorLog;
        $this->log('Error: ' . $errorLog, 'error');
        echo json_encode(array('error' => $errorFront));
        exit($code);
    }

    /**
     * @param string $message
     * @param string $type
     */
    private function log(string $message, string $type = 'info')
    {
        Drupal::logger('yookassa')->$type($message);
    }

    /**
     * Проверка на существование записи платежного шлюза в БД по машинному имени
     *
     * @param Request $request
     * @return AjaxResponse
     */
    public function checkPaymentMethod(Request $request): AjaxResponse
    {
        $machineName = $request->get('name') ?: null;
        $response = new AjaxResponse();
        if (!$machineName) {
            $response->setData([
                'error' => $this->t('The "Machine Name" field can\'t be empty')
            ]);
            return $response;
        }

        $response->setData([
            'error' => YooKassaPaymentMethodHelper::check($machineName)
                ? $this->t('The "@machineName" payment gateway already exists. Please enter a different name', ['@machineName' => $machineName])
                : null
        ]);

        return $response;
    }

    /**
     * @param null $shopId
     * @return YooKassaLoggerHelper
     */
    public function getKassaLogger($shopId): YooKassaLoggerHelper
    {
        return new YooKassaLoggerHelper($shopId);
    }
}
