<?php

namespace Drupal\yookassa\EventSubscriber;

use Drupal;
use Drupal\commerce_log\LogStorageInterface;
use Drupal\commerce_order\Entity\Order;
use Drupal\commerce_order\Event\OrderEvent;
use Drupal\commerce_order\Event\OrderEvents;
use Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException;
use Drupal\Component\Plugin\Exception\PluginNotFoundException;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\TypedData\Exception\MissingDataException;
use Drupal\yookassa\Helpers\YooKassaLoggerHelper;
use Drupal\yookassa\Helpers\YooKassaLudwigRequireHelper;
use Drupal\yookassa\Oauth\YooKassaClientFactory;
use Exception;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use YooKassa\Client;
use YooKassa\Common\AbstractRequest;
use YooKassa\Common\Exceptions\ApiException;
use YooKassa\Common\Exceptions\BadApiRequestException;
use YooKassa\Common\Exceptions\ExtensionNotFoundException;
use YooKassa\Common\Exceptions\ForbiddenException;
use YooKassa\Common\Exceptions\InternalServerError;
use YooKassa\Common\Exceptions\NotFoundException;
use YooKassa\Common\Exceptions\ResponseProcessingException;
use YooKassa\Common\Exceptions\TooManyRequestsException;
use YooKassa\Common\Exceptions\UnauthorizedException;
use YooKassa\Model\Receipt\PaymentMode;
use YooKassa\Model\ReceiptCustomer;
use YooKassa\Model\ReceiptItem;
use YooKassa\Model\ReceiptType;
use YooKassa\Model\Settlement;
use YooKassa\Request\Receipts\CreatePostReceiptRequest;
use YooKassa\Request\Receipts\PaymentReceiptResponse;
use YooKassa\Request\Receipts\ReceiptResponseItemInterface;

/**
 * Send Second Receipt when the order transitions to Fulfillment.
 */
class YooKassaEventSubscriber implements EventSubscriberInterface
{

    /**
     * The entity type manager.
     *
     * @var EntityTypeManagerInterface
     */
    protected $entityTypeManager;

    /**
     * The module handler.
     *
     * @var ModuleHandlerInterface
     */
    protected $moduleHandler;

    /**
     * @var Client
     */
    public $apiClient;

    /**
     * Конфигурация платежного шлюза
     * @var array
     */
    protected $config;

    /**
     * @var YooKassaLoggerHelper
     */
    private $kassaLogger;

    /**
     * Constructor for YooKassaEventSubscriber.
     * @param EntityTypeManagerInterface $entity_type_manager
     * @param ModuleHandlerInterface $module_handler
     */
    public function __construct(EntityTypeManagerInterface $entity_type_manager, ModuleHandlerInterface $module_handler)
    {
        $this->entityTypeManager = $entity_type_manager;
        $this->moduleHandler = $module_handler;
    }

    /**
     * @return string[]
     */
    public static function getSubscribedEvents(): array
    {
        return [
            OrderEvents::ORDER_PRESAVE => 'onSendSecondReceipt'
        ];
    }

    /**
     * @param $message
     * @param $type
     */
    private function log($message, $type)
    {
        Drupal::logger('yookassa')->$type($message);
    }

    /**
     * Send Second Receipt.
     * @param OrderEvent $event
     * @return void|null
     * @throws ApiException
     * @throws BadApiRequestException
     * @throws Drupal\Core\Entity\EntityStorageException
     * @throws ExtensionNotFoundException
     * @throws ForbiddenException
     * @throws InternalServerError
     * @throws InvalidPluginDefinitionException
     * @throws NotFoundException
     * @throws PluginNotFoundException
     * @throws ResponseProcessingException
     * @throws TooManyRequestsException
     * @throws UnauthorizedException|MissingDataException
     */
    public function onSendSecondReceipt(OrderEvent $event)
    {
        YooKassaLudwigRequireHelper::checkLudwigRequire();
        /** @var Order $order */
        $order = $event->getOrder();
        $isSentSecondReceipt = $order->getData('send_second_receipt') ?? false;
        $state = $order->getState()->getValue();
        $this->config = $this->getPaymentConfig($order);

        if (
            $isSentSecondReceipt
            || empty($this->config['second_receipt_status'])
            || empty($this->config['second_receipt_enabled'])
        ) {
            return;
        }

        if (
            !$this->config['second_receipt_enabled']
            || $state['value'] !== $this->config['second_receipt_status']
        ) {
            return;
        }

        $this->kassaLogger = $this->getKassaLogger($config['shop_id'] ?? null);
        $this->kassaLogger->sendHeka(['second-receipt.webhook.init']);

        try {
            $this->apiClient = YooKassaClientFactory::getYooKassaClient($this->config);
        } catch (Exception $e) {
            $this->kassaLogger->sendAlertLog('SecondReceipt error when creating a client', [
                'exception' => $e,
            ], ['second-receipt.webhook.fail']);
            $this->log('SecondReceipt Error: ' . $e->getMessage(), 'error');
        }

        $orderId = $order->get('order_id')->getString();
        $payments = $this->entityTypeManager->getStorage('commerce_payment')->loadByProperties(['order_id' => $orderId]);
        $paymentId = !empty($payments) ? array_shift($payments)->getRemoteId() : null;
        if (!$paymentId) {
            $this->kassaLogger->sendHeka(['second-receipt.webhook.skip']);
            return;
        }

        $amount = $order->getTotalPrice()->getNumber();
        $customerEmail = $order->get('mail')->getString();

        if (!$lastReceipt = $this->getLastReceipt($paymentId)) {
            $this->kassaLogger->sendHeka(['second-receipt.webhook.fail']);
            $this->log('LastReceipt is empty!', 'error');
            return;
        }

        $receiptRequest = $this->buildSecondReceipt($lastReceipt, $paymentId, $customerEmail);

        if ($receiptRequest === null) {
            $this->kassaLogger->sendHeka(['second-receipt.webhook.fail']);
            $this->log('Build second receipt error', 'error');
            return;
        }

        try {
            $this->kassaLogger->sendHeka(['second-receipt.send.init']);
            $this->apiClient->createReceipt($receiptRequest);
            $order->setData('send_second_receipt', true);
            $this->log('SecondReceipt Send: ' . json_encode($receiptRequest), 'info');

            if ($this->moduleHandler->moduleExists('commerce_log')) {
                /** @var LogStorageInterface $logStorage */
                $logStorage = $this->entityTypeManager->getStorage('commerce_log');
                $logStorage->generate($order, 'order_sent_second_reciept', ['amount' => number_format((float)$amount, 2, '.', '')])->save();
            }
            $this->kassaLogger->sendHeka(['second-receipt.send.success']);
        } catch (ApiException $e) {
            $this->kassaLogger->sendAlertLog('Request second receipt error', [
                'methodid' => 'POST/onSendSecondReceipt',
                'exception' => $e,
            ], ['second-receipt.send.fail', 'second-receipt.webhook.fail']);
            $this->log('SecondReceipt Error: ' . json_encode([
                    'error' => $e->getMessage(),
                    'request' => $receiptRequest->toArray(),
                    'lastReceipt' => $lastReceipt->toArray(),
                    'paymentId' => $paymentId,
                    'customEmail' => $customerEmail
                ]), 'error');
        }
    }

    /**
     * @param $paymentId
     * @return mixed|null
     * @throws ApiException
     * @throws ExtensionNotFoundException
     * @throws BadApiRequestException
     * @throws ForbiddenException
     * @throws InternalServerError
     * @throws NotFoundException
     * @throws ResponseProcessingException
     * @throws TooManyRequestsException
     * @throws UnauthorizedException
     */
    private function getLastReceipt($paymentId)
    {
        $receipts = $this->apiClient->getReceipts(array('payment_id' => $paymentId))->getItems();

        return array_pop($receipts);
    }

    /**
     * @param PaymentReceiptResponse $lastReceipt
     * @param string $paymentId
     * @param string $customerEmail
     * @return AbstractRequest|null
     */
    private function buildSecondReceipt(PaymentReceiptResponse $lastReceipt, string $paymentId, string $customerEmail)
    {
        $this->kassaLogger->sendHeka(['second-receipt.create.init']);

        if ($lastReceipt->getType() === "refund") {
            $this->kassaLogger->sendHeka(['second-receipt.create.skip']);
            $this->log('Last receipt is refund', 'error');
            return null;
        }

        $resendItems = $this->getResendItems($lastReceipt->getItems());

        if (!count($resendItems['items'])) {
            $this->kassaLogger->sendHeka(['second-receipt.create.skip']);
            $this->log('Second receipt is not required', 'error');
            return null;
        }

        try {
            $customer = new ReceiptCustomer();
            $customer->setEmail($customerEmail);

            if (empty($customer)) {
                $this->log('Customer email for second receipt are empty', 'error');
                return null;
            }

            $settlement = new Settlement([
                'type' => 'prepayment',
                'amount' => [
                    'value' => $resendItems['amount'],
                    'currency' => 'RUB',
                ],
            ]);
            $receiptBuilder = CreatePostReceiptRequest::builder();
            $receiptBuilder->setObjectId($paymentId)
                ->setType(ReceiptType::PAYMENT)
                ->setItems($resendItems['items'])
                ->setSettlements([$settlement])
                ->setCustomer($customer)
                ->setSend(true)
                ->setTaxSystemCode($lastReceipt->getTaxSystemCode());
            $this->kassaLogger->sendHeka(['second-receipt.create.success']);
            return $receiptBuilder->build();
        } catch (Exception $e) {
            $this->kassaLogger->sendAlertLog('Build SecondReceipt error', [
                'methodid' => 'POST/buildSecondReceipt',
                'exception' => $e,
            ], ['second-receipt.create.fail']);
            $this->log('Build second receipt error: ' . json_encode([
                    'message' => $e->getMessage()
                ]), 'error');
        }

        return null;
    }

    /**
     * @param ReceiptResponseItemInterface[] $items
     *
     * @return array
     */
    private function getResendItems(array $items): array
    {
        $result = array(
            'items' => array(),
            'amount' => 0,
        );

        foreach ($items as $item) {
            if ($this->isNeedResendItem($item->getPaymentMode())) {
                $item->setPaymentMode(PaymentMode::FULL_PAYMENT);
                $result['items'][] = new ReceiptItem($item->toArray());
                $result['amount'] += $item->getAmount() / 100.0;
            }
        }

        return $result;
    }

    /**
     * @param string $paymentMode
     *
     * @return bool
     */
    private function isNeedResendItem(string $paymentMode): bool
    {
        return in_array($paymentMode, self::getValidPaymentMode());
    }

    /**
     * Список доступных методов оплаты
     *
     * @return array
     */
    private static function getValidPaymentMode(): array
    {
        return array(
            PaymentMode::FULL_PREPAYMENT,
            PaymentMode::PARTIAL_PREPAYMENT,
            PaymentMode::ADVANCE,
            PaymentMode::PARTIAL_PAYMENT,
            PaymentMode::CREDIT,
            PaymentMode::CREDIT_PAYMENT
        );
    }

    /**
     * Получение конфигурации платежного шлюза из данных заказа
     *
     * @param Order $order
     * @return mixed|null
     */
    private function getPaymentConfig(Order $order)
    {
        $machineName = $order->get('payment_gateway')->getString();

        return $machineName ? Drupal::config('commerce_payment.commerce_payment_gateway.' . $machineName)->getOriginal('configuration') : null;
    }

    /**
     * @param null $shopId
     * @return YooKassaLoggerHelper
     */
    private function getKassaLogger($shopId)
    {
        return new YooKassaLoggerHelper($shopId);
    }
}
