<?php

namespace Drupal\yookassa\Helpers;

use Drupal;
use Drupal\commerce_payment\Entity\Payment;
use Drupal\yookassa\Oauth\YooKassaClientFactory;
use Drupal\yookassa\Plugin\Commerce\PaymentGateway\YooKassa;
use Exception;
use YooKassa\Model\Notification\AbstractNotification;
use YooKassa\Model\NotificationEventType;
use YooKassa\Model\PaymentInterface;
use YooKassa\Model\PaymentStatus;
use YooKassa\Request\Payments\Payment\CreateCaptureRequest;

class YooKassaNotificationHelper
{
    /** @var YooKassa */
    private $kassa;

    /** @var YooKassaDatabaseHelper */
    private $databaseHelper;

    /**
     * @param YooKassa $kassa
     */
    public function __construct(YooKassa $kassa)
    {
        $this->kassa = $kassa;
        $this->databaseHelper = new YooKassaDatabaseHelper();
    }

    /**
     * Обработка уведомления от кассы
     *
     * @param AbstractNotification $notificationObj
     * @param PaymentInterface $paymentInfo
     * @param Payment $payment
     * @return bool
     * @throws Exception
     */
    public function processNotification(AbstractNotification $notificationObj, PaymentInterface $paymentInfo, Payment $payment): bool
    {
        $apiClient = YooKassaClientFactory::getYooKassaClient($this->kassa->config);

        if (
            $notificationObj->getEvent() === NotificationEventType::PAYMENT_SUCCEEDED
            && $paymentInfo->getStatus() === PaymentStatus::SUCCEEDED
        ) {
            $this->tryToUpdatePaymentStatus($payment, $paymentInfo->getStatus(), 'completed');
            $stat = $this->databaseHelper->getSuccessPaymentStat();
            if (!empty($stat)) {
                $host = str_replace(array('http://', 'https://', '.', '/', ':'), array('', '', '-', '-', '-'), Drupal::request()->getHost());
                $this->getKassaLogger()->sendHeka([
                    'shop.' . $this->getKassaLogger()->getShopId() . '.payment.succeeded',
                    'shop.' . $this->getKassaLogger()->getShopId() . '.host.' . $host . '.payment-count' => [
                        'metric_type' => "gauges",
                        'metric_count' => $stat['count']
                    ],
                    'shop.' . $this->getKassaLogger()->getShopId() . '.host.' . $host . '.payment-total' => [
                        'metric_type' => "gauges",
                        'metric_count' => $stat['total']
                    ],
                ]);
            }
            return true;
        }

        if (
            $notificationObj->getEvent() === NotificationEventType::PAYMENT_WAITING_FOR_CAPTURE
            && $paymentInfo->getStatus() === PaymentStatus::WAITING_FOR_CAPTURE
        ) {
            $captureRequest  = CreateCaptureRequest::builder()->setAmount($paymentInfo->getAmount())->build();
            $captureResponse = $apiClient->capturePayment($captureRequest, $paymentInfo->getId());
            $this->getLog('Payment info after capture: ' . json_encode($captureResponse));

            if ($captureResponse->status == PaymentStatus::SUCCEEDED) {
                $this->tryToUpdatePaymentStatus($payment, $paymentInfo->getStatus(), 'completed');
            }

            if ($captureResponse->status == PaymentStatus::CANCELED) {
                $this->tryToUpdatePaymentStatus($payment, $paymentInfo->getStatus(), 'canceled');
            }
            $this->getKassaLogger()->sendHeka(['shop.'.$this->getKassaLogger()->getShopId().'.payment.waiting_for_capture']);
            return true;
        }

        if (
            $notificationObj->getEvent() === NotificationEventType::PAYMENT_CANCELED
            && $paymentInfo->getStatus() === PaymentStatus::CANCELED
        ) {
            $this->tryToUpdatePaymentStatus($payment, $paymentInfo->getStatus(), 'canceled');
            $this->getKassaLogger()->sendHeka(['shop.'.$this->getKassaLogger()->getShopId().'.payment.canceled']);
        }

        if ($paymentInfo->getStatus() === PaymentStatus::PENDING) {
            $this->tryToUpdatePaymentStatus($payment, $paymentInfo->getStatus(), 'pending');
            return true;
        }

        return false;
    }

    /**
     * Обработка уведомления возвращение на страницу с заказом после оплаты
     *
     * @param PaymentInterface $paymentInfoResponse
     * @param Payment $payment
     * @return bool
     */
    public function processReturn(PaymentInterface $paymentInfoResponse, Payment $payment): bool
    {
        if ($paymentInfoResponse->getStatus() === PaymentStatus::SUCCEEDED) {
            $this->tryToUpdatePaymentStatus($payment, $paymentInfoResponse->getStatus(), 'completed');
            return true;
        }

        if ($paymentInfoResponse->getStatus() === PaymentStatus::PENDING && $paymentInfoResponse->getPaid()) {
            $this->tryToUpdatePaymentStatus($payment, $paymentInfoResponse->getStatus(), 'pending');
            return true;
        }

        if ($paymentInfoResponse->getStatus() === PaymentStatus::CANCELED) {
            $this->tryToUpdatePaymentStatus($payment, $paymentInfoResponse->getStatus(), 'canceled');
            return false;
        }

        $this->getLog('Wrong payment status: ' . $paymentInfoResponse->getStatus(), 'error');
        return false;
    }

    /**
     * @param Payment $payment
     * @param string $remoteState
     * @param string $state
     * @return void
     */
    private function tryToUpdatePaymentStatus(Payment $payment, string $remoteState, string $state)
    {
        try {
            $this->getKassaLogger()->sendHeka(['order-status.change.init']);
            $payment->setRemoteState($remoteState);
            $payment->setState($state);
            $payment->save();
            $this->getLog('Payment ' . $state);
            $this->getKassaLogger()->sendHeka(['order-status.change.success']);
        } catch (Exception $e) {
            $this->getKassaLogger()->sendHeka(['order-status.change.fail']);
        }
    }

    /**
     * @return YooKassaLoggerHelper|null
     */
    private function getKassaLogger()
    {
        return $this->kassa->kassaLogger;
    }

    /**
     * @param string $message
     * @param string $type
     * @return void
     */
    private function getLog(string $message, string $type = 'info')
    {
        $this->kassa->log($message, $type);
    }
}