<?php

declare(strict_types=1);

namespace Drupal\commerce_yotpo\EventSubscriber;

use Drupal\commerce_order\Entity\OrderInterface;
use Drupal\commerce_yotpo\Service\OrderExporter;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\state_machine\Event\WorkflowTransitionEvent;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Sends order data to Yotpo when an order is placed.
 */
class OrderPlacedSubscriber implements EventSubscriberInterface {

  /**
   * The config factory.
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * The order exporter service.
   */
  protected OrderExporter $exporter;

  /**
   * Logger channel.
   */
  protected LoggerChannelInterface $logger;

  /**
   * Constructs the subscriber.
   */
  public function __construct(ConfigFactoryInterface $configFactory, OrderExporter $exporter, LoggerChannelInterface $logger) {
    $this->configFactory = $configFactory;
    $this->exporter = $exporter;
    $this->logger = $logger;
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      'commerce_order.place.post_transition' => 'onOrderPlaced',
    ];
  }

  /**
   * Reacts to placed orders.
   */
  public function onOrderPlaced(WorkflowTransitionEvent $event): void {
    $order = $event->getEntity();
    if (!$order instanceof OrderInterface) {
      return;
    }

    $config = $this->configFactory->get('commerce_yotpo.settings');
    if (!$config->get('enable_checkout_tracking')) {
      return;
    }

    $this->logger->debug('Sending placed order @order to Yotpo.', ['@order' => $order->id()]);
    $this->exporter->sendOrder($order);
  }

}
