<?php

declare(strict_types=1);

namespace Drupal\commerce_yotpo\Service;

use Drupal\commerce_order\Entity\OrderInterface;
use Drupal\commerce_order\Entity\OrderItemInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;

/**
 * Prepares and sends order data to Yotpo.
 */
class OrderExporter {

  use StringTranslationTrait;

  /**
   * Yotpo client.
   */
  protected YotpoClient $client;

  /**
   * Product data builder.
   */
  protected ProductDataBuilder $productDataBuilder;

  /**
   * Logger channel.
   */
  protected LoggerChannelInterface $logger;

  /**
   * Constructs the exporter.
   */
  public function __construct(YotpoClient $client, ProductDataBuilder $productDataBuilder, LoggerChannelInterface $logger) {
    $this->client = $client;
    $this->productDataBuilder = $productDataBuilder;
    $this->logger = $logger;
  }

  /**
   * Sends an order to Yotpo.
   */
  public function sendOrder(OrderInterface $order): void {
    if (!$this->client->isConfigured()) {
      $this->logger->debug('Skipped sending order @order to Yotpo: missing credentials.', ['@order' => $order->id()]);
      return;
    }

    $token = $this->client->getAccessToken();
    if (!$token) {
      $this->logger->warning('Unable to retrieve a Yotpo access token when exporting order @order.', ['@order' => $order->id()]);
      return;
    }

    $email = $order->getEmail();
    if (!$email) {
      $this->logger->debug('Order @order has no customer email; skipping Yotpo export.', ['@order' => $order->id()]);
      return;
    }

    $products = $this->buildProducts($order);
    if (!$products) {
      $this->logger->debug('Order @order has no shippable products; skipping Yotpo export.', ['@order' => $order->id()]);
      return;
    }

    $total_price = $order->getTotalPrice();

    $payload = [
      'utoken' => $token,
      'email' => $email,
      'customer_name' => $this->resolveCustomerName($order),
      'order_id' => $order->getOrderNumber() ?: $order->id(),
      'order_date' => date('Y-m-d', (int) $order->getCreatedTime()),
      'currency_iso' => $total_price ? $total_price->getCurrencyCode() : '',
      'platform' => 'general',
      'products' => $products,
    ];

    $this->client->createPurchase($payload);
  }

  /**
   * Builds the product details for each order item.
   *
   * @return array<string, array<string, string>>
   *   A keyed array keyed by product id.
   */
  protected function buildProducts(OrderInterface $order): array {
    $products = [];
    /** @var \Drupal\commerce_order\Entity\OrderItemInterface $order_item */
    foreach ($order->getItems() as $order_item) {
      if (!$order_item instanceof OrderItemInterface) {
        continue;
      }
      $purchased_entity = $order_item->getPurchasedEntity();
      if (!$purchased_entity) {
        continue;
      }

      $product_data = $this->productDataBuilder->buildFromPurchasable($purchased_entity);
      $product_id = $product_data['product_id'] ?? NULL;
      if (!$product_id) {
        continue;
      }
      unset($product_data['product_id']);

      $price = $order_item->getUnitPrice();
      $product_data['price'] = $price ? $price->getNumber() : '0';

      $products[$product_id] = $product_data;
    }

    return $products;
  }

  /**
   * Resolves the customer's display name.
   */
  protected function resolveCustomerName(OrderInterface $order): string {
    $profile = $order->getBillingProfile();
    if ($profile && $profile->hasField('address') && !$profile->get('address')->isEmpty()) {
      $address = $profile->get('address')->first();
      $parts = array_filter([
        $address->given_name ?? NULL,
        $address->family_name ?? NULL,
      ]);
      if ($parts) {
        return implode(' ', $parts);
      }
    }

    $customer = $order->getCustomer();
    if ($customer && $customer->id()) {
      return $customer->getDisplayName();
    }

    return (string) $this->t('Customer');
  }

}
