<?php

declare(strict_types=1);

namespace Drupal\commerce_yotpo\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\State\StateInterface;
use Drupal\Component\Datetime\TimeInterface;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\GuzzleException;

/**
 * Lightweight client for the Yotpo REST API.
 */
class YotpoClient {

  private const TOKEN_STATE_KEY = 'commerce_yotpo.access_token';
  private const TOKEN_ENDPOINT = 'https://api.yotpo.com/oauth/token';
  private const PURCHASE_ENDPOINT_TEMPLATE = 'https://api.yotpo.com/apps/%s/purchases';

  /**
   * The HTTP client.
   */
  protected ClientInterface $httpClient;

  /**
   * The config factory.
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * The state storage.
   */
  protected StateInterface $state;

  /**
   * The request time service.
   */
  protected TimeInterface $time;

  /**
   * Logger channel.
   */
  protected LoggerChannelInterface $logger;

  /**
   * Constructs the client.
   */
  public function __construct(ClientInterface $httpClient, ConfigFactoryInterface $configFactory, StateInterface $state, TimeInterface $time, LoggerChannelInterface $logger) {
    $this->httpClient = $httpClient;
    $this->configFactory = $configFactory;
    $this->state = $state;
    $this->time = $time;
    $this->logger = $logger;
  }

  /**
   * Checks whether the client has the required configuration.
   */
  public function isConfigured(): bool {
    $config = $this->configFactory->get('commerce_yotpo.settings');
    return !empty($config->get('app_key')) && !empty($config->get('app_secret'));
  }

  /**
   * Creates or reuses an access token.
   */
  public function getAccessToken(): ?string {
    if (!$this->isConfigured()) {
      return NULL;
    }

    $cached = $this->state->get(self::TOKEN_STATE_KEY);
    if (is_array($cached) && !empty($cached['token']) && !empty($cached['expires']) && $cached['expires'] > $this->time->getRequestTime()) {
      return (string) $cached['token'];
    }

    return $this->requestAccessToken();
  }

  /**
   * Pushes an order payload to Yotpo.
   *
   * @param array<string, mixed> $payload
   *   The payload to send.
   *
   * @return bool
   *   TRUE on success.
   */
  public function createPurchase(array $payload): bool {
    $config = $this->configFactory->get('commerce_yotpo.settings');
    $app_key = $config->get('app_key');
    if (!$app_key) {
      return FALSE;
    }

    $endpoint = sprintf(self::PURCHASE_ENDPOINT_TEMPLATE, $app_key);

    try {
      $response = $this->httpClient->request('POST', $endpoint, [
        'json' => $payload,
        'http_errors' => FALSE,
        'headers' => [
          'Accept' => 'application/json',
        ],
      ]);
    }
    catch (GuzzleException $exception) {
      $this->logger->error('Failed to send purchase to Yotpo: @message', ['@message' => $exception->getMessage()]);
      return FALSE;
    }

    $success = $response->getStatusCode() >= 200 && $response->getStatusCode() < 300;
    if (!$success) {
      $body = (string) $response->getBody();
      $this->logger->warning('Yotpo responded with @code when posting a purchase. Response: @body', [
        '@code' => $response->getStatusCode(),
        '@body' => $body,
      ]);
    }

    return $success;
  }

  /**
   * Requests a fresh access token from Yotpo.
   */
  protected function requestAccessToken(): ?string {
    $config = $this->configFactory->get('commerce_yotpo.settings');
    $app_key = $config->get('app_key');
    $app_secret = $config->get('app_secret');

    try {
      $response = $this->httpClient->request('POST', self::TOKEN_ENDPOINT, [
        'form_params' => [
          'grant_type' => 'client_credentials',
          'client_id' => $app_key,
          'client_secret' => $app_secret,
        ],
        'http_errors' => FALSE,
        'headers' => [
          'Accept' => 'application/json',
        ],
      ]);
    }
    catch (GuzzleException $exception) {
      $this->logger->error('Unable to request a Yotpo access token: @message', ['@message' => $exception->getMessage()]);
      return NULL;
    }

    $data = json_decode((string) $response->getBody(), TRUE) ?: [];
    if (empty($data['access_token'])) {
      $this->logger->warning('Unexpected response when requesting a Yotpo access token. Response: @response', [
        '@response' => json_encode($data),
      ]);
      return NULL;
    }

    $expires_in = isset($data['expires_in']) ? (int) $data['expires_in'] : 3600;
    $expires_at = $this->time->getRequestTime() + max(60, $expires_in - 60);
    $this->state->set(self::TOKEN_STATE_KEY, [
      'token' => $data['access_token'],
      'expires' => $expires_at,
    ]);

    return (string) $data['access_token'];
  }

}
