<?php

declare(strict_types=1);

namespace Drupal\commerce_yotpo\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configure Yotpo integration settings.
 */
class YotpoSettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['commerce_yotpo.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'commerce_yotpo_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('commerce_yotpo.settings');

    $form['credentials'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('API credentials'),
      '#tree' => FALSE,
    ];

    $form['credentials']['app_key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('App key'),
      '#default_value' => $config->get('app_key'),
      '#description' => $this->t('Found in your Yotpo account under Settings > General Settings > API Credentials.'),
      '#required' => TRUE,
    ];

    $form['credentials']['app_secret'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Secret key'),
      '#default_value' => $config->get('app_secret'),
      '#description' => $this->t('Also known as the API secret.'),
      '#required' => TRUE,
    ];

    $form['product_mapping'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Product mapping'),
    ];

    $form['product_mapping']['product_image_field'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Product image field'),
      '#default_value' => $config->get('product_image_field') ?? 'field_images',
      '#description' => $this->t('Machine name of the image field on product variations used for the product thumbnail.'),
    ];

    $form['product_mapping']['product_description_field'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Product description field'),
      '#default_value' => $config->get('product_description_field') ?? '',
      '#description' => $this->t('Optional machine name of a text field used for the product description sent to Yotpo. Leave empty to fall back to the variation title.'),
    ];

    $form['advanced'] = [
      '#type' => 'details',
      '#title' => $this->t('Advanced settings'),
      '#open' => FALSE,
    ];

    $form['advanced']['enable_checkout_tracking'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Send orders to Yotpo when they are placed'),
      '#default_value' => $config->get('enable_checkout_tracking') ?? TRUE,
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    parent::validateForm($form, $form_state);

    $app_key = $form_state->getValue('app_key');
    $secret = $form_state->getValue('app_secret');

    if (strlen($app_key) < 10 || strlen($secret) < 10) {
      $form_state->setErrorByName('app_key', $this->t('The provided credentials look incomplete. Please double-check the app key and secret.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    parent::submitForm($form, $form_state);

    $this->configFactory()->getEditable('commerce_yotpo.settings')
      ->set('app_key', $form_state->getValue('app_key'))
      ->set('app_secret', $form_state->getValue('app_secret'))
      ->set('product_image_field', $form_state->getValue('product_image_field'))
      ->set('product_description_field', $form_state->getValue('product_description_field'))
      ->set('enable_checkout_tracking', (bool) $form_state->getValue('enable_checkout_tracking'))
      ->save();
  }

}
