<?php

declare(strict_types=1);

namespace Drupal\commerce_yotpo\Plugin\Block;

use Drupal\commerce_yotpo\Service\ProductDataBuilder;
use Drupal\Core\Block\BlockBase;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Base class for Yotpo blocks.
 */
abstract class YotpoBlockBase extends BlockBase implements ContainerFactoryPluginInterface {

  /**
   * The config factory.
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * Product data builder.
   */
  protected ProductDataBuilder $productDataBuilder;

  /**
   * Request stack.
   */
  protected RequestStack $requestStack;

  /**
   * Current route match.
   */
  protected RouteMatchInterface $routeMatch;

  /**
   * Constructs the block plugin.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, ConfigFactoryInterface $configFactory, ProductDataBuilder $productDataBuilder, RequestStack $requestStack, RouteMatchInterface $routeMatch) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->configFactory = $configFactory;
    $this->productDataBuilder = $productDataBuilder;
    $this->requestStack = $requestStack;
    $this->routeMatch = $routeMatch;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('config.factory'),
      $container->get('commerce_yotpo.product_data_builder'),
      $container->get('request_stack'),
      $container->get('current_route_match')
    );
  }

  /**
   * Loads the configured Yotpo app key.
   */
  protected function getAppKey(): ?string {
    $value = $this->configFactory->get('commerce_yotpo.settings')->get('app_key');
    return $value ? (string) $value : NULL;
  }

  /**
   * Provides shared attachments for Yotpo blocks.
   */
  protected function buildBaseRenderArray(): array {
    $app_key = $this->getAppKey();
    if (!$app_key) {
      return [];
    }

    return [
      '#attached' => [
        'library' => ['commerce_yotpo/widget_loader'],
        'drupalSettings' => [
          'commerceYotpo' => [
            'appKey' => $app_key,
          ],
        ],
      ],
    ];
  }

  /**
   * Gets the current request domain.
   */
  protected function getDomain(): string {
    $request = $this->requestStack->getCurrentRequest();
    return $request ? $request->getHost() : '';
  }

}
