<?php

declare(strict_types=1);

namespace Drupal\commerce_yotpo\Service;

use Drupal\commerce\PurchasableEntityInterface;
use Drupal\commerce_product\Entity\ProductInterface;
use Drupal\commerce_product\Entity\ProductVariationInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\File\FileUrlGeneratorInterface;
use Drupal\Core\Routing\RouteMatchInterface;

/**
 * Prepares product information for Yotpo payloads.
 */
class ProductDataBuilder {

  /**
   * The configuration factory.
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * Generates absolute file URLs.
   */
  protected FileUrlGeneratorInterface $fileUrlGenerator;

  /**
   * Constructs the builder.
   */
  public function __construct(ConfigFactoryInterface $configFactory, FileUrlGeneratorInterface $fileUrlGenerator) {
    $this->configFactory = $configFactory;
    $this->fileUrlGenerator = $fileUrlGenerator;
  }

  /**
   * Builds product payload data from a purchasable entity.
   *
   * @param \Drupal\commerce\PurchasableEntityInterface $entity
   *   The purchased entity.
   *
   * @return array<string, string>
   *   The Yotpo-compatible product data.
   */
  public function buildFromPurchasable(PurchasableEntityInterface $entity): array {
    $config = $this->configFactory->get('commerce_yotpo.settings');
    $image_field = $config->get('product_image_field') ?? 'field_images';
    $description_field = $config->get('product_description_field') ?? '';

    $product_id = $this->buildProductId($entity);
    $name = $entity->label();
    $url = $this->buildProductUrl($entity);
    $description = $this->buildDescription($entity, $description_field) ?: $name;
    $image = $this->buildImageUrl($entity, $image_field);

    return [
      'product_id' => $product_id,
      'name' => $name,
      'url' => $url,
      'image' => $image,
      'description' => $description,
    ];
  }

  /**
   * Builds product information from the current route.
   *
   * @param \Drupal\Core\Routing\RouteMatchInterface $route_match
   *   The route match.
   *
   * @return array<string, string>|null
   *   The product data, or NULL if it cannot be determined.
   */
  public function buildFromRoute(RouteMatchInterface $route_match): ?array {
    $entity = $route_match->getParameter('commerce_product_variation');
    if ($entity instanceof PurchasableEntityInterface) {
      return $this->buildFromPurchasable($entity);
    }

    $product = $route_match->getParameter('commerce_product');
    if ($product instanceof ProductInterface) {
      $variation = $product->getDefaultVariation();
      if (!$variation && $product->hasVariations()) {
        $variations = $product->getVariations();
        $variation = reset($variations) ?: NULL;
      }
      if ($variation instanceof PurchasableEntityInterface) {
        return $this->buildFromPurchasable($variation);
      }
    }

    return NULL;
  }

  /**
   * Generates a product identifier suitable for Yotpo.
   */
  protected function buildProductId(PurchasableEntityInterface $entity): string {
    if (method_exists($entity, 'getSku') && $entity->getSku()) {
      return (string) $entity->getSku();
    }
    return (string) $entity->id();
  }

  /**
   * Builds an absolute product URL.
   */
  protected function buildProductUrl(PurchasableEntityInterface $entity): string {
    $product = $this->extractProduct($entity);
    $link = $product ? $product->toUrl('canonical', ['absolute' => TRUE]) : $entity->toUrl('canonical', ['absolute' => TRUE]);
    return $link->toString(TRUE)->getGeneratedUrl();
  }

  /**
   * Builds a human-friendly description string.
   */
  protected function buildDescription(PurchasableEntityInterface $entity, string $field_name): string {
    $description = '';

    if ($field_name && $entity->hasField($field_name) && !$entity->get($field_name)->isEmpty()) {
      $description = $this->extractText($entity->get($field_name)->first()?->getValue());
    }

    if ($description) {
      return $description;
    }

    $product = $this->extractProduct($entity);
    if ($product && $field_name && $product->hasField($field_name) && !$product->get($field_name)->isEmpty()) {
      $description = $this->extractText($product->get($field_name)->first()?->getValue());
    }

    return $description;
  }

  /**
   * Builds an absolute image URL.
   */
  protected function buildImageUrl(PurchasableEntityInterface $entity, string $field_name): string {
    $file_uri = '';
    if ($entity->hasField($field_name) && !$entity->get($field_name)->isEmpty()) {
      $file = $entity->get($field_name)->entity;
      if ($file) {
        $file_uri = $file->getFileUri();
      }
    }

    if (!$file_uri) {
      $product = $this->extractProduct($entity);
      if ($product && $product->hasField($field_name) && !$product->get($field_name)->isEmpty()) {
        $file = $product->get($field_name)->entity;
        if ($file) {
          $file_uri = $file->getFileUri();
        }
      }
    }

    return $file_uri ? $this->fileUrlGenerator->generateAbsoluteString($file_uri) : '';
  }

  /**
   * Extracts the parent product when possible.
   */
  protected function extractProduct(PurchasableEntityInterface $entity): ?ProductInterface {
    if ($entity instanceof ProductVariationInterface) {
      return $entity->getProduct();
    }
    return NULL;
  }

  /**
   * Safely extracts a plain text string from a typed data value.
   *
   * @param array<string, mixed>|null $value
   *   The raw field value.
   */
  protected function extractText(?array $value): string {
    if (!$value) {
      return '';
    }

    if (isset($value['value'])) {
      $text = $value['value'];
    }
    elseif (isset($value['processed'])) {
      $text = $value['processed'];
    }
    else {
      $text = reset($value) ?: '';
    }

    return trim(strip_tags((string) $text));
  }

}
